#!/bin/bash

getConfig() {
  __result=$1
  __key=$2
  value=$(grep "^${__key}" ../etc/config.properties | sed "s/^${__key}=//")
  eval $__result="'$value'"
}

setConfig() {
  __file=$1
  __key=$2
  __value=$3

  if [[ "${OSTYPE//[0-9.]/}" == "darwin" ]]; then
    $(sed -i '' "s|^${__key}.*|${__key}=${__value}|" "$__file")
  else
    $(sed -i "s|^${__key}.*|${__key}=${__value}|" "$__file")
  fi
}

## Is docker installed?
version=$(docker --version)
if [ ! $? -eq 0 ]; then
  echo "Please install Docker before proceeding"
  exit
fi

echo "Found $version"

## Requirements
echo "*************************************************************************************"
echo "Welcome to the P4 Render installation script"
echo "*************************************************************************************"
echo "  Before you begin, make sure you have the following ready:"
echo "    1. A working P4 Search URL"
echo "    2. A P4 Search X-Auth-Token"
echo "    3. A Maya installer for Linux"
echo "  This script will build and start p4render and p4render-maya containers "
echo "  to generate previews for your 3D assets."
echo "*************************************************************************************"
echo

read -p "  Do you want to proceed? (y/n): " yn
yn=${yn:-"y"}
if ! [[ $yn =~ ^(Yes|yes|Y|y| ) ]]; then
  echo "  Exiting..."
  exit
fi

echo

## Look for Maya binary
files_dir="$(pwd)/p4render-maya/files/maya"

if ! [ -f "$files_dir/Autodesk_Maya_2026_2_Update_Linux_64bit.tgz" ];
then
  echo "Maya installer not found."
  echo
  echo "If you already have Maya 2026.2, copy it to $files_dir."
  echo "If you don't have Maya 2026.2, you can download it from "
  echo "https://manage.autodesk.com/products/all or we can download it for you."
  echo

  read -p "  Would you like us to download Maya 2026.2 for you? (y/n): " yn
  if ! [[ $yn =~ ^(Yes|yes|Y|y| ) ]]; then
    echo "  Exiting..."
    exit
  fi
  download_link="https://efulfillment.autodesk.com/NetSWDLD/prd/2026/MAYA/901073B8-F0A1-3952-A459-9CA36A875C41/Autodesk_Maya_2026_2_Update_Linux_64bit.tgz"
  echo "  Downloading..."
  mkdir -p $files_dir; cd $files_dir && { curl -O $download_link ; cd -; }
  echo
else
  echo "  Found Maya installer."
  echo
fi

## Read P4Search url and xAuthToken
getConfig p4searchUrlDefault "com.perforce.p4search.service.external-url"
read -p "  Enter your P4 Search URL with the port [$p4searchUrlDefault]: " p4searchUrl
p4searchUrl=${p4searchUrl:-$p4searchUrlDefault}

getConfig xAuthTokenDefault "com.perforce.p4search.service.token"
read -p "  Enter your X-Auth-Token [$xAuthTokenDefault]: " xAuthToken
xAuthToken=${xAuthToken:-$xAuthTokenDefault}

setConfig ".env" "P4SEARCH_URL" $p4searchUrl
setConfig ".env" "X_AUTH_TOKEN" $xAuthToken
setConfig ".env" "UID" $(id -u)

### Build the containers
echo "Creating shared volume p4render-shared..."
mkdir -p p4render-shared
echo "  Building containers... This may take several minutes."
docker compose --file docker-compose.yml build
#
### Run containers
echo "  Starting containers..."
docker compose up
echo "  Done."