import logging
from pathlib import Path
import os
import sys, shutil
from time import sleep
from maya import standalone, cmds
from maya import mel

# ----------------------------------------------------------------------------------------------------------------------

class Main:
    def __init__(self, home_dir):
        logging.info("Initializing...")

        standalone.initialize("Python")
        cmds.loadPlugin("fbxmaya")
        standalone.initialize()

        self.home_dir = home_dir

    # ----------------------------------------------------------------------------------------------------------------------

    def process_files(self):
        # Recursively list files and directories
        cur_dir = self.home_dir
        logging.info(f"Looking for Maya files in: {cur_dir}")
        maya_files = list(cur_dir.rglob("*.m[ab]"))
        for maya_file in maya_files:
            try:
                # Convert to fbx
                self.convert_file(maya_file)
            except Exception as e:
                logging.error(str(e))
            finally:
                # Delete the file
                logging.info(f"Deleting file... {maya_file}")
                os.remove(maya_file)

    # ----------------------------------------------------------------------------------------------------------------------

    def convert_file(self, maya_file):
        logging.info(f"Converting file... {maya_file}")
        cmds.file(newFile=True, force=True)
        cmds.file({maya_file}, i=True)
        cmds.select(all=True)
        dest_file = str(maya_file) + ".fbx"
        logging.info(f"Destination file: {dest_file}")
        maya_command = "FBXExport -f" + " \"" + dest_file + "\""
        mel.eval(maya_command)

# ----------------------------------------------------------------------------------------------------------------------

def welcome_message():
    logging.info("""Welcome to Perforce Maya render service""")

# ----------------------------------------------------------------------------------------------------------------------
if __name__ == "__main__":
    # Initialise logger
    root = logging.getLogger()
    root.setLevel(logging.INFO)
    welcome_message()

    home_dir = Path("/home/perforce/p4render-maya/files")
    main = Main(home_dir)
    # Wait forever
    while True:
        try:
            main.process_files()
            sleep(1)
        except Exception as e:
            logging.error(str(e))
            logging.info(f"Cleaning up... {home_dir}")
            for root, dirs, files in os.walk(home_dir):
                for f in files:
                    os.unlink(os.path.join(root, f))
                for d in dirs:
                    shutil.rmtree(os.path.join(root, d), ignore_errors=True)
            sys.exit(1)
