<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */
namespace Slack\Service;

use Activity\Model\Activity;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition as IDef;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\Factory\InvokableService;
use Application\Log\SwarmLogger;
use Exception;
use Interop\Container\ContainerInterface;
use Laminas\Http\Client;
use Laminas\Http\Request;
use Notifications\Model\INotification;
use P4\Spec\Change;
use Projects\Model\Project;
use Record\Key\AbstractKey;
use Reviews\Model\Review;
use Slack\Model\Message;
use Slack\Model\Slack as SlackModel;

/**
 * Class SlackService.
 * @package Slack\Service
 */
class Slack implements ISlack, InvokableService
{
    private $services;
    private $logger;
    private $utility;
    const LOG_PREFIX = Slack::class;
    private $purePrivate;

    /**
     * Slack service constructor.
     * @param ContainerInterface $services
     * @param array|null $options
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services = $services;
        $this->logger   = $services->get(SwarmLogger::SERVICE);
        $this->utility  = $services->get(IUtility::SERVICE_NAME);
    }

    /**
     * @inheritDoc
     */
    public function handleCreateSlackMessage(Change $change, Activity $activity, $review = null, $projects = null)
    {
        $topic    = $this->getTopic($change, $review);
        $services = $this->services;
        $dao      = $services->get(IDao::SLACK_DAO);
        $p4       = $services->get(ConnectionFactory::P4_ADMIN);

        $this->logger->trace(
            sprintf("%s: Topic %s is running handleCreateSlackMessage", self::LOG_PREFIX, $topic)
        );
        $slackModelExists = $dao->exists($topic, $p4);

        // If we have a model just return early as we shouldn't be creating a new post if it exists.
        if ($slackModelExists) {
            return;
        }
        $services = $this->services;
        $config   = $services->get(IDef::CONFIG);

        $bypassRestrictedChangelist = ConfigManager::getValue(
            $config,
            IDef::SLACK_BYPASS_RESTRICTED_CHANGELIST,
            false
        );

        $type = $change->getType();
        if ($type !== Change::PUBLIC_CHANGE && !$bypassRestrictedChangelist) {
            $this->logger->debug(sprintf("%s: Not allowed to post private information...", self::LOG_PREFIX));
            return;
        }
        $projects = $projects ?? $this->getProjects($change, $review);
        $message  = new Message($this->services, $change, $projects, $review, $activity);

        $channels = ConfigManager::getValue($config, IDef::SLACK_PROJECT_CHANNELS, []);

        $channelThreads = null;
        // Channels this review is associated with
        $projectChannels = $this->utility->getProjectMappings($projects, $channels, $this->purePrivate);
        $this->logger->trace(
            sprintf("%s: ProjectChannels are:", self::LOG_PREFIX)
            . json_encode($projectChannels, true)
        );
        foreach ($projectChannels as $channel => $linkedProjects) {
            $this->logger->debug(sprintf("%s: posting to %s", self::LOG_PREFIX, $channel));
            $response = $this->postSlackMessage($message->getSummary($channel, $linkedProjects));
            $this->getChannelThread($channelThreads, $response, $channel);
        }
        $this->createSlackModel($change, $channelThreads, $review);
    }

    /**
     * @inheritDoc
     */
    public function handleMissingSlackMessage(
        Change $change,
        string $channel,
        Review $review,
        array $projects = null,
        array $linkedProjects = []
    ): ?string {
        $services = $this->services;
        $config   = $services->get(IDef::CONFIG);

        $bypassRestrictedChangelist = ConfigManager::getValue(
            $config,
            IDef::SLACK_BYPASS_RESTRICTED_CHANGELIST,
            false
        );

        $type = $change->getType();
        if ($type !== Change::PUBLIC_CHANGE && !$bypassRestrictedChangelist) {
            $this->logger->debug(sprintf("%s: Not allowed to post private information...", self::LOG_PREFIX));
            return '';
        }
        $message = new Message($this->services, $change, $projects, $review);

        $channelThreads = null;
        $this->logger->debug(sprintf("%s: posting to %s", self::LOG_PREFIX, $channel));
        $response = $this->postSlackMessage($message->getSummary($channel, $linkedProjects));
        $this->getChannelThread($channelThreads, $response, $channel);
        $this->createSlackModel($change, $channelThreads, $review);
        return $channelThreads[$channel]["threadId"];
    }

    /**
     * @inheritDoc
     */
    public function handleThreadedSlackMessage($change, $activity, Review $review = null)
    {
        $topic    = $this->getTopic($change, $review);
        $projects = $this->getProjects($change, $review);
        $message  = new Message($this->services, $change, $projects, $review);
        $services = $this->services;
        $dao      = $services->get(IDao::SLACK_DAO);
        $p4       = $services->get(ConnectionFactory::P4_ADMIN);

        $this->logger->trace(
            sprintf("%s: Topic %s is running handleThreadedSlackMessage", self::LOG_PREFIX, $topic)
        );
        $slackModelExists = $dao->exists($topic, $p4);

        // If we have no model for this topic attempt to create them.
        if (!$slackModelExists) {
            $this->logger->warn(sprintf("%s: No threaded ids for %s", self::LOG_PREFIX, $topic));
            $this->handleCreateSlackMessage($change, $activity, $review, $projects);
        }
        $slackModel = $dao->fetchById($topic, $p4);
        $threadIds  = $slackModel->getThreadIds();
        $config     = $services->get(IDef::CONFIG);
        $channels   = ConfigManager::getValue($config, IDef::SLACK_PROJECT_CHANNELS, []);

        $projectChannels = $this->utility->getProjectMappings($projects, $channels, $this->purePrivate);
        $this->logger->trace(
            sprintf("%s: ProjectChannels are:", self::LOG_PREFIX)
            . json_encode($projectChannels, true)
        );
        foreach ($projectChannels as $channel => $linkedProjects) {
            $this->logger->debug(sprintf("%s: posting to %s", self::LOG_PREFIX, $channel));
            $parent = $threadIds[$channel]['threadId'] ?? null;
            if ($parent) {
                $this->postSlackMessage($message->getReply($activity, $channel, $parent, $linkedProjects));
            } else {
                $parent = $this->handleMissingSlackMessage($change, $channel, $review, $projects, $linkedProjects);
                if ($parent) {
                    $this->postSlackMessage($message->getReply($activity, $channel, $parent, $linkedProjects));
                } else {
                    $this->logger->warn(sprintf("%s: No threaded ids for %s", self::LOG_PREFIX, $channel));
                }
            }
        }
    }

    /**
     * @inheritDoc
     */
    public function postSlackMessage($body)
    {
        $config = $this->services->get(IDef::CONFIG);
        $this->logger->debug(sprintf("%s: posting message", self::SERVICE_NAME));

        $token   = ConfigManager::getValue($config, IDef::SLACK_TOKEN);
        $enabled = ConfigManager::getValue($config, IDef::SLACK_USER_ENABLED, true);
        $user    = null;
        $icon    = null;

        if ($enabled === true) {
            $user            = ConfigManager::getValue($config, IDef::SLACK_USER_NAME);
            $icon            = ConfigManager::getValue($config, IDef::SLACK_USER_ICON, null);
            $forceUserHeader = ConfigManager::getValue($config, IDef::SLACK_USER_FORCE_USER_HEADER, false);

            if ($forceUserHeader === true) {
                $user = $user . "-" . $this->getTime();
            }
        }

        $body['username'] = $user;
        $body['icon_url'] = $icon;
        $endpoint         = "https://slack.com/api/chat.postMessage";
        $headers          = [
            "Content-type: application/json",
            "Authorization: Bearer " . $token
        ];

        $json = json_encode($body);
        return $this->postSlackAPI($endpoint, $headers, $json);
    }

    /**
     * This is for only tests to be mock.
     * @return int
     */
    public function getTime(): int
    {
        return time();
    }

    /**
     * @inheritDoc
     */
    public function postSlackAPI($endpoint, $headers, $body, bool $json = true)
    {
        $config = $this->services->get(IDef::CONFIG);
        $this->logger->debug(sprintf("%s: posting to endpoint %s", self::SERVICE_NAME, $endpoint));
        $response = [];
        try {
            $request = new Request();
            $request->setMethod(Request::METHOD_POST);
            $request->setUri($endpoint);
            $request->getHeaders()->addHeaders($headers);

            if ($json === true) {
                $request->setContent($body);
            } else {
                foreach ($body as $key => $value) {
                    $request->getPost()->set($key, $value);
                }
            }
            $this->logger->debug(sprintf("%s: request headers %s", self::SERVICE_NAME, json_encode($headers)));
            $this->logger->debug(sprintf("%s: request body %s", self::SERVICE_NAME, $body));

            $client = new Client();
            $client->setEncType(Client::ENC_FORMDATA);
            $client->setUri($endpoint)
                   ->setHeaders($headers);

            // Set the http client options; including any special overrides for our host
            $options = $config + [
                'http_client_options' => []
                ];
            $options = (array)$options['http_client_options'];
            $this->logger->trace(sprintf("%s: slack host %s", self::SERVICE_NAME, $client->getUri()->getHost()));
            if (isset($options['hosts'][$client->getUri()->getHost()])) {
                $this->logger->trace(
                    sprintf(
                        "%s: slack host matched applying options for %s",
                        self::SERVICE_NAME,
                        $client->getUri()->getHost()
                    )
                );
                $options = (array)$options['hosts'][$client->getUri()->getHost()] + $options;
            }
            unset($options['hosts']);
            $client->setOptions($options);

            // POST request
            $response = $client->dispatch($request);
            $this->logger->debug(sprintf("%s: slack response %s", self::SERVICE_NAME, $response->getBody()));

            if (!$response->isSuccess()) {
                $this->logger->err(
                    sprintf(
                        "%s: failed to post to slack endpoint %s, error code %s, reason %s",
                        self::SERVICE_NAME,
                        $endpoint,
                        $response->getStatusCode(),
                        $response->getReasonPhrase()
                    )
                );
            }
            // Decode the response so we can access data.
            $response = json_decode($response->getBody());
        } catch (Exception $e) {
            $this->logger->err(sprintf("%s: unexpected error %s", self::SERVICE_NAME, $e->getMessage()));
        }
        return $response;
    }

    /**
     * Common function to get the channelThreads
     *
     * @param array|null $channelThreads The array of channels and their thread ids.
     * @param object     $response       The response from the request.
     * @param string     $channel        The channel we are working with.
     * @return void
     */
    private function getChannelThread(?array &$channelThreads, object $response, string $channel)
    {
        $parentThread = $response->ts ?? null;
        if ($parentThread) {
            $channelThreads[$channel]["threadId"] = $parentThread;
        } else {
            $this->logger->debug(
                sprintf(
                    "%s: found no thread id for post response for %s",
                    self::LOG_PREFIX,
                    $channel
                )
            );
        }
    }

    /**
     * If we have ChannelThreads to create we should do that now.
     *
     * @param Change      $change         - The changelist object
     * @param array|null  $channelThreads - The array of channels and their thread ids.
     * @param Review|null $review         - If set, the changelist object represents a review rather than a change
     * @return void
     */
    private function createSlackModel(Change $change, ?array $channelThreads, ?Review $review)
    {
        if ($channelThreads) {
            $id = $review
                ? INotification::REVIEW_TOPIC . $review->getId() : INotification::CHANGE_TOPIC . $change->getId();
            try {
                $this->logger->debug(sprintf("%s: Creating a thread ID for %s", self::LOG_PREFIX, $id));
                $services   = $this->services;
                $p4         = $services->get(ConnectionFactory::P4_ADMIN);
                $slackDao   = $this->services->get(IDao::SLACK_DAO);
                $slackExist = $slackDao->exists($id, $p4);
                if ($slackExist) {
                    $slack     = $slackDao->fetchById($id, $p4);
                    $threadIds = $slack->getThreadIds();
                    $slack->setThreadIds(array_merge($channelThreads, $threadIds));
                } else {
                    $slack = new SlackModel($p4, $id);
                    $slack->setThreadIds($channelThreads);
                }
                $slackDao->save($slack);
            } catch (Exception $saveThreadError) {
                $this->logger->debug(sprintf("%s: failed to create a thread ID for %s", self::LOG_PREFIX, $id));
                $this->logger->debug($saveThreadError->getMessage());
            }
        }
    }

    /**
     * If the review is not null get the projects from the review, otherwise find the affected projects
     * based on the change
     * @param Change $change the change
     * @param mixed $review the review
     * @return mixed
     */
    private function getProjects(Change $change, $review = null)
    {
        $services = $this->services;
        $p4       = $services->get(ConnectionFactory::P4_ADMIN);

        if ($review !== null) {
            $projects = $review->getProjects();
        } else {
            $affected = $services->get(Services::AFFECTED_PROJECTS);
            $projects = $affected->findByChange($p4, $change);
        }
        // If we have projects we will fetch all the projects and filter out private projects. If we are returned no
        // projects we can assume that all projects are private. If one is returned then one of the linked projects is
        // public, so we can publish the message to the public.
        if ($projects) {
            $projectDAO        = $services->get(IDao::PROJECT_DAO);
            $this->purePrivate = !$projectDAO->fetchAll(
                [
                    AbstractKey::FETCH_BY_IDS => array_unique(array_keys($projects)),
                    Project::FETCH_INCLUDE_PRIVATE => false
                ],
                $p4
            )->count();
        }
        return $projects;
    }

    /**
     * Return the topic.
     *
     * @param Change      $change - The changelist object
     * @param Review|null $review - If set, the changelist object represents a review rather than a change
     * @return string
     */
    public function getTopic(Change $change, ?Review $review): string
    {
        return $review
            ? INotification::REVIEW_TOPIC . $review->getId() : INotification::CHANGE_TOPIC . $change->getId();
    }
}
