<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Slack\Listener;

use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition as IDef;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Log\SwarmLogger;
use Comments\Model\Comment as CommentModel;
use Events\Listener\AbstractEventListener;
use Laminas\ServiceManager\ServiceLocatorInterface as ServiceLocator;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Record\Lock\Lock;
use Reviews\Model\IReview;
use Reviews\Model\Review;
use Slack\Model\Slack;
use Slack\Service\ISlack;
use Laminas\EventManager\Event;
use Exception;

/**
 * Comment method that calls other services.
 */
class Comment extends AbstractEventListener
{
    const LOG_PREFIX = Comment::class;

    protected mixed $slackService = null;

    /**
     * Ensure we get a service locator and event config on construction.
     *
     * @param ServiceLocator $services    the service locator to use
     * @param array          $eventConfig the event config for this listener
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function __construct(ServiceLocator $services, array $eventConfig)
    {
        parent::__construct($services, $eventConfig);
        $this->slackService = $this->services->get(ISlack::SERVICE_NAME);
    }

    /**
     * Attaches this event only if the Slack token has been provided.
     *
     * @param mixed $eventName   the event name
     * @param array $eventDetail the event detail
     * @return bool true if the Slack token is set
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function shouldAttach($eventName, $eventDetail): bool
    {
        $config = $this->services->get(IDef::CONFIG);
        try {
            ConfigManager::getValue($config, IDef::SLACK_TOKEN);
        } catch (Exception $e) {
            return false;
        }
        $this->logger->trace(
            sprintf(
                "%s: Slack token has been provided",
                self::LOG_PREFIX
            )
        );
        return true;
    }

    /**
     * This function is called when a review comment is event
     *
     * @param Event $event
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws ConfigException
     */
    public function handleReview(Event $event): void
    {
        parent::log($event);
        $services = $this->services;
        $logger   = $services->get(SwarmLogger::SERVICE);
        $id       = $event->getParam(IReview::FIELD_ID);
        $activity = $event->getParam('activity');
        $p4Admin  = $services->get(ConnectionFactory::P4_ADMIN);

        try {
            // fetch comment record
            $comment = CommentModel::fetch($id, $p4Admin);
            $topic   = $comment->get('topic');
            // handle review comments
            if (str_starts_with($topic, 'reviews/')) {
                $context  = $comment->getFileContext();
                $review   = $context['review'] ?: str_replace('reviews/', '', $topic);
                $review   = Review::fetch($review, $p4Admin);
                $reviewId = $review->getId();
                $lock     = new Lock(Slack::KEY_PREFIX . "review-" . $reviewId, $p4Admin);
                try {
                    $lock->lock();
                    $changeDAO = $this->services->get(IDao::CHANGE_DAO);
                    $change    = $changeDAO->fetchById($review->getChanges()[0], $p4Admin);
                    $this->slackService->handleThreadedSlackMessage($change, $activity, $review);
                } catch (Exception $e) {
                    $logger->err($e);
                } finally {
                    $lock->unlock();
                }
            }
        } catch (Exception $e) {
            $logger->err($e);
        }
    }
}
