<?php

namespace Projects\Controller;

use Api\Controller\AbstractRestfulController;
use Api\IRequest;
use Application\Checker;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\I18n\TranslatorFactory;
use Application\InputFilter\InputFilter;
use Application\Model\IModelDAO;
use Application\Permissions\Exception\ForbiddenException;
use Application\Permissions\Exception\UnauthorizedException;
use Application\Permissions\IPermissions;
use Application\Validator\ConnectedAbstractValidator;
use Exception;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use Projects\Checker\IProject as ProjectChecks;
use Projects\Model\IProject;
use Projects\Validator\Branches;
use Projects\Validator\BranchPath;
use Record\Exception\NotFoundException as RecordNotFoundException;
use InvalidArgumentException;

class BranchApi extends AbstractRestfulController
{
    const DATA_BRANCHES = IProject::BRANCHES;
    const IDENTIFIER    = 'branch';

    // route for this is .../projects/:id/branches[/:branch][?...]
    protected $identifierName = self::IDENTIFIER;

    /**
     * Gets all branch data for a given project
     * Example success response
     * {
     *  "error": null,
     *  "messages": [],
     *  "data": {
     *        "branches": [
     *        {
     *            id: "dev",
     *            name: "Dev",
     *            workflow: 1,
     *            paths: [],
     *            ...
     *         },
     *        {
     *            id: "main",
     *            name: "Main",
     *            workflow: null,
     *            paths: [],
     *            ...
     *         },
     *      ],
     *    }
     * }
     *
     * Query parameters supported:
     *  fields - filter by fields
     *
     * Example error response
     *
     * Unauthorized response 401, if require_login is true
     * {
     *   "error": "Unauthorized"
     * }
     *
     * 500 error response
     * {
     *   "error": 500,
     *   "messages": [
     *       {
     *           "code": 500,
     *           "text": "Something really bad happened"
     *       }
     *   ],
     *   "data": null
     * }
     * @param mixed $id The Branch ID
     * @return JsonModel
     */
    public function getList()
    {
        $services  = $this->services;
        $projectId = $this->getEvent()->getRouteMatch()->getParam(IProject::ID);
        $p4Admin   = $services->get(ConnectionFactory::P4_ADMIN);
        $dao       = $services->get(IDao::PROJECT_DAO);
        $fields    = $this->getRequest()->getQuery(IRequest::FIELDS);
        $errors    = null;
        $project   = null;
        try {
            $project = $dao->fetchById($projectId, $p4Admin, [IModelDAO::FILTER_PRIVATES => true]);
        } catch (\InvalidArgumentException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = $this->buildMessage(Response::STATUS_CODE_400, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($errors) {
            $json = $this->error([$errors], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_BRANCHES => $this->limitFieldsForAll($project->getBranches(), $fields)]);
        }
        return $json;
    }

    /**
     * Gets a branch
     * Example success response
     * {
     *  "error": null,
     *  "messages": [],
     *  "data": {
     *        "branches": [
     *        {
     *            id: "main",
     *            name: "Main",
     *            workflow: null,
     *            paths: [],
     *            ...
     *         },
     *      ],
     *    }
     * }
     *
     * Query parameters supported:
     *  fields - filter by fields
     *
     * Example error response
     *
     * Unauthorized response 401, if require_login is true
     * {
     *   "error": "Unauthorized"
     * }
     *
     * 404 error response
     * {
     *   "error": 404,
     *   "messages": [
     *       {
     *           "code": 404,
     *           "text": Cannot fetch entry. mybranch does not exist in the empty project."
     *       }
     *   ],
     *   "data": null
     * }
     *
     * 500 error response
     * {
     *   "error": 500,
     *   "messages": [
     *       {
     *           "code": 500,
     *           "text": "Something really bad happened"
     *       }
     *   ],
     *   "data": null
     * }
     * @param mixed $id The Branch ID
     * @return JsonModel
     */
    public function get($id)
    {
        $services  = $this->services;
        $projectId = $this->getEvent()->getRouteMatch()->getParam(IProject::ID);
        $p4Admin   = $services->get(ConnectionFactory::P4_ADMIN);
        $dao       = $services->get(IDao::PROJECT_DAO);
        $fields    = $this->getRequest()->getQuery(IRequest::FIELDS);
        $errors    = null;
        $branch    = null;
        try {
            $project = $dao->fetchById($projectId, $p4Admin, [IModelDAO::FILTER_PRIVATES => true]);
            $branch  = array_values(
                array_filter(
                    $project->getBranches(),
                    function ($branch) use ($id) {
                        return $branch[IProject::ID] === $id;
                    }
                )
            );
            if (!count($branch)) {
                throw new RecordNotFoundException(
                    $services->get(TranslatorFactory::SERVICE)->t(
                        "Cannot fetch entry. %s does not exist in the %s project.",
                        [$id, $projectId]
                    )
                );
            }
            $branch = $this->limitFieldsForAll($branch, $fields);
        } catch (\InvalidArgumentException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = $this->buildMessage(Response::STATUS_CODE_400, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            // Project/branch id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($errors) {
            $json = $this->error([$errors], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_BRANCHES =>$branch]);
        }
        return $json;
    }

    /**
     * Create a branch for the project with the id in the request path
     * Example success
     * Data payload
     *  {
     *      "id": "one",
     *      "name": "one",
     *      "paths": ["//depot/saturn/...   "],
     *      ...
     *  }
     * Response
     * {
     *  "error": null,
     *  "messages": [],
     *  "data": {
     *        "branches": [
     *        {
     *            id: "one",
     *            name: "one",
     *            workflow: null,
     *            paths: [],
     *            ...
     *         },
     *      ],
     *    }
     * }
     *
     * Example error response
     *
     * Unauthorized response 401, if require_login is true
     * {
     *   "error": "Unauthorized"
     * }
     *
     * 500 error response
     * {
     *   "error": 500,
     *   "messages": [
     *       {
     *           "code": 500,
     *           "text": "Something really bad happened"
     *       }
     *   ],
     *   "data": null
     * }
     *
     * 400 error response for invalid values
     * {
     *   "error": 400,
     *   "messages": [
     *      {
     *          "branches": {
     *          "duplicateId": "Branches cannot have the same id, 'main' is already in use."
     *      }
     *  ],
     *  "data": null
     * }
     * @param mixed $data branch data
     * @return JsonModel
     */
    public function create($data): JsonModel
    {
        $services = $this->services;
        $errors   = null;
        $project  = null;
        try {
            $projectId = $this->getEvent()->getRouteMatch()->getParam(IProject::ID);
            $p4Admin   = $services->get(ConnectionFactory::P4_ADMIN);
            $dao       = $services->get(IDao::PROJECT_DAO);
            $project   = $dao->fetchById($projectId, $p4Admin, [IModelDAO::FILTER_PRIVATES => true]);
            $services->get(Services::CONFIG_CHECK)->checkAll(
                [
                    IPermissions::AUTHENTICATED_CHECKER,
                    ProjectChecks::BRANCHES_ADMIN_ONLY_CHECKER,
                    ProjectChecks::PROJECT_EDIT_CHECKER + [Checker::OPTIONS => [Checker::VALUE => $project]]
                ]
            );
            $currentBranches = $project->getBranches();
            $filter          = $services->build(
                Services::BRANCHES_FILTER,
                [
                    Branches::EXISTING_BRANCH_IDS => array_map(
                        function ($branch) {
                            return $branch[IProject::ID];
                        },
                        $currentBranches
                    )
                ]
            );
            $data['paths']   = array_values(array_filter($data['paths'], 'strlen'));
            $filter->setData([IProject::BRANCHES=>[$data]]);
            if ($filter->isValid()) {
                $createdBranch             = $filter->getValues()[IProject::BRANCHES];
                $createdBranchPaths        = $createdBranch[0]['paths'];
                $branchPaths               = $dao->expandPaths($createdBranchPaths);
                $createdBranch[0]['paths'] = $branchPaths;
                $project                   = $project->setBranches(array_merge($currentBranches, $createdBranch));
                $dao->save($project);
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $errors = $filter->getMessages();
            }
        } catch (InvalidArgumentException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = $this->buildMessage(Response::STATUS_CODE_400, $e->getMessage());
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = $this->buildMessage(Response::STATUS_CODE_403, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = $this->buildMessage(Response::STATUS_CODE_401, 'Unauthorized');
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($errors) {
            $json = $this->error([$errors], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_BRANCHES => $project->getBranches()]);
        }
        return $json;
    }

    /**
     * Update a branch for the project with the id in the request path
     * Example success
     * Data payload
     *  {
     *      "name": "one",
     *      "paths": ["//depot/saturn/...   "],
     *      ...
     *  }
     * Response
     * {
     *  "error": null,
     *  "messages": [],
     *  "data": {
     *        "branches": [
     *        {
     *            id: "one",
     *            name: "one",
     *            workflow: null,
     *            paths: [],
     *            ...
     *         },
     *      ],
     *    }
     * }
     *
     * Example error response
     *
     * Unauthorized response 401, if require_login is true
     * {
     *   "error": "Unauthorized"
     * }
     *
     * 500 error response
     * {
     *   "error": 500,
     *   "messages": [
     *       {
     *           "code": 500,
     *           "text": "Something really bad happened"
     *       }
     *   ],
     *   "data": null
     * }
     *
     * 400 error response for invalid values
     * {
     *   "error": 400,
     *   "messages": [
     *      {
     *          "branches": {
     *          "duplicateId": "Branches cannot have the same id, 'main' is already in use."
     *      }
     *  ],
     *  "data": null
     * }
     * @param $id
     * @param mixed $data branch data
     * @return JsonModel
     */
    public function patch($id, $data): JsonModel
    {
        $services = $this->services;
        $errors   = null;
        $project  = null;
        try {
            $p4Admin   = $services->get(ConnectionFactory::P4_ADMIN);
            $dao       = $services->get(IDao::PROJECT_DAO);
            $projectId = $this->getEvent()->getRouteMatch()->getParam(IProject::ID);
            $project   = $dao->fetchById($projectId, $p4Admin, [IModelDAO::FILTER_PRIVATES => true]);
            $services->get(Services::CONFIG_CHECK)->checkAll(
                [
                    IPermissions::AUTHENTICATED_CHECKER,
                    ProjectChecks::BRANCHES_ADMIN_ONLY_CHECKER,
                    ProjectChecks::PROJECT_EDIT_CHECKER + [Checker::OPTIONS => [Checker::VALUE => $project]]
                ]
            );
            $branches    = $project->getBranches();
            $branchIndex = key($this->getBranch($branches, $id)??[]);
            if ($branchIndex===null) {
                throw new RecordNotFoundException(
                    $services->get(TranslatorFactory::SERVICE)->t(
                        "The %s project does not have a %s branch.",
                        [$projectId, $id]
                    )
                );
            }
            $thisBranch = $branches[$branchIndex];
            foreach ($data as $field => $value) {
                $thisBranch[$field] = $value;
            }
            $filter = $services->build(
                Services::BRANCHES_FILTER,
                [
                    Branches::EXISTING_BRANCH_IDS =>
                        array_map(
                            function ($branch) {
                                return $branch[IProject::ID];
                            },
                            array_filter(
                                $branches,
                                function ($branch) use ($id) {
                                    return $branch[IProject::ID] !== $id;
                                }
                            )
                        ),
                    "validateWorkflowOwner" => array_key_exists('workflow', $data)
                ]
            )->setMode(InputFilter::MODE_EDIT)->setData([IProject::BRANCHES=>[$thisBranch]]);
            if ($filter->isValid()) {
                $branches[$branchIndex]  = $filter->getValues()[IProject::BRANCHES][0];
                $branchPaths             = $dao->expandPaths($branches[$branchIndex][IProject::BRANCH_PATHS]);
                $branches[$branchIndex]
                [IProject::BRANCH_PATHS] = $branchPaths;
                $project->setBranches($branches);
                $project = $dao->save($project);
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $errors = $filter->getMessages();
            }
        } catch (\InvalidArgumentException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = $this->buildMessage(Response::STATUS_CODE_400, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = $this->buildMessage(Response::STATUS_CODE_403, $e->getMessage());
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, 'Unauthorized')];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($errors) {
            $json = $this->error([$errors], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_BRANCHES => $project->getBranches()]);
        }
        return $json;
    }

    /**
     * Hard delete a project branch. Requires authentication and that the user must be an owner
     * if the project has owners or a member if there are no owners.
     * Example: DELETE http://<host>/api/<version>/projects/<project-id>/branches/<branch-id>,
     * responds with the project
     * 200:
     * {
     *   "error": null,
     *   "messages": [],
     *   "data": {
     *         "branches": [
     *                       {
     *                      "id": "development",
     *                      "name": "Development",
     *                       "workflow": "1",
     *                           "paths": [
     *                           "//jam/main/src/..."
     *                           ],
     *                           "defaults": {
     *                           "reviewers": {
     *                           "Anna_Schmidt": [],
     *                           "dai": {
     *                           "required": true
     *                           },
     *                           "swarm-group-WebDesigners": {
     *                           "required": true
     *                           },
     *                           "swarm-group-WebMasters": {
     *                           "required": "1"
     *                           }
     *                           }
     *                           },
     *                           "minimumUpVotes": 1,
     *                           "retainDefaultReviewers": true,
     *                           "moderators-groups": [
     *                           "testers"
     *                           ],
     *                           "moderators": []
     *                           },
     *                       ]
     *   }
     * }
     *
     * Where a branch does not exist or access is not permitted
     *
     * 404:
     * {
     *       "error": 404,
     *       "messages": [
     *          {
     *              "code": 404
     *              "text": "Cannot fetch entry. Id does not exist."
     *          }
     *       ],
     *       "data": null
     * }
     *
     * When a user does not have the correct privilege
     *
     * 401:
     * {
     *       "error": 401,
     *       "messages": [
     *          {
     *              "code": 401
     *              "text": "This operation is limited to project or group owners."
     *          }
     *       ],
     *       "data": null
     * }
     *
     * Unauthorized
     * {
     *   "error": "Unauthorized",
     * }
     * @param mixed $strId id of the branch to delete
     * @return JsonModel the project that has list of all remaining branches after hard deleting branch
     */
    public function delete($strId) : JsonModel
    {
        $arrMixErrors     = [];
        $objProject       = null;
        $isPathPermission = true;
        try {
            $objP4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
            $objProjectDao = $this->services->get(IDao::PROJECT_DAO);
            $intProjectId  = $this->getEvent()->getRouteMatch()->getParam(IProject::ID);
            $objProject    = $objProjectDao->fetchById(
                $intProjectId,
                $objP4Admin,
                [IModelDAO::FILTER_PRIVATES => true]
            );
            $objChecker    = $this->services->get(Services::CONFIG_CHECK);
            // Authentication is required, delete is permitted if
            // - user is an admin
            // - user is an owner if the project has owners
            // - user is a member if there are no owners
            $objChecker->checkAll(
                [
                    IPermissions::AUTHENTICATED_CHECKER,
                    ProjectChecks::PROJECT_EDIT_CHECKER + [Checker::OPTIONS => [Checker::VALUE => $objProject]]
                ]
            );
            $arrMixBranches = $objProject->getBranches();
            $intBranchIndex = key($this->getBranch($arrMixBranches, $strId)??[]);
            if (!$arrMixBranches || $intBranchIndex===null) {
                throw new RecordNotFoundException(
                    $this->services->get(TranslatorFactory::SERVICE)->t(
                        "The %s project does not have a %s branch.",
                        [$intProjectId, $strId]
                    )
                );
            }
            // check if user is having correct permission for all the branch paths of the deleted branch
            if (!empty($arrMixBranches[$intBranchIndex][IProject::BRANCH_PATHS])) {
                $pathValidator = new BranchPath(
                    [
                        ConnectedAbstractValidator::CONNECTION => $this->services->get(ConnectionFactory::P4_ADMIN),
                        ConnectedAbstractValidator::SERVICES => $this->services
                    ]
                );
                if (!$pathValidator->isValid($arrMixBranches[$intBranchIndex][IProject::BRANCH_PATHS])) {
                    $isPathPermission = false;
                    $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                    $arrMixErrors = $pathValidator->getMessages();
                }
            }

            if ($isPathPermission) {
                unset($arrMixBranches[$intBranchIndex]);
                // Adding array_values to reshuffle the index.
                // As based on index front end logic is written to display list of branches
                $arrMixUpdatedBranchesList = array_values($arrMixBranches);
                $objProject->setBranches($arrMixUpdatedBranchesList);
                $objProjectDao->save($objProject);
            }
        } catch (UnauthorizedException|ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $arrMixErrors = [$this->buildMessage(Response::STATUS_CODE_401, $e->getMessage())];
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $arrMixErrors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $arrMixErrors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($arrMixErrors) {
            $json = $this->error([$arrMixErrors], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_BRANCHES => $objProject->getBranches()]);
        }
        return $json;
    }

    /**
     * Get the entry in the branches array for a given id, note that the returned element preserves
     * the index of the original array.
     * @param $branches array the list of branches
     * @param $id string the id to find
     * @return array
     */
    protected function getBranch($branches, $id) : array
    {
        return array_filter(
            $branches,
            function ($branch) use ($id) {
                return $branch[IProject::ID] === $id;
            }
        );
    }
}
