<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Files\Model;

use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\Helper\StringHelper;
use Application\Model\AbstractDAO;
use Application\Permissions\Exception\ForbiddenException;
use Application\Permissions\IpProtects;
use Application\Permissions\Protections;
use Exception;
use Files\Filter\IFile;
use Files\Filter\Diff\IDiff;
use P4\Connection\Connection;
use P4\Connection\ConnectionInterface;
use P4\Exception as P4Exception;
use P4\File\Diff;
use P4\File\Exception\Exception as FileException;
use P4\File\Exception\NotFoundException as FileNotFoundException;
use P4\File\File;
use P4\Filter\Utf8;
use P4\OutputHandler\Limit;
use P4\Spec\Change;
use P4\Spec\Depot;
use P4\Spec\Exception\NotFoundException as SpecNotFoundException;
use P4\Spec\Stream;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;

/**
 * Class FileDAO
 * @package Files\Model
 */
class FileDAO extends AbstractDAO
{
    const MODEL = File::class;

    // Defaults
    const DEFAULT_CONTEXT_LINES = 5; //used this in case of config exception for lines
    const DEFAULT_MAX_SIZE      = 1048576; //1MB used this in case of config exception for maxSize
    const STREAM                = 'stream';
    const FILE                  = 'file';
    const LEFT                  = 'left';
    const RIGHT                 = 'right';
    const MAX_ROWS_IN_EDIT_TYPE = 500; //(must in integer) maximum number of rows shown in edit type diff
    const MAX_CHUNK_SIZE_VALUE  = 100020; //(must in bytes) maximum size of the diff chunk (0.3 MB or 300kb)

    const MAX_CHUNK_SIZE = 'maxChunkSize';
    // Based on these value show more content is shown for large content file
    const CONTENT_CHUNK            = 'isContentChunk';
    const CONTENT_CHUNK_START      = 'contentChunkStart';
    const CONTENT_CHUNK_END        = 'contentChunkEnd';
    const CONTENT_CHUNK_START_LINE = 'contentChunkStartLine';
    const CONTENT_CHUNK_END_LINE   = 'contentChunkEndLine';
    const WHOLE_DIFF_CONTENT_SIZE  = 'wholeDiffSize';
    const CURRENT_DIFF_CHUNK_SIZE  = 'currentDiffChunkSize';


    const ORIGINAL_LINES_DATA = 'originalLinesData';
    const ORIGINAL_LINES      = 'originalLines';
    // Based on these value show more content is shown for large content file
    const CONTENT_CHUNK_LEFT_START        = 'contentChunkLeftStart';
    const CONTENT_CHUNK_LEFT_TOTAL_LINES  = 'contentChunkLeftTotalLines';
    const CONTENT_CHUNK_LEFT_END          = 'contentChunkLeftEnd';
    const CONTENT_CHUNK_LEFT_START_LINE   = 'contentChunkLeftStartLine';
    const CONTENT_CHUNK_LEFT_END_LINE     = 'contentChunkLeftEndLine';
    const CONTENT_CHUNK_RIGHT_START       = 'contentChunkRightStart';
    const CONTENT_CHUNK_RIGHT_END         = 'contentChunkRightEnd';
    const CONTENT_CHUNK_RIGHT_START_LINE  = 'contentChunkRightStartLine';
    const CONTENT_CHUNK_RIGHT_END_LINE    = 'contentChunkRightEndLine';
    const CONTENT_CHUNK_RIGHT_TOTAL_LINES = 'contentChunkRightTotalLines';
    const CURRENT_DIFF_CHUNK_ROWS         = 'currentDiffChunkRows';
    const LARGER_CHUNK                    = 'largerChunk';
    const CONFIG_DIFF_CHUNK_ROW           = 'configDiffChunkRow';

    /**
     * Reads the full content of the file provided by the path
     * @param string        $path       full depot path to the file
     * @param mixed         $connection optional connection to use, defaults to current user
     * @return mixed file content
     * @throws ForbiddenException
     */
    public function read($path, ConnectionInterface $connection = null)
    {
        $this->checkPermission($path, Protections::MODE_READ);
        $fileService = $this->services->get(IConfigDefinition::DEPOT_STORAGE);
        if ($connection) {
            $fileService->setConnection($connection);
        }
        return $fileService->read($path);
    }

    /**
     * Update the content of the file provided by the path.
     * @param string        $path           full depot path to the file
     * @param mixed         $content        file contents to write
     * @param string        $description    description for the change
     * @param mixed         $options        options
     * @return FileUpdateResult
     * @throws ForbiddenException
     * @throws SpecNotFoundException
     */
    public function update(string $path, $content, string $description, $options = [])
    {
        $this->checkPermission($path, Protections::MODE_WRITE);
        $options += [
            IFile::ACTION => IFile::SUBMIT
        ];

        $editedFile  = null;
        $fileService = $this->services->get(IConfigDefinition::DEPOT_STORAGE);
        $connection  = $this->services->get(ConnectionFactory::P4);
        $fileService = $fileService->setConnection($connection);
        $stream      = $this->streamFromFile($path, $connection);
        $clientPool  = $connection->getService('clients');
        try {
            $change = null;
            $file   = $fileService->manipulateFile(
                $path,
                function ($file) use (
                    $path,
                    $content,
                    $description,
                    $options,
                    $connection,
                    $stream,
                    $clientPool,
                    &$change
                ) {
                    if ($stream) {
                        $clientPool->grab();
                        $clientPool->reset(true, $stream);
                    }
                    if ($options[IFile::ACTION] === IFile::SUBMIT) {
                        $file->sync()->edit()->setLocalContents($content);
                    } elseif ($options[IFile::ACTION] === IFile::SHELVE) {
                        // For now shelve into a new change, we may add to this functionality as required later
                        // (may want to pass a changeId via options)
                        $changeService = $this->services->get(Services::CHANGE_SERVICE);
                        $change        = new Change($connection);
                        $file          = $file->sync()->edit($change->getId())->setLocalContents($content);
                        $change        = $change->addFile($file)->setDescription($description)->save();
                        $changeService->shelve($connection, [], $change);
                    }
                    return $description;
                },
                $options
            );
            return new FileUpdateResult($file, $change);
        } finally {
            if ($stream) {
                $clientPool->release();
            }
        }
    }

    /**
     * Grab the depot off the first file and check if it points to a stream depot if so, return the //<depot> followed
     * by path components equal to stream depth (this field is present only on new servers, on older ones we take just
     * the first one)
     * @param string        $file       the file specification
     * @param mixed         $connection the current connection
     * @return string|null a stream name or null if one cannot be determined
     * @throws SpecNotFoundException
     */
    public function streamFromFile($file, $connection)
    {
        $pathComponents = array_filter(explode('/', $file));
        $depot          = Depot::fetchById(current($pathComponents), $connection);
        if ($depot->get('Type') == 'stream') {
            $depth = $depot->hasField('StreamDepth') ? $depot->getStreamDepth() : 1;
            return count($pathComponents) > $depth
                ? '//' . implode('/', array_slice($pathComponents, 0, $depth + 1))
                : null;
        }
        return null;
    }

    /**
     * Check if there is permission on the file for the protections mode
     * @param string        $path       full depot path to the file
     * @param string        $mode       protections mode to check
     * @param string        $type       file/stream
     * @throws ForbiddenException
     */
    public function checkPermission($path, $mode, $type = self::FILE)
    {
        $ipProtects = $this->services->get(IpProtects::IP_PROTECTS);
        if (!$ipProtects->filterPaths($path, $mode)) {
            throw new ForbiddenException(sprintf("You do not have '%s' permission for $type '%s'.", $mode, $path));
        }
    }

    /**
     * Processes a filePath and some params to make a call to the Diff library's diff method.
     * Additionally, does some post-processing of the results, according to the params
     * @param string $filePath   depot path of file/stream
     * @param array $params     diff options
     *
     * @return array               Array with four or five elements
     *                             - lines   - array of unified diff hunks strings
     *                             - isCut   - boolean indicating if diff results were truncated
     *                             - isSame  - boolean indicating if from & to are identical
     *                             - header  - unified diff header string in git format
     *                             - summary - array with number of adds, deletes & updates
     *
     * @throws FileException
     * @throws FileNotFoundException
     * @throws ForbiddenException
     * @throws SpecNotFoundException
     * @throws P4Exception
     * @throws Exception
     */
    public function diff(string $filePath, array $params): array
    {
        // Get the files with their correct revisions
        $p4     = $this->services->get(ConnectionFactory::P4);
        $differ = new Diff($p4);
        // Set the diff options and run the diff
        $options = $this->setDiffOptions($params);
        if ($params[IDiff::TYPE] == self::STREAM) {
            $this->checkPermission($filePath, Protections::MODE_READ, self::STREAM);
            $streamRevisions = $this->getStreamRevisions($filePath, $params, $p4);
            $fromPath        = $streamRevisions[IDiff::FROM];
            $toPath          = $streamRevisions[IDiff::TO];
            $diff            = $differ->diffStream($toPath, $fromPath, $options);
        } else {
            $fileRevisions = $this->getFileRevisions($filePath, $params, $p4);
            $fromFile      = $fileRevisions[IDiff::FROM];
            $toFile        = $fileRevisions[IDiff::TO];
            // Check the file permissions
            // Since we are diffing revisions of the same file, we just need to check the common depot path
            $path = $fromFile ? $fromFile->getDepotFilename() : ($toFile ? $toFile->getDepotFilename() : '');
            $this->checkPermission($path, Protections::MODE_READ);
            $options["isCutEnabled"] = true;
            $diff                    = $this->performdiff($differ, $toFile, $fromFile, $options);
        }
        $this->formatDiff($diff);
        $this->handlePaging($diff, $options);
        return $diff;
    }

    /**
     * Compare left/right files.
     * @param Object $differ Diff library object
     * @param File|null $right optional - right-hand file
     * @param File|null $left optional - left-hand file
     * @param array $options optional - influence diff behavior
     *                                    LINES - number of context lines, (defaults to 5)
     *                                IGNORE_WS - ignore whitespace and line-ending changes (defaults to 0)
     *                                            1 -> IGNORE_ALL_WHITESPACE
     *                                            2 -> IGNORE_WHITESPACE
     *                                            3 -> IGNORE_LINE_ENDINGS
     *                                RAW_DIFFS - if true, get raw diffs
     *                                SUMMARY   - if true and RAW_DIFFS is also true, get summary info as well
     *                             UTF8_CONVERT - attempt to covert non UTF-8 to UTF-8
     *                            UTF8_SANITIZE - replace invalid UTF-8 sequences with �
     *                            SUMMARY_LINES - if we are getting the summary counts of changes will be based on
     *                                            chunks unless this value is true in which case it will be based
     *                                            on lines
     * @return  array   array with three or five elements:
     *                     lines - added/deleted and contextual (common) lines
     *                             for RAW_DIFFS, will be in a unified git format
     *                     isCut - true if lines exceed max filesize (>1MB)
     *                    isSame - true if left and right file contents are equal
     *                    header - header in unified git format (only included if RAW_DIFFS is true)
     *                   summary - array of number of adds, deletes and edits (only included if SUMMARY is true)
     * @throws FileException
     * @throws P4Exception
     */
    public function performDiff(object $differ, File $right = null, File $left = null, array $options = []): array
    {
        $options += [
            Diff::LINES         => static::DEFAULT_CONTEXT_LINES,
            Diff::IGNORE_WS     => 0,
            Diff::UTF8_CONVERT  => false,
            Diff::UTF8_SANITIZE => false,
            Diff::RAW_DIFF      => false,
            Diff::SUMMARY       => false
        ];
        if (!$right && !$left) {
            throw new \InvalidArgumentException("Cannot diff. Must specify at least one file to diff.");
        }
        $diff = [
            Diff::LINES => [],
            Diff::CUT   => false,
            Diff::SAME  => false
        ];
        // only examine contents if both sides are non-binary and at least one has content
        $leftIsBinary    = $left?->isBinary() ?? false;
        $leftHasContent  = $left  && !$left->isDeletedOrPurged();
        $rightIsBinary   = $right?->isBinary() ?? false;
        $rightHasContent = $right && !$right->isDeletedOrPurged();
        $canDiffText     = !$leftIsBinary && !$rightIsBinary && ($leftHasContent || $rightHasContent);
        if ($canDiffText) {
            // if only one file given or either file was deleted/purged,
            // can't use diff2, must print the file contents instead.
            $eitherSideMissing = !$left || !$right || $left->isDeletedOrPurged() || $right->isDeletedOrPurged();

            $diff = $eitherSideMissing ? $this->diffAddDelete($diff, $right, $left, $options)
                : $this->diffEdit($differ, $diff, $right, $left, $options);
        }
        // compare digests if we have no diff lines (need both sides)
        if (isset($diff[Diff::LINES]) && empty($diff[Diff::LINES]) && $left && $right) {
            $leftDigest       = $left->hasStatusField('digest')  ? $left->getStatus('digest')  : null;
            $rightDigest      = $right->hasStatusField('digest') ? $right->getStatus('digest') : null;
            $diff[Diff::SAME] = $leftDigest === $rightDigest;
        }
        if (!(is_null($right) || is_null($left))) {
            $fullContentOption   = [File::MAX_SIZE => 'unlimited'];
            $toFileFullContent   = $right->getFullDepotContents($fullContentOption);
            $fromFileFullContent = $left->getFullDepotContents($fullContentOption);

            $diff[FILE::FETCH_CONTENT] = !(strlen($toFileFullContent) > $options[File::MAX_SIZE]
                || strlen($fromFileFullContent) > $options[File::MAX_SIZE]
            );

            if (!$diff[FILE::FETCH_CONTENT]) {
                $toFileFullContentLines   = preg_split(
                    "/(\r\n|\n|\r)/",
                    $toFileFullContent,
                    null,
                    PREG_SPLIT_DELIM_CAPTURE
                );
                $fromFileFullContentLines = preg_split(
                    "/(\r\n|\n|\r)/",
                    $fromFileFullContent,
                    null,
                    PREG_SPLIT_DELIM_CAPTURE
                );
                // We need to divide by 2 to get the actual count of lines because above function
                // getFullDepotContents returns lines with extra new lines
                $diff[FILE::LINES_COUNT] =
                    max(
                        floor(count($toFileFullContentLines) / 2),
                        floor(count($fromFileFullContentLines) / 2)
                    );
            }
        }
        return $diff;
    }

    /**
     * Get file contents of added/deleted files.
     *
     * @param array $diff diff result array we are building.
     * @param File|null $right optional - right-hand file.
     * @param File|null $left optional - left-hand file.
     * @param array|null $options influences diff behavior.
     * @return  array   diff result with lines added.
     * @throws FileException
     */
    protected function diffAddDelete(
        array $diff,
        ?File $right = null,
        ?File $left = null,
        ?array $options = null
    ): array {
        // contents must come from the side we have, or the side that is not deleted/purged
        // contents from right imply add, contents from left imply delete
        $file         = $right && !$right->isDeletedOrPurged() ? $right : $left;
        $isAdd        = $file === $right;
        $contentChunk = false;
        $cropped      = false;

        $content = $this->getDepotContents($file, $options, $cropped, $contentChunk);
        $name    = $file->getDepotFilenameWithRevision();

        $diff[self::CONTENT_CHUNK]            = $contentChunk ? $options[self::MAX_CHUNK_SIZE] : false;
        $diff[self::CONTENT_CHUNK_START]      = $contentChunk ? $options[self::CONTENT_CHUNK_START] : false;
        $diff[self::CONTENT_CHUNK_END]        = $contentChunk ? $options[self::CONTENT_CHUNK_END] : false;
        $diff[self::CONTENT_CHUNK_START_LINE] = $contentChunk ? $options[self::CONTENT_CHUNK_START_LINE] : false;
        $diff[self::CONTENT_CHUNK_END_LINE]   = $contentChunk ? $options[self::CONTENT_CHUNK_END_LINE] : false;
        $diff[self::WHOLE_DIFF_CONTENT_SIZE]  = $contentChunk ? $options[self::WHOLE_DIFF_CONTENT_SIZE] : false;
        $diff[self::CURRENT_DIFF_CHUNK_SIZE]  = $contentChunk ? $options[self::CURRENT_DIFF_CHUNK_SIZE] : false;
        $diff[Diff::CUT]                      = $contentChunk ? $options[self::MAX_CHUNK_SIZE] : false;
        return $this->diffContentGenerator($diff, $name, $isAdd, $content, $options);
    }

    /**
     * @param   array       $diff         - an array containing existing diff data
     * @param   string      $name         - name of the file/spec
     * @param   bool        $isAdd        - It's add or delete operation
     * @param   string      $content      - Content of file or spec
     * @param   array       $options      - influences diff behavior.
     * @return  array return the diff content for add/delete for file/stream spec
     */
    protected function diffContentGenerator($diff, $name, $isAdd, $content, $options): array
    {
        $diff[Diff::LINES][] = [
            Diff::VALUE      => null,
            Diff::TYPE       => Diff::META,
            Diff::LEFT_LINE  => null,
            Diff::RIGHT_LINE => null
        ];
        // Set loop vars
        $diffMeta = &$diff[Diff::LINES][count($diff[Diff::LINES]) - 1];
        $sign     = $isAdd ? '+' : '-';
        $type     = $isAdd ? Diff::ADD : Diff::DELETE;
        $lines    = preg_split("/(\r\n|\n|\r)/", $content, null, PREG_SPLIT_DELIM_CAPTURE);
        $rawLines = [];
        if (!$diff[self::CONTENT_CHUNK_START_LINE] && $diff[self::CONTENT_CHUNK_END_LINE]) {
            // fetching previous chunk from pressed button of show previous
            $count     = ($diff[self::CONTENT_CHUNK_END_LINE]>=1) ? $diff[self::CONTENT_CHUNK_END_LINE]
                : $diff[self::CONTENT_CHUNK_END_LINE] + 1;
            $lineCount = 0;
            end($lines);
            $lastIndex = key($lines);
            if (isset($lines[$lastIndex]) && strlen(trim($lines[$lastIndex])) === 0
                && isset($lines[$lastIndex-1]) && strlen(trim($lines[$lastIndex-1])) === 0) {
                unset($lines[$lastIndex]);
            }
            for ($i = 0,$j=$diff[self::CONTENT_CHUNK_END_LINE]; $i < count($lines); $i += 2, $j--) {
                $line         = $lines[$i];
                $value        = $sign . $line;
                $rawLines[$j] = $value;
                $count--;
                $lineCount++;
            }
            $diff[self::CONTENT_CHUNK_START_LINE] = $j+1;

            $postInitPosition = $count+1;
            $initialPosition  = $count;
            $lastPosition     = ($diff[self::CONTENT_CHUNK_END_LINE] == 1) ? $diff[self::CONTENT_CHUNK_END_LINE]
                : $diff[self::CONTENT_CHUNK_END_LINE] + 1;
            $meta             = '@@ ' . ($isAdd ?  '-1,' . $initialPosition . ' +' . $postInitPosition
                    . ','. $lastPosition
                    : '-' . $diff[self::CONTENT_CHUNK_START_LINE] . ',' . $count . ' +0,1' ) . ' @@';
        } else {
            // fetching the next chunk
            $diff[self::CONTENT_CHUNK_START_LINE] = $diff[self::CONTENT_CHUNK_END_LINE] ?: 1;

            $count            = ($diff[self::CONTENT_CHUNK_START_LINE]>1)
                ? $diff[self::CONTENT_CHUNK_START_LINE] : $diff[self::CONTENT_CHUNK_START_LINE] - 1;
            $initialPosition  = $count;
            $postInitPosition = ($count===0) ? $count+1 : $count;
            $lineCount        = 0;
            end($lines);
            $lastIndex = key($lines);
            if (isset($lines[$lastIndex]) && strlen(trim($lines[$lastIndex])) === 0
                && isset($lines[$lastIndex-1]) && strlen(trim($lines[$lastIndex-1])) === 0) {
                unset($lines[$lastIndex]);
            }
            for ($i = 0,$j=$diff[self::CONTENT_CHUNK_START_LINE]; $i < count($lines); $i += 2, $j++) {
                $line         = $lines[$i];
                $value        = $sign . $line;
                $rawLines[$j] = $value;
                $count++;
                $lineCount++;
            }
            $diff[self::CONTENT_CHUNK_END_LINE] = $j-1;

            $meta = '@@ ' . ($isAdd ?  '-1,' . $initialPosition . ' +' . $postInitPosition . ','. $count
                    : '-' . $diff[self::CONTENT_CHUNK_START_LINE] . ','
                    . $diff[self::CONTENT_CHUNK_END_LINE] . ' +0,1' ) . ' @@';
        }
        $header        = '--- a/' . ($isAdd ? "dev/null" : $name) . "\n" . '+++ b/' . ($isAdd ? $name : 'dev/null');
        $rawLineString = $meta . "\n" . implode("\n", $rawLines);
        $summary       = [];
        if ($options[Diff::SUMMARY]) {
            $summary = [
                Diff::SUMMARY_ADDS    => $isAdd ? $lineCount : 0,
                Diff::SUMMARY_DELETES => $isAdd ? 0 : $lineCount,
                Diff::SUMMARY_UPDATES => 0
            ];
        }
        $diff[Diff::HEADER]  = $header;
        $diff[Diff::LINES]   = [$rawLineString];
        $diff[Diff::SUMMARY] = $summary;
        return $diff;
    }

    /**
     * Get the contents of the file in Perforce.
     *
     * File content is fetched once and then cached in the instance
     * (unless the file is truncated due to max-filesize).
     *
     * The cache can be primed via setContentCache().
     * It can be cleared via clearContentCache().
     *
     * @param null|array $options updated by reference to influence behaviour
     *                          MAX_FILESIZE - crop the file after this many bytes - won't split
     *                                         multibyte chars if UTF8_SANITIZE is set
     *                          UTF8_CONVERT - attempt to covert non UTF-8 to UTF-8
     *                         UTF8_SANITIZE - replace invalid UTF-8 sequences with �
     * @param bool $cropped updated by reference, indicates the file contents
     *                                    exceeded max-filesize and were truncated.
     * @param bool $contentChunk updated by reference, indicates the file contents
     *                                    exceeded max_chunk_size and were truncated.
     * @param object $file file object
     * @return  string      the contents of the file in the depot.
     * @throws  FileException   if the print command fails.
     * @throws Exception
     */
    public function getDepotContents(
        object $file,
        array  &$options = null,
        bool   &$cropped = false,
        bool   &$contentChunk = false
    ): string {
        $cropped      = false;
        $options      = (array) $options + [
                File::UTF8_CONVERT  => false,
                File::UTF8_SANITIZE => false,
            ];
        $maxSizeChunk = $options[self::MAX_CHUNK_SIZE];

        $convert  = $options[File::UTF8_CONVERT];
        $sanitize = $options[File::UTF8_SANITIZE];
        // if cache is empty, get content from the server
        if (!$file->getContentCache()) {
            // verify we have a filespec set; throws if invalid/missing
            $filespec = $file->getFilespec();
            if (empty($filespec)) {
                throw new FileException("Cannot complete operation, no filespec has been specified");
            }
            if (!is_string($filespec) ||
                !strlen($filespec) ||
                strpos($filespec, "*")   !== false ||
                strpos($filespec, "...") !== false) {
                throw new FileException(
                    "Invalid filespec provided. In this context, "
                    . "filespecs must be a reference to a single file."
                );
            }
            // setup output handler to support limiting file content length
            // this is necessary to avoid running out of memory.
            $content = "";
            $handler = new Limit;
            $handler->setOutputCallback(
                function ($data, $type) use (&$content) {
                    if ($type !== 'text' && $type !== 'binary') {
                        return Limit::HANDLER_REPORT;
                    }
                    if (is_array($data)) {
                        return Limit::HANDLER_HANDLED;
                    }

                    $content .= $data;
                    return Limit::HANDLER_HANDLED;
                }
            );
            // run the print command with our output handler
            // ensure depot syntax to avoid multiple file output if overlay mappings in use.
            $result = $file->getConnection()->runHandler($handler, 'print', $file->getDepotFilenameWithRevision());
            // check for warnings.
            if ($result->hasWarnings()) {
                throw new Exception(
                    "Failed to get depot contents: " . implode(", ", $result->getWarnings())
                );
            }

            $file->setContentCache($content);
        } else {
            $content = $file->getContentCache();
        }

        if (strlen($content) > $maxSizeChunk
            || (isset($options[self::CONTENT_CHUNK_START])
            && $options[self::CONTENT_CHUNK_START] > 0)) {
            $startIndex = ($options[self::CONTENT_CHUNK_START] > 0 ) ? $options[self::CONTENT_CHUNK_START] : 0;
            $startIndex = ($options[self::CONTENT_CHUNK_START] > strlen($content) )
                ? ($options[self::CONTENT_CHUNK_START] - $maxSizeChunk/2): $startIndex;
            $chunkEnd   = ($options[self::CONTENT_CHUNK_END] > strlen($content))
                ? strlen($content) : $options[self::CONTENT_CHUNK_END];

            $options[self::WHOLE_DIFF_CONTENT_SIZE] = strlen($content);
            $content                                = substr(
                $content,
                $startIndex,
                $maxSizeChunk
            );
            $contentChunk                           = true;
            $options[self::CURRENT_DIFF_CHUNK_SIZE] = strlen($content);
            $options[self::CONTENT_CHUNK_START]     = $startIndex;
            $options[self::CONTENT_CHUNK_END]       = $chunkEnd;
        }
        $nonUtf8Encodings = $options[Utf8::NON_UTF8_ENCODINGS] ?? Utf8::$fallbackEncodings;
        // if we are requested to convert or replace; return filtered
        if ($convert || $sanitize) {
            $filter  = new Utf8;
            $content = $filter->setConvertEncoding($convert)
                ->setReplaceInvalid($sanitize)
                ->setNonUtf8Encodings($nonUtf8Encodings)
                ->filter($content);

            // if we cropped the file and the caller requested sanitized output,
            // check if the last character is '�' and remove it (likely our fault)
            if ($cropped && $sanitize && substr($content, -3) === "\xEF\xBF\xBD") {
                $content = substr($content, 0, -3);
            }
        }        // if no options; just return cached directly
        return $content;
    }

    /**
     * Processes a filePath and some params to make a call to the Diff library's diff method.
     * Additionally, does some post processing of the results, according to the params
     * @param string   $filePath   depot path of file/stream
     * @param array    $params     diff options
     *
     * @return array               Array with four or five elements
     *                             - lines   - array of unified diff hunks strings
     *                             - isCut   - boolean indicating if diff results were truncated
     *                             - isSame  - boolean indicating if from & to are identical
     *                             - header  - unified diff header string in git format
     *                             - summary - array with number of adds, deletes & updates
     *
     * @throws FileException
     * @throws FileNotFoundException
     * @throws ForbiddenException
     * @throws SpecNotFoundException
     * @throws P4Exception
     * @throws Exception
     */
    public function getFileContent($filePath, $params)
    {
        // Get the files with their correct revisions
        $p4     = $this->services->get(ConnectionFactory::P4);
        $differ = new Diff($p4);
        // Set the diff options and run the diff
        $options = $this->setDiffOptions($params);
        if ($params[IDiff::TYPE] == self::STREAM) {
            $this->checkPermission($filePath, Protections::MODE_READ, self::STREAM);
            $streamRevisions = $this->getStreamRevisions($filePath, $params, $p4);
            $fromPath        = $streamRevisions[IDiff::FROM];
            $toPath          = $streamRevisions[IDiff::TO];
            $fileContent     = $differ->diffStream($toPath, $fromPath, $options);
        } else {
            $fileRevisions = $this->getFileRevisions($filePath, $params, $p4);
            $fromFile      = $fileRevisions[IDiff::FROM];
            $toFile        = $fileRevisions[IDiff::TO];
            // Check the file permissions
            // Since we are diffing revisions of the same file, we just need to check the common depot path
            $path = $fromFile ? $fromFile->getDepotFilename() : ($toFile ? $toFile->getDepotFilename() : '');
            $this->checkPermission($path, Protections::MODE_READ);
            $fileContent = $toFile->getDepotContents();
        }
        return $fileContent;
    }

    /**
     * Builds from and to revisions of the given file path to diff against
     * @param string       $filePath   depot path of file
     * @param array        $params     diff options
     * @param Connection   $p4         p4 connection
     *
     * @return array
     * @throws FileException
     * @throws FileNotFoundException
     */
    protected function getFileRevisions($filePath, $params, $p4)
    {
        $leftFile  = null;
        $rightFile = null;
        $from      = isset($params[IDiff::FROM]) ? $params[IDiff::FROM] : null;
        $to        = $params[IDiff::TO];

        // Get the left file path with full revision
        if (isset($params[IDiff::FROM_FILE])) {
            // If a fromFile is specified, it will be base64 encoded, so we must decode it here
            // Additionally, if a fromFile is specified, we assume it exists. So, if there's not from,
            // we assume it's the head revision
            $leftFilePath = StringHelper::base64DecodeUrl($params[IDiff::FROM_FILE]);
            $left         = $leftFilePath . ($from ?? Diff::REVISION_HEAD);
        } else {
            $left = $from ? $filePath . $from : null;
        }

        // Get the left File object
        try {
            $leftFile = $left ? File::fetch($left, $p4) : null;
        } catch (FileException $e) {
            // Allow 404 when head is requested, as this may or may not be a new file
            if (strpos($left, Diff::REVISION_HEAD) === false) {
                throw $e;
            }
        }

        // Get the right File object
        $right     = $filePath . $to;
        $rightFile = $right ? File::fetch($right, $p4) : null;
        return [IDiff::FROM => $leftFile, IDiff::TO => $rightFile];
    }

    /**
     * Gets a connection
     * @param ConnectionInterface|null $connection the connection
     * @return ConnectionInterface if the connection is not passed then new connection is returned
     */
    protected function getConnection(ConnectionInterface $connection = null)
    {
        if (!$connection) {
            return $this->services->get(ConnectionFactory::P4);
        }

        return $connection;
    }

    /**
     * Gets the diff options from the params, falling back to the config and in some cases local defaults
     * @param  array  $params   Passed parameters from query string to diff api
     * @return array
     * @throws Exception
     */
    protected function setDiffOptions($params): array
    {
        $config       = $this->services->get(IConfigDefinition::CONFIG);
        $uft8_convert = ConfigManager::getValue(
            $config,
            IConfigDefinition::TRANSLATOR_UTF8_CONVERT,
            false
        );

        return [
            Diff::RAW_DIFF           => true,
            Diff::SUMMARY            => true,
            IDiff::LINES             => $this->getContextLines($params, $config),
            IDiff::MAX_SIZE          => $this->getMaxSize($params, $config),
            IDiff::OFFSET            => isset($params[IDiff::OFFSET]) ? (int)$params[IDiff::OFFSET] : 0,
            Diff::UTF8_CONVERT       => $params[Diff::UTF8_CONVERT] ?? $uft8_convert,
            Utf8::NON_UTF8_ENCODINGS => $this->getNonUtf8Encodings($config),
            Diff::SUMMARY_LINES      => $params[Diff::SUMMARY_LINES] ?? false,
            Diff::UTF8_SANITIZE      => true,
            Diff::IGNORE_WS          => $params[Diff::IGNORE_WS] ?? null,
            self::CONTENT_CHUNK_START => $params[self::CONTENT_CHUNK_START] ?? 0,
            self::CONTENT_CHUNK_END => $params[self::CONTENT_CHUNK_END] ?? self::MAX_CHUNK_SIZE_VALUE,
            self::CONTENT_CHUNK_START_LINE => $params[self::CONTENT_CHUNK_START_LINE] ?? false,
            self::CONTENT_CHUNK_END_LINE => $params[self::CONTENT_CHUNK_END_LINE] ?? false,
            self::MAX_CHUNK_SIZE => self::MAX_CHUNK_SIZE_VALUE
        ];
    }

    /**
     * Formats the diff into a form that is expected by the API
     * @param array    $diff   diff array reference
     */
    protected function formatDiff(array &$diff)
    {
        $diff[IFile::DIFFS] = $diff[Diff::LINES];
        unset($diff[Diff::LINES]);

        if (isset($diff[Diff::HEADER])) {
            $diff[Diff::HEADER] = $diff[Diff::HEADER] . "\n";
        }
    }

    /**
     * Truncates the number of diff sections (hunks) that we will return and populates the paging keys
     * @param array    $diff      diff array reference
     * @param array    $options   diff options
     */
    protected function handlePaging(array &$diff, array $options)
    {
        $maxDiffs      = null;
        $offset        = $options[IDiff::OFFSET];
        $totalDiffs    = count($diff[IFile::DIFFS]);
        $diffsReturned = $totalDiffs;
        $nextOffset    = null;

        if (!is_null($maxDiffs) || $offset > 0) {
            if ($offset <= $totalDiffs) {
                $diff[IFile::DIFFS] = array_slice($diff[IFile::DIFFS], $offset, $maxDiffs);
                $diffsReturned      = count($diff[IFile::DIFFS]);
                $nextOffset         = $offset + $diffsReturned;
                $nextOffset         = $nextOffset < $totalDiffs ? $nextOffset : null;
            } else {
                $diff[IFile::DIFFS] = [];
                $diffsReturned      = 0;
            }
        }

        $diff[IFile::PAGING][IFile::DIFFS]  = $diffsReturned;
        $diff[IFile::PAGING][IFile::OFFSET] = $nextOffset;
    }

    /**
     * Get the context lines value. NULL or negative value return the default value
     * from config else return the passed value
     * @param  array   $params    Passed parameters from query string to diff api
     * @param  array   $config    Swarm config
     * @return int
     * @throws Exception
     */
    protected function getContextLines($params, $config)
    {
        if (isset($params[IDiff::LINES]) && $params[IDiff::LINES] >= 0) {
            $lines = $params[IDiff::LINES];
        } else {
            $lines = ConfigManager::getValue($config, ConfigManager::DIFF_CONTEXT_LINES, self::DEFAULT_CONTEXT_LINES);
        }
        return $lines;
    }

    /**
     * Get the maxSize value
     * 1. NULL or Zero value return the value from config.
     * 2. -1 value points to no limit so return 'unlimited'
     * 3. Other than NULL or Zero or -1 return the passed value
     * @param  array   $params    Passed parameters from query string to diff api
     * @param  array   $config    Swarm config
     * @return mixed
     * @throws Exception
     */
    protected function getMaxSize($params, $config)
    {
        $param = $params[IDiff::MAX_SIZE] ?? null;

        if (!$param || $param == 0 || $param < -1) {
            $fileSize = ConfigManager::getValue(
                $config,
                ConfigManager::FILES_MAX_SIZE,
                self::DEFAULT_MAX_SIZE
            );
        } elseif ($param == -1) {
            $fileSize = 'unlimited';
        } else {
            $fileSize = $param;
        }

        return $fileSize;
    }

    /**
     * Get the TRANSLATOR_NON_UTF8_ENCODINGS from the config, otherwise default to the Utf8::$fallbackEncodings
     * @param array         $config   swarm config
     * @return mixed
     * @throws Exception
     */
    protected function getNonUtf8Encodings(array $config)
    {
        return ConfigManager::getValue($config, ConfigManager::TRANSLATOR_NON_UTF8_ENCODINGS, Utf8::$fallbackEncodings);
    }

    /**
     * Fetch file. Overrides the parent to check protections (also File does not support fetchById)
     * @param string                    $fileSpec       the file spec
     * @param ConnectionInterface|null  $connection     the current connection
     * @return mixed|File file model
     * @throws ForbiddenException
     * @throws FileNotFoundException
     */
    public function fetch($fileSpec, ConnectionInterface $connection = null)
    {
        $this->checkPermission($fileSpec, Protections::MODE_READ);
        return File::fetch($fileSpec, $this->getConnection($connection));
    }

    /**
     * Checks whether stream is exists or not and return
     * the revisions path
     *
     * @param string       $streamPath stream spec path
     * @param array        $params     diff options
     * @param Connection   $p4         p4 connection
     * @return array
     * @throws SpecNotFoundException
     */
    protected function getStreamRevisions($streamPath, $params, $p4)
    {
        Stream::fetchById($streamPath,  $p4);
        $from  = $params[IDiff::FROM];
        $to    = $params[IDiff::TO];
        $left  = $from ? $streamPath . $from : null;
        $right = $streamPath . $to;

        return[IDiff::FROM => $left, IDiff::TO => $right];
    }

    /**
     * Run p4 diff2 against left/right files and parse output into array.
     * Note: this asks for normal output rather than z tag
     * This will return completely different output depending on whether the RAW_DIFF option is specified or not.
     * If RAW_DIFF is specified, it will skip the processDiffEdit step and return an array in the following form:
     *     [
     *         static::RAW_DIFF => [
     *             <header line>,
     *             <chunk1>,
     *             <chunk2>,
     *             ...
     *         ],
     *         static::SUMMARY => [
     *             static::ADDS    => <num adds>,
     *             static::DELETES => <num deletes>,
     *             static::UPDATES => <num updates>,
     *         ]
     *     ]
     * @param object $differ     Diff library object
     * @param   array   $diff       diff result array we are building.
     * @param   File    $right      right-hand file.
     * @param   File    $left       left-hand file.
     * @param   array   $options    influences diff behavior.
     * @return  array   diff result with lines added (if raw diff is not specified).
     * @throws  P4Exception
     */
    protected function diffEdit(object $differ, array $diff, File $right, File $left, array $options): array
    {
        $whitespace = $this->getWhitespaceFlag($options[Diff::IGNORE_WS]);
        $leftSpec   = $left->getFilespec();
        $rightSpec  = $right->getFilespec();
        // Set the flags for the diff2 command
        $unifiedFlags = array_merge(
            $whitespace,
            [
                Diff::FORCE_BINARY_DIFF,
                Diff::UNIFIED_MODE . $options[Diff::LINES],
                $leftSpec,
                $rightSpec
            ]
        );
        $diffEdit     = $differ->getConnection()->run('diff2', $unifiedFlags, null, false)->getData();
        $diff         = $this->processDiffEdit($diff, $diffEdit, $options);
        // Get header & summary info
        // Get the unified header
        $header = sprintf(
            "--- a/%s\n+++ b/%s",
            $left->getDepotFilenameWithRevision(),
            $right->getDepotFilenameWithRevision()
        );
        // For now, we only concern ourselves with a summary if a raw diff is specified in the options
        $summaryFlags        = array_merge(
            $whitespace,
            [Diff::FORCE_BINARY_DIFF, Diff::SUMMARY_MODE, $leftSpec, $rightSpec]
        );
        $rawSummary          = $differ->getConnection()->run('diff2', $summaryFlags, null, false)->getData();
        $summary             = $this->processSummary($rawSummary, $options);
        $diff[Diff::HEADER]  = $header;
        $diff[Diff::SUMMARY] = $summary;
        return $diff;
    }

    /**
     * Break the response from a diff2 into lines, in preparation for being rendered into a diff pane.
     * It is expected that the format of the output is:
     *     ==== //cherwell/main@12123 - //cherwell/main@12127 ==== content
     *     @@ -34,11 +34,12 @@\n Parent:    none\n
     *     ...
     * The first line, file name, is discarded; the rest are concatenated, have utf8 conversions applied and then
     * split into an array of lines with metadata that can be used in the rendering process.
     * @param $diff         - an array containing existing diff data
     * @param $diff2Output  - the raw output of a diff2 command
     * @param $options      - the original options for the diff process, includes the utf8 settings for this function
     * @return array        - the original $diff array with the lines attribute set
     */
    public function processDiffEdit($diff, $diff2Output, $options): array
    {
        // diff output puts a file header in the first data block
        // (which we skip) and the diffs in one or more following blocks.
        $diffs = implode("\n", array_slice($diff2Output, 1));
        // if we are requested to convert or replace; do so prior to split
        if ((isset($options[Diff::UTF8_CONVERT]) && $options[Diff::UTF8_CONVERT])) {
            $filter = (new Utf8)
                ->setConvertEncoding($options[Diff::UTF8_CONVERT])
                ->setReplaceInvalid($options[Diff::UTF8_SANITIZE])
                ->setNonUtf8Encodings($options[Utf8::NON_UTF8_ENCODINGS] ?? Utf8::$fallbackEncodings);

            $diffs = $filter->filter($diffs);
        }
        // parse diff block into lines
        // capture line-ending, so we can detect line-end changes.
        $types    = array('@' => Diff::META, ' ' => Diff::NO_DIFF, '-' => Diff::DELETE, '+' => Diff::ADD);
        $lines    = preg_split("/(\r\n|\n|\r)/", $diffs, null, PREG_SPLIT_DELIM_CAPTURE);
        $rawLines = [];
        $numLines = count($lines);
        for ($i = 0; $i < $numLines; $i += 2) {
            $line = $lines[$i];
            $end  = $lines[$i+1] ?? '';
            // skip empty
            if (!strlen($line) || $line === '\ No newline at end of file') {
                continue;
            }
            // set type correctly when unexpected output
            $type = $types[$line[0]] ?? Diff::NO_DIFF;
            // extract starting left/right line numbers from meta block, has the format of "@@ -133,29 +133,27 @@"
            if ($type === Diff::META && !empty($rawLines)) {
                $diff     = $this->prepareChunkData($diff, $rawLines, $options[static::CURRENT_DIFF_CHUNK_ROWS] ?? 0);
                $rawLines = [];
            }
            $rawLines[] = $line . $end;
        }
        // Need to reset raw line if there are multiple diffs in a file
        if (!empty($rawLines)) {
            $diff     = $this->prepareChunkData($diff, $rawLines);
            $rawLines = [];
        }
        return $diff;
    }

    /**
     * prepare the chunk data & return it
     * @param $diff  - an array containing existing diff data
     * @param $rawLines - update by reference an array of diff data
     * @param int $currentChunkRows number of rows to be shown on UI
     * @return array
     */
    public function prepareChunkData($diff, &$rawLines, int $currentChunkRows = 0): array
    {

        $metaDataOfDiff               = explode(' ', $rawLines[0]);
        $metaDataForLeftSide          = explode(',', $metaDataOfDiff[1]);
        $metaDataForRightSide         = explode(',', $metaDataOfDiff[2]);
        $originalMetaDataOfDiff       = $metaDataOfDiff;
        $originalMetaDataForLeftSide  = $metaDataForLeftSide;
        $originalMetaDataForRightSide = $metaDataForRightSide;
        $maxDiffRows                  = self::MAX_ROWS_IN_EDIT_TYPE; // 1000 is its value
        // getting the number of lines from resulted diff & checking if number of lines are more than
        // the specified or configured rows
        if ((int) $metaDataForLeftSide[1] > $maxDiffRows || (int) $metaDataForRightSide[1] > $maxDiffRows) {
            $diff[self::CONFIG_DIFF_CHUNK_ROW] = $maxDiffRows;
            $diff[Diff::CUT]                   = $maxDiffRows;
            $originalRawLines                  = $rawLines;
            $largerChunkSide                   = ((int) $metaDataForLeftSide[1] > (int) $metaDataForRightSide[1])
                ? self::LEFT : self::RIGHT;
            $sliceLength                       = $currentChunkRows + $maxDiffRows;
            [$insertedLines, $deletedLines]    = $this->countInsertedOrDeletedLinesFromChunk(
                $originalRawLines,
                $maxDiffRows
            );
            // store the original metadata before overwriting it with chunk metadata
            if (self::LEFT === $largerChunkSide) {
                $sliceLength                   += $insertedLines;
                $originalMetaDataForLeftSide[1] = $sliceLength;
                $originalMetaDataOfDiff[1]      = implode(',', $originalMetaDataForLeftSide);
            } else {
                $sliceLength                    += $deletedLines;
                $originalMetaDataForRightSide[1] = $sliceLength;
                $originalMetaDataOfDiff[2]       = implode(',', $originalMetaDataForRightSide);
            }

            $originalRawLines[0] = implode(' ', $originalMetaDataOfDiff);
            $chunkedLines        = array_slice($originalRawLines, 0, $sliceLength + 1);
            $diff[Diff::LINES][] = implode('', $chunkedLines);
            $chunkAddedIndex     = array_key_last($diff[Diff::LINES]);

            $leftStart = (int) str_replace('-', '', $metaDataForLeftSide[0]);
            $leftCount = (int) $metaDataForLeftSide[1];
            $leftEnd   = $leftStart + $leftCount - 1;

            $rightStart = (int) str_replace('-', '', $metaDataForRightSide[0]);
            $rightCount = (int) $metaDataForRightSide[1];
            $rightEnd   = $rightStart + $rightCount - 1;

            $diff[static::ORIGINAL_LINES_DATA][$chunkAddedIndex] = [
                static::ORIGINAL_LINES                  => $rawLines,
                static::CONTENT_CHUNK_LEFT_START        => $metaDataForLeftSide[0],
                static::CONTENT_CHUNK_LEFT_START_LINE   => $leftStart,
                static::CONTENT_CHUNK_LEFT_END_LINE     => $leftEnd,
                static::CONTENT_CHUNK_LEFT_END          => $leftEnd,
                static::CONTENT_CHUNK_LEFT_TOTAL_LINES  => $leftCount,
                static::LARGER_CHUNK                    => $largerChunkSide,
                static::CONTENT_CHUNK_RIGHT_START       => $metaDataForRightSide[0],
                static::CONTENT_CHUNK_RIGHT_START_LINE  => $rightStart,
                static::CONTENT_CHUNK_RIGHT_END_LINE    => $rightEnd,
                static::CONTENT_CHUNK_RIGHT_END         => $rightEnd,
                static::CONTENT_CHUNK_RIGHT_TOTAL_LINES => $rightCount,
                static::CURRENT_DIFF_CHUNK_ROWS         => count($chunkedLines) - 1,
            ];
            $rawLines                                            = [];
        } else {
            $diff[Diff::LINES][] = implode('', $rawLines);
            $rawLines            = [];
        }
        return $diff;
    }

    /**
     * Function used to count number of inserted or deleted lines in hunk
     * @param array $originalRawLines
     * @return array
     */
    public function countInsertedOrDeletedLinesFromChunk(array $originalRawLines, int $chunkLimit): array
    {
        $insertedLines = 0;
        $deletedLines  = 0;
        for ($i=1; $i<=$chunkLimit; $i++) {
            $rawLine = $originalRawLines[$i];
            if (str_starts_with($rawLine, '+')) {
                $insertedLines +=1;
            }
            if (str_starts_with($rawLine, '-')) {
                $deletedLines +=1;
            }
        }
        return [$insertedLines, $deletedLines];
    }

    /**
     * Converts diff2 summary output into a simple array
     * @param array $rawSummary     diff2 summary output
     * @param array $options        current options. Used in this function to determine if summary counts relate to
     *                              chunks or diffs. Chunks is default unless $options[self::SUMMARY_LINES] is true
     * @return array
     */
    protected function processSummary($rawSummary, $options)
    {
        // Summary can be based on lines or chunks. Default is chunks unless
        // self::SUMMARY_LINES is set to true
        $options += [
            Diff::SUMMARY_LINES => false
        ];
        $chunks   = !$options[Diff::SUMMARY_LINES];
        $summary  = [
            Diff::SUMMARY_ADDS    => 0,
            Diff::SUMMARY_DELETES => 0,
            Diff::SUMMARY_UPDATES => 0
        ];
        if (count($rawSummary) <= 1) {
            return $summary;
        }

        $pattern = "#add (\d+) chunks (\d+).+\ndeleted (\d+) chunks (\d+).+\nchanged (\d+) chunks (\d+) / (\d+)#";
        if (preg_match($pattern, $rawSummary[1], $matches)) {
            $summary = [
                Diff::SUMMARY_ADDS    => (int)$matches[$chunks ? 1 : 2],
                Diff::SUMMARY_DELETES => (int)$matches[$chunks ? 3 : 4],
                Diff::SUMMARY_UPDATES => ($chunks ? (int)$matches[5] : max((int)$matches[6], (int)$matches[7]))
            ];
        }
        return $summary;
    }

    /**
     * Maps an integer value to a flag that can be used by the diff2 command for ignoring whitespace or line endings.
     *    0               => [] no ignore flags
     *    1               => ['-dw'] ignore all whitespace
     *    2               => ['-db'] ignore whitespace changes
     *    3               => ['-dl'] ignore line endings
     *    any other value => no ignore flags
     *
     * @param mixed   $ignoreWs either unset, false or an int between 0 and 3, inclusive
     * @return array
     */
    protected function getWhitespaceFlag($ignoreWs)
    {
        switch ($ignoreWs) {
            case 1:
                $flag = [Diff::IGNORE_ALL_WHITESPACE];
                break;
            case 2:
                $flag = [Diff::IGNORE_WHITESPACE];
                break;
            case 3:
                $flag = [Diff::IGNORE_LINE_ENDINGS];
                break;
            default:
                $flag = [];
        }
        return $flag;
    }
}
