<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */
namespace Comments\Filter;

use Api\IRequest;
use Application\Connection\ConnectionFactory;
use Application\Factory\InvokableService;
use Application\InputFilter\DirectInput;
use Application\InputFilter\InputFilter;
use Application\Validator\IsArray;
use Comments\Model\IComment as ModelInterface;
use Comments\Validator\Attachments;
use Comments\Validator\Body;
use Comments\Validator\TaskState;
use Interop\Container\ContainerInterface;
use Laminas\Filter\StringTrim;
use Laminas\Validator\StringLength;

/**
 * Class Comment
 * @package Comments\Filter
 */
abstract class Comment extends InputFilter implements InvokableService
{
    protected $translator;
    protected $services;
    protected $apiVersion;

    /**
     * Validate body value
     */
    protected function addBodyValidator($mandatory = false)
    {
        $input = new DirectInput(ModelInterface::BODY);
        $input->setRequired($mandatory);
        $input->getFilterChain()->attach(new StringTrim());
        if ($this->apiVersion >= 'v11') {
            // V11 allows an empty body, if there are attachments
            $input->getValidatorChain()->attach(new Body($this->services));
        } else {
            $input->getValidatorChain()->attach(new StringLength(['min'=>1]));
        }
        $this->add($input);
    }

    /**
     * Validate task state value
     */
    protected function addTaskStateValidator()
    {
        $input = new DirectInput(ModelInterface::TASK_STATE);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new TaskState($this->translator, [TaskState::SUPPORT_ARRAYS=>false]));
        $this->add($input);
    }

    /**
     * Validate attachments by testing that the value is an array and that all the items correspond to valid attachment
     * ids.
     * @return void
     */
    protected function addAttachmentsValidator()
    {
        $p4Admin = $this->services->get(ConnectionFactory::P4_ADMIN);
        $input   = new DirectInput(ModelInterface::ATTACHMENTS);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new IsArray())->attach(new Attachments(['connection' => $p4Admin]));
        $this->add($input);
    }
}
