<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */
namespace Attachments\Filter;

use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\I18n\TranslatorFactory;
use Application\InputFilter\DirectInput;
use Application\Validator\BetweenInt;
use Attachments\Model\Attachment;
use Attachments\Model\IAttachmentDAO;
use Interop\Container\ContainerInterface;
use Laminas\Filter\StringTrim;
use Laminas\InputFilter\InputFilter;
use Laminas\Validator\Between;
use Laminas\Validator\StringLength;
use Application\Config\ConfigException;

class File extends InputFilter implements IFile
{
    private $maxSize;
    private $services;

    /**
     * Construct the filter, this should be done via services->get
     * @param ContainerInterface $services
     * @param array|null $options
     * @throws ConfigException
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services = $services;
        foreach ([IAttachmentDAO::TMP_NAME, Attachment::FIELD_NAME, Attachment::FIELD_TYPE] as $field) {
            $this->addRequiredString($field);
        }
        $config        = $services->get(IConfigDefinition::CONFIG);
        $this->maxSize = ConfigManager::getValue($config, IConfigDefinition::ATTACHMENTS_MAX_FILE_SIZE);
        $this->addSize();
    }

    /**
     * Add required string validation
     * @param string $field     required field name
     * @return void
     */
    private function addRequiredString(string $field)
    {
        $input = new DirectInput($field);
        $input->setRequired(true);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()->attach(new StringLength(['min' => 1]));
        $this->add($input);
    }

    /**
     * Add size validation. Must be between 0 and maxSize, inclusively
     * @return void
     */
    private function addSize()
    {
        $input = new DirectInput(Attachment::FIELD_SIZE);
        $input->setRequired(true);
        $input->getValidatorChain()->attach(new BetweenInt(['min' => 0, 'max' => $this->maxSize]));
        $this->add($input);
    }

    /**
     * Override messages to provide more information for the cases of failed temporary name and file size
     * @return array
     */
    public function getMessages() : array
    {
        $translator = $this->services->get(TranslatorFactory::SERVICE);
        $messages   = parent::getMessages();
        if (isset($messages[IAttachmentDAO::TMP_NAME])) {
            $messages[IAttachmentDAO::TMP_NAME] =
                [self::INVALID => $translator->t("Cannot add attachment. File did not upload correctly.")];
        }
        if (isset($messages[Attachment::FIELD_SIZE][Between::NOT_BETWEEN])) {
            $messageSuffix = $translator->t(
                "'%d' is the maximum size allowed in the Swarm configuration",
                [$this->maxSize]
            );

            $messages[Attachment::FIELD_SIZE] =
                [
                    Between::NOT_BETWEEN => $messages[Attachment::FIELD_SIZE][Between::NOT_BETWEEN]
                        . ". " . $messageSuffix
                ];
        }
        return $messages;
    }
}
