<?php
/**
 * P4PHP Perforce StreamLoginResponse handler.
 *
 * This handler allows the user to specify a maximum number of output blocks
 * to report. If more are received, they will go unreported and the command
 * cancelled.
 *
 * Additionally, you can specify an 'filter' callback. If set, blocks failing
 * the filter will not be reported in the result and won't count against max
 * if a limit has been specified.
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */
namespace P4\OutputHandler;

use Authentication\Service\SsoLoginAdapter;

class StreamLoginResponse extends \P4_OutputHandlerAbstract
{
    const LOG_PREFIX     = StreamLoginResponse::class;
    const TTL            = 60;
    const HANDLER_REPORT = 0;
    private $cacheService;
    private $loggerService;
    private $p4;
    private $userId;
    /**
     *
     */
    public function __construct($cacheService, $loggerService = null, $userP4 = null, $currentUser = null)
    {
        $this->p4            = $userP4;
        $this->userId        = $currentUser;
        $this->cacheService  = $cacheService;
        $this->loggerService = $loggerService;
    }
    public function outputStat($data): int
    {
        return self::HANDLER_REPORT;
    }
    public function outputInfo($data): int
    {
        $user = $this->p4->getUser();
        // delete the redis cache key if it already exists
        $this->cacheService->delete($user . SsoLoginAdapter::SSO_NAV_URL_REDIS_KEY);
        $url = str_ireplace("Navigate to URL: ", "", $data);
        if (filter_var($url, FILTER_VALIDATE_URL)) {
            $this->loggerService->info(
                sprintf(
                    "Setting navigation url for the redis key: [%s] value: [%s]",
                    $user . SsoLoginAdapter::SSO_NAV_URL_REDIS_KEY,
                    $url
                )
            );
            // set the redis cache key as "username" + navigationUrl & value as redirection url
            $this->cacheService->set(
                $user . SsoLoginAdapter::SSO_NAV_URL_REDIS_KEY,
                $url,
                self::TTL
            );
        } else {
            $this->loggerService->info(
                sprintf(
                    "Failed to set navigation url for the redis key: [%s] ",
                    $user . SsoLoginAdapter::SSO_NAV_URL_REDIS_KEY
                )
            );
        }


        // @TODO set ttl so that the data gets auto removed
        return self::HANDLER_REPORT;
    }
    public function outputText($data): int
    {
        return self::HANDLER_REPORT;
    }
    public function outputBinary($data): int
    {
        return self::HANDLER_REPORT;
    }
    public function outputWarning($data): int
    {
        $this->loggerService->warn(
            sprintf(
                "Warning [%s] generated while streaming the the navigation url for the redis key: [%s]",
                $data,
                $this->p4->getUser(). SsoLoginAdapter::SSO_NAV_URL_REDIS_KEY
            )
        );
        return self::HANDLER_REPORT;
    }
    public function outputError($data): int
    {
        $this->loggerService->err(
            sprintf(
                "Error [%s] occurred while streaming the the navigation url for the redis key: [%s]",
                $data,
                $this->p4->getUser(). SsoLoginAdapter::SSO_NAV_URL_REDIS_KEY
            )
        );
        return self::HANDLER_REPORT;
    }
}
