<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.2/2785633
 */

namespace Comments\Controller;

use Api\Controller\AbstractRestfulController;
use Api\IRequest;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Api\Exception\ConflictException;
use Application\Permissions\Exception\ForbiddenException;
use Application\Permissions\Exception\UnauthorizedException;
use Application\Permissions\IPermissions;
use Comments\Filter\IComment as ICommentFilter;
use Comments\Filter\IMarkAsRead;
use Comments\Filter\IAddRemoveReaction;
use Comments\Filter\IParameters;
use Comments\Model\Comment;
use Comments\Model\IComment;
use Comments\Validator\Notify;
use Exception;
use InvalidArgumentException;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use P4\Model\Fielded\Iterator;
use P4\Spec\Exception\NotFoundException;
use Record\Exception\NotFoundException as RecordNotFoundException;
use Record\Key\AbstractKey;

/**
 * Class CommentApi
 * @package Comments\Controller
 */
class CommentApi extends AbstractRestfulController implements ICommentApi
{
    const DATA_COMMENTS = 'comments';

    /**
     * Create a new comment for Swarm
     * @param mixed $data
     * @return  JsonModel
     */
    public function create($data): JsonModel
    {
        $errors     = [];
        $comment    = null;
        $filterData = [];

        $userDelayNotificationCount = 0;
        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->check(IPermissions::AUTHENTICATED_CHECKER);
            $paramsFilter = $this->services->get(IParameters::EDIT_COMMENTS_PARAMETERS_FILTER);
            $bodyFilter   = $this->services->build(
                ICommentFilter::COMMENTS_CREATE_FILTER,
                [IRequest::VERSION => $this->getEvent()->getRouteMatch()->getParam(IRequest::VERSION)]
            );

            $params                       = $this->getRequest()->getQuery()->toArray();
            $params[Notify::NOTIFY_FIELD] = $params[Notify::NOTIFY_FIELD] ?? Notify::DELAYED;
            unset($params[IRequest::FORMAT]);
            if ($params) {
                $paramsFilter->setData($params);
                if ($paramsFilter->isValid()) {
                    $filterData = $paramsFilter->getValues();
                } else {
                    $errors = $paramsFilter->getMessages();
                }
            }

            $bodyFilter->setData($this->prepareDataForFilter($data));
            if ($bodyFilter->isValid()) {
                $filterData = array_merge($filterData, $bodyFilter->getValues());
            } else {
                $errors = array_merge($errors, $bodyFilter->getMessages());
            }

            if ($errors) {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            } else {
                $p4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
                $commentDao = $this->services->get(IDao::COMMENT_DAO);
                $comment    = new Comment($p4Admin);
                if (!empty($filterData['context']['content'])) {
                    $content = $filterData['context']['content'];
                    if (is_string($content)) {
                        $decodedContent                   = base64_decode($content, true);
                        $content                          = json_decode($decodedContent, true);
                        $filterData['context']['content'] = $content;
                    }
                }
                $comment->set($filterData);
                $comment = $commentDao->save($comment, $filterData[Notify::NOTIFY_FIELD]);
                // Get the delayed comment after creating this comment.
                $userDelayNotificationCount = $commentDao->getUserDelayedNotificationCount(
                    $comment->get(IComment::TOPIC)
                );
            }
        } catch (ForbiddenException $e) {
            $errors = [$this->buildMessage(Response::STATUS_CODE_403, $e->getMessage())];
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
        } catch (RecordNotFoundException | NotFoundException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (InvalidArgumentException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_409);
            $errors = [$this->buildMessage(Response::STATUS_CODE_409, $e->getMessage())];
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, 'Unauthorized')];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success(
                [self::DATA_COMMENTS => [$comment->toArray()],
                IComment::USER_DELAYED_NOTIFICATION_COUNT => $userDelayNotificationCount]
            );
        }
        return $json;
    }

    /**
     * @inheritDoc
     */
    public function editAction() : JsonModel
    {
        $errors  = [];
        $data    = [];
        $comment = null;

        $userDelayNotificationCount = 0;

        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->check(IPermissions::AUTHENTICATED_CHECKER);
            $request      = $this->getRequest();
            $query        = $request->getQuery();
            $paramsFilter = $this->services->get(IParameters::EDIT_COMMENTS_PARAMETERS_FILTER);
            $bodyFilter   = $this->services->build(
                ICommentFilter::COMMENTS_EDIT_FILTER,
                [IRequest::VERSION => $this->getEvent()->getRouteMatch()->getParam(IRequest::VERSION)]
            );
            $commentId    = $this->getEvent()->getRouteMatch()->getParam(IComment::ID);
            $params       = $query->toArray();
            unset($params[IRequest::FORMAT]);
            if ($params) {
                $paramsFilter->setData($query->toArray());
                if ($paramsFilter->isValid()) {
                    $data = $paramsFilter->getValues();
                } else {
                    $errors = $paramsFilter->getMessages();
                }
            }
            $bodyFilter->setData(json_decode($request->getContent(), true));
            if ($bodyFilter->isValid()) {
                $data = array_merge($data, $bodyFilter->getValues());
            } else {
                $errors = array_merge($errors, $bodyFilter->getMessages());
            }
            if ($errors) {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            } else {
                $dao     = $this->services->get(IDao::COMMENT_DAO);
                $comment = $dao->edit($commentId, $data);
                // get the delayed comment after editing this comment.
                $userDelayNotificationCount = $dao->getUserDelayedNotificationCount($comment->get(IComment::TOPIC));
            }
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = [$this->buildMessage(Response::STATUS_CODE_403, $e->getMessage())];
        } catch (RecordNotFoundException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, 'Unauthorized')];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success(
                [self::DATA_COMMENTS => [$comment->toArray()],
                IComment::USER_DELAYED_NOTIFICATION_COUNT => $userDelayNotificationCount]
            );
        }
        return $json;
    }

    /**
     * @inheritDoc
     */
    public function addReactionAction() : Jsonmodel
    {
        return $this->processReactionsData();
    }

    /**
     * @inheritDoc
     */
    public function removeReactionAction() : Jsonmodel
    {
        return $this->processReactionsData();
    }

    /**
     * Processes reactions data.
     * @return JsonModel
     */
    protected function processReactionsData() : JsonModel
    {
        $errors  = [];
        $comment = null;

        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->check(IPermissions::AUTHENTICATED_CHECKER);
            $request   = $this->getRequest();
            $commentId = $this->getEvent()->getRouteMatch()->getParam(IComment::ID);
            $filter    = $this->services->get(IAddRemoveReaction::ADD_REMOVE_REACTION_FILTER);
            $filter->setData(json_decode($request->getContent(), true));
            if ($filter->isValid()) {
                $data    = $filter->getValues();
                $dao     = $this->services->get(IDao::COMMENT_DAO);
                $comment = $dao->updateReactions($commentId, $data, $this->getRequest()->getMethod());
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $errors = $filter->getMessages();
            }
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = [$this->buildMessage(Response::STATUS_CODE_403, $e->getMessage())];
        } catch (RecordNotFoundException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, 'Unauthorized')];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success(
                [self::DATA_COMMENTS => [$comment->toArray()]]
            );
        }
        return $json;
    }

    /**
     * get comment by comment id
     * @param mixed $id
     * @return JsonModel
     */
    public function get($id): JsonModel
    {
        $p4Admin  = $this->services->get(ConnectionFactory::P4_ADMIN);
        $error    = null;
        $comments = [];
        try {
            $fields       = $this->getRequest()->getQuery(IRequest::FIELDS);
            $dao          = $this->services->get(IDao::COMMENT_DAO);
            $commentsData = $dao->fetch($id, $p4Admin)->toArray();
            $comments     = $this->limitFields($commentsData, $fields);
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $error = $this->buildMessage(Response::STATUS_CODE_403, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            // comment id is good but no record found
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $error = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (InvalidArgumentException $e) {
            // Comment doesn't have a validator on how a comment ID should be.
            // Leaving this here in case we do later.
            // comment id not correct form, part of the path so use 404
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $error = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (Exception $e) {
            // unknown error just catch and return.
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $error = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($error) {
            $json = $this->error([$error], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_COMMENTS => $comments]);
        }
        return $json;
    }

    /**
     * @inheritDoc
     */
    public function sendNotificationAction(): JsonModel
    {
        $error = null;
        $count = [];
        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->check(IPermissions::AUTHENTICATED_CHECKER);
            $topic_id = $this->getEvent()->getRouteMatch()->getParam(ICommentApi::TOPIC_ID);
            $topic    = $this->getEvent()->getRouteMatch()->getParam(IComment::TOPIC);
            $dao      = $this->services->get(IDao::COMMENT_DAO);
            $count    = $dao->sendDelayedComments($topic. '/'.$topic_id);
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $error = $this->buildMessage(Response::STATUS_CODE_403, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            // topic and topic_id is good but no record found
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $error = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $error = $this->buildMessage(Response::STATUS_CODE_401, "Unauthorized");
        } catch (Exception $e) {
            // unknown error just catch and return.
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $error = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($error) {
            $json = $this->error([$error], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([IComment::USER_DELAYED_NOTIFICATION_COUNT => $count]);
        }
        return $json;
    }

    /**
     * @inheritDoc
     */
    public function getDelayNotificationCountAction(): JsonModel
    {
        $error                      = null;
        $userDelayNotificationCount = [];
        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->check(IPermissions::AUTHENTICATED_CHECKER);
            $topic_id = $this->getEvent()->getRouteMatch()->getParam(ICommentApi::TOPIC_ID);
            $topic    = $this->getEvent()->getRouteMatch()->getParam(IComment::TOPIC);
            $dao      = $this->services->get(IDao::COMMENT_DAO);
            // Get the users delayed notifications count.
            $userDelayNotificationCount = $dao->getUserDelayedNotificationCount($topic. '/'.$topic_id);
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $error = $this->buildMessage(Response::STATUS_CODE_403, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            // topic and topic_id is good but no record found
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $error = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $error = $this->buildMessage(Response::STATUS_CODE_401, "Unauthorized");
        } catch (Exception $e) {
            // unknown error just catch and return.
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $error = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($error) {
            $json = $this->error([$error], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([IComment::USER_DELAYED_NOTIFICATION_COUNT => $userDelayNotificationCount]);
        }
        return $json;
    }

    /**
     * @inheritDoc
     */
    public function getCommentsByTopicIdAction(): JsonModel
    {
        $topic        = $this->getEvent()->getRouteMatch()->getParam(IComment::TOPIC);
        $id           = $this->getEvent()->getRouteMatch()->getParam(IComment::ID);
        $comments     = [];
        $commentsData = null;
        $errors       = null;
        $request      = $this->getRequest();
        $query        = $request->getQuery();
        try {
            $options = $query->toArray();
            $filter  = $this->services->build(IParameters::COMMENTS_PARAMETERS_FILTER, $options);
            $filter->setData($options);
            if ($filter->isValid()) {
                $options      = $filter->getValues();
                $fields       = $this->getRequest()->getQuery(IRequest::FIELDS);
                $dao          = $this->services->get(IDao::COMMENT_DAO);
                $commentsData = $dao->fetchByTopic($topic, $id, $options);
                $comments     = $this->limitFieldsForAll($this->modelsToArray($commentsData), $fields);
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $errors = $filter->getMessages();
            }
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = [$this->buildMessage(Response::STATUS_CODE_403, $e->getMessage())];
        } catch (NotFoundException | RecordNotFoundException | InvalidArgumentException $e) {
            // Record id is either not correctly formed or does not reference a valid record
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (Exception $e) {
            // unknown error just catch and return.
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $tasksOnly  = $query->get('tasksOnly');
            $totalCount = $tasksOnly && $tasksOnly === 'true'
                ? count($comments)
                : $commentsData->getProperty(AbstractKey::FETCH_TOTAL_COUNT);

            $result = [
                self::DATA_COMMENTS             => $comments,
                AbstractKey::FETCH_TOTAL_COUNT  => $totalCount,
                AbstractKey::LAST_SEEN          => $commentsData->getProperty(AbstractKey::LAST_SEEN),
            ];
            $json   = $this->success($result);
        }
        return $json;
    }

    /**
     * @inheritDoc
     */
    public function markAllCommentsAsReadOrUnreadAction() : JsonModel
    {
        $topic       = $this->getEvent()->getRouteMatch()->getParam(IComment::TOPIC);
        $readByEvent = $this->getEvent()->getRouteMatch()->getParam(IComment::READ_BY_EVENT);
        $topicId     = $this->getEvent()->getRouteMatch()->getParam(IComment::TOPIC_ID);
        $errors      = null;
        $comments    = [];
        try {
            $fields       = $this->getRequest()->getQuery(IRequest::FIELDS);
            $dao          = $this->services->get(IDao::COMMENT_DAO);
            $commentsData = $dao->markAllCommentsAs($readByEvent, $topic, $topicId);
            $comments     = $this->limitFieldsForAll($this->modelsToArray($commentsData), $fields);
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = [$this->buildMessage(Response::STATUS_CODE_403, $e->getMessage())];
        } catch (NotFoundException | RecordNotFoundException $e) {
            // Topic id is either not correctly formed or does not reference a valid record
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (Exception $e) {
            // unknown error just catch and return.
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $result = [
                self::DATA_COMMENTS => $comments,
            ];
            $json   = $this->success($result);
        }
        return $json;
    }

    /**
     * @inheritDoc
     */
    public function markCommentAsReadOrUnreadAction(): JsonModel
    {
        $readByEvent = $this->getEvent()->getRouteMatch()->getParam(IComment::READ_BY_EVENT);
        $id          = $this->getEvent()->getRouteMatch()->getParam(IComment::ID);
        $data        = null;
        $errors      = null;
        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->check(IPermissions::AUTHENTICATED_CHECKER);
            $dao = $this->services->get(IDAO::COMMENT_DAO);
            if ($readByEvent === IComment::READ) {
                $filter = $this->services->get(IMarkAsRead::MARK_AS_READ_UPDATE_FILTER);
                $filter->setData($this->getRequest()->getQuery()->toArray());
                if ($filter->isValid()) {
                    $params = $filter->getValues();
                    $data   = $dao->markCommentAsRead($id, $params);
                } else {
                    $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                    $errors = $filter->getMessages();
                }
            } else {
                $data = $dao->markCommentAsUnread($id);
            }
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = [$this->buildMessage(Response::STATUS_CODE_403, $e->getMessage())];
        } catch (NotFoundException | RecordNotFoundException | InvalidArgumentException $e) {
            // Record id is either not correctly formed or does not reference a valid record
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, "Unauthorized")];
        } catch (ConflictException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_409);
            $errors = [$this->buildMessage(Response::STATUS_CODE_409, $e->getMessage())];
            $data   = [
                self::DATA_COMMENTS => [$e->getData()->toArray()],
            ];
        } catch (Exception $e) {
            // unknown error just catch and return.
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode(), $data);
        } else {
            $result = [
                self::DATA_COMMENTS => [$data->toArray()],
            ];
            $json   = $this->success($result);
        }
        return $json;
    }

    /**
     * Convert an iterator of Comments to an array representation
     * @param Iterator     $comments            iterator of comments
     * @return array
     */
    protected function modelsToArray(Iterator $comments): array
    {
        return array_values($comments->toArray());
    }

    /**
     * Prepare data for filter from route params
     * @param mixed $data
     * @return array
     */
    protected function prepareDataForFilter($data): array
    {
        $routeMatch = $this->getEvent()->getRouteMatch();
        $topic      = $routeMatch->getParam(IComment::TOPIC);
        $topic_id   = $routeMatch->getParam(ICommentApi::TOPIC_ID);
        $comment_id = $routeMatch->getParam(IComment::ID);

        if ($topic && $topic_id) {
            $data[IComment::TOPIC] = $topic . '/' . $topic_id;
        }
        if ($comment_id) {
            $data[IComment::CONTEXT][IComment::COMMENT] = (int) $comment_id;
        }

        return $data;
    }

    /**
     * @inheritDoc
     */
    public function archiveOrUnArchiveAction() : JsonModel
    {
        $error      = null;
        $comments   = [];
        $routeMatch = $this->getEvent()->getRouteMatch();
        $commentId  = $routeMatch->getParam(IComment::ID);
        $archive    = $routeMatch->getParam(IComment::ARCHIVE_OPERATION) === IComment::ACTION_ARCHIVE;
        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->check(IPermissions::AUTHENTICATED_CHECKER);
            $commentDao = $this->services->get(IDao::COMMENT_DAO);
            $comments   = $archive
                ? $commentDao->archiveComment($commentId)
                : $commentDao->unArchiveComment($commentId);
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $error = $this->buildMessage(Response::STATUS_CODE_403, $e->getMessage());
        } catch (RecordNotFoundException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $error = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $error = $this->buildMessage(Response::STATUS_CODE_401, $e->getMessage());
        } catch (Exception $e) {
            // unknown error just catch and return.
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $error = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($error) {
            $json = $this->error([$error], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_COMMENTS => $comments->toArray()]);
        }
        return $json;
    }
}
