<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.2/2785633
 */
namespace AiAnalysis\Model;

use Application\Config\IConfigDefinition as IConfig;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Helper\StringHelper;
use Application\I18n\TranslatorFactory;
use Application\Model\AbstractDAO;
use Application\Permissions\Exception\ForbiddenException;
use Application\Permissions\IpProtects;
use Application\Permissions\RestrictedChanges;
use DateTime;
use P4\Connection\ConnectionInterface;
use P4\File\Exception\NotFoundException;
use P4\File\File;
use P4\Model\Fielded\Iterator;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Record\Key\AbstractKey;

/**
 * Class AiAnalysisDAO to handle access to Perforce AI Analysis data
 * @package AiAnalysis\Model
 */
class AiAnalysisDAO extends AbstractDAO implements IAiReviewSummary
{
    // The Perforce class that handles AI analysis
    const MODEL = AiReviewSummary::class;

    /**
     * Save the AiReviewSummary
     * @param AiReviewSummary $model  The AiReviewSummary model.
     * @return AiReviewSummary the saved AiReviewSummary
     */
    public function save($model): AiReviewSummary
    {
        return parent::save($model);
    }

    /**
     * Check to save or update of record for ai summary done
     * Check currently done before update:
     *  - check if topic is available
     *      - If yes then check if the ai topic is available
     *  - If the topic is not available then create the new entry
     *
     * @param array $data The posted data.
     * @param array $aiPackage The ai package details with which review is analyzed.
     * @param string|null $dataReceived The data received from AI vendor.
     * @param AiReviewSummary|null $aiReviewSummary
     * @param string|null $anyErrorMsg Any error received from AI vendor.
     * @return AiReviewSummary the saved data
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function saveOrUpdate(
        array $data,
        ?array $aiPackage,
        ?string $dataReceived,
        ?AiReviewSummary $aiReviewSummary,
        ?string $anyErrorMsg
    ): AiReviewSummary {
        if (!$aiReviewSummary) {
            $p4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
            $filterData = [
                $data[IAiReviewSummary::REVIEW_ID] => [
                    $data[IAiReviewSummary::FROM_VERSION] . '_' . $data[IAiReviewSummary::TO_VERSION] => [
                        $data[IAiReviewSummary::FILE_ID] => [
                            $data[IAiReviewSummary::DIFF_START] . '_' . $data[IAiReviewSummary::DIFF_END] => [
                                $aiPackage[IConfig::AI_PACKAGE_ID] => [
                                    IConfig::AI_PACKAGE_ID => $aiPackage[IConfig::AI_PACKAGE_ID],
                                    IAiReviewSummary::AI_SERVICE_NAME => $aiPackage[IConfig::AI_PACKAGE_VALUE],
                                    IAiReviewSummary::AI_PACKAGE_TYPE => $aiPackage[IConfig::AI_PACKAGE_TYPE],
                                    IAiReviewSummary::AI_MODEL => $aiPackage[IConfig::AI_MODEL],
                                    IAiReviewSummary::USERS => [$this->services->get(ConnectionFactory::USER)->getId()],
                                    IAiReviewSummary::DIGEST => $data[IAiReviewSummary::DIGEST],
                                    IAiReviewSummary::LEFT_LINE => $data[IAiReviewSummary::LEFT_LINE],
                                    IAiReviewSummary::CONTENT => $dataReceived,
                                    IAiReviewSummary::ERROR => $anyErrorMsg ?? false,
                                ]
                            ]
                        ]
                    ]
                ],
                IAiReviewSummary::TOPIC => IAiReviewSummary::REVIEW . '/' . $data[IAiReviewSummary::REVIEW_ID]
            ];

            $aiReviewSummary = AiReviewSummary::createAiReviewSummary($filterData, $p4Admin);
        }
        $this->save($aiReviewSummary);
        return $aiReviewSummary;
    }

    /**
     * Based on the passed parameters check if the result already available
     * for the given diff and review
     * @param int $id The posted data
     * @return AiReviewSummary
     * @throws ForbiddenException
     */
    public function fetchIfResultAvailable(int $id): AiReviewSummary
    {
        $p4Admin         = $this->services->get(ConnectionFactory::P4_ADMIN);
        $aiReviewSummary = $this->fetchById($id, $p4Admin);
        $this->checkRestrictedAccess($aiReviewSummary->get(IAiReviewSummary::TOPIC));
        return $aiReviewSummary;
    }

    /**
     * fetch AI review summary data according topic and topic id
     * @param string $topic topic string (review)
     * @param string $id topic id
     * @param array $options options to filter result
     * @return Iterator|string
     * @throws ForbiddenException
     */
    public function fetchByTopic(string $topic, string $id, array $options = []): Iterator|string
    {
        return match ($topic) {
            IAiReviewSummary::TOPIC_REVIEW => $this->fetchByReview($id, $options),
            default => '',
        };
    }

    /**
     * Fetch all the ai review summary for a given Review Id.
     * @param mixed $reviewId the review ID
     * @param mixed $options fetch options
     * @return Iterator
     * @throws ForbiddenException
     */
    public function fetchByReview($reviewId, $options = []) : Iterator
    {
        $p4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
        $topic      = self::TOPIC_REVIEW.'/'.$reviewId;
        $ipProtects = $this->services->get(IpProtects::IP_PROTECTS);
        // To check access.
        $canAccess = $this->checkRestrictedAccess($topic);
        if ($canAccess) {
            $options += [
                self::TOPIC => [$topic],
                AbstractKey::FETCH_TOTAL_COUNT => true
            ];
        }
        return $this->fetchAllWithProtection($options, $p4Admin, $ipProtects);
    }

    /**
     * check whether user has access to review
     * @param string $topicString
     * @return bool
     * @throws ForbiddenException
     */
    public function checkRestrictedAccess(string $topicString): bool
    {
        if (preg_match(
            "#(".self::TOPIC_REVIEW.")/([0-9a-zA-z]+)#",
            $topicString,
            $matches
        )
        ) {
            $topic      = $matches[1];
            $id         = $matches[2];
            $p4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
            $translator = $this->services->get(TranslatorFactory::SERVICE);
            // To check access.
            if ($topic === self::TOPIC_REVIEW) {
                $reviewDAO = $this->services->get(IDao::REVIEW_DAO);
                $review    = $reviewDAO->fetchById($id, $p4Admin);
                if ($review === false || !$this->services->get(RestrictedChanges::class)->canAccess($review->getId())) {
                    throw new ForbiddenException(
                        $translator->t($id." does not exist, or you do not have permission to view it.")
                    );
                }
            }
        }
        return true;
    }

    /**
     * Validate that the review contains the file id passed.
     * @param string $fileId
     * @param int $reviewId
     * @param int $version
     * @return bool
     * @throws ContainerExceptionInterface
     */
    public function validateFile(string $fileId, int $reviewId, int $version): bool
    {
        $p4User    = $this->services->get(ConnectionFactory::P4_USER);
        $p4Admin   = $this->services->get(ConnectionFactory::P4_ADMIN);
        $reviewDAO = $this->services->get(IDao::REVIEW_DAO);
        $review    = $reviewDAO->fetchById($reviewId, $p4Admin);

        try {
            $change = $review->getChangeOfVersion($version, true);
        } catch (\Exception $e) {
            throw new \InvalidArgumentException(
                sprintf("Passed version [%s] is not a valid version for this review", $version)
            );
        }

        // check if file path exists in the review
        try {
            $file = StringHelper::base64DecodeUrl($fileId);
            $file = trim($file, '/');
            $file = strlen($file) ? '//' . $file : null;
            $file = File::fetch($file ? $file . '@=' . $change : null, $p4User);
        } catch (NotFoundException $e) {
            throw new \InvalidArgumentException(
                sprintf("File id passed [%s] is not a valid file for this review", $fileId)
            );
        }
        return true;
    }

    /**
     * As the fetchAll function in the model takes extra param of protections we have to write
     * this wrapper to handle that.
     * Retrieves all records that match the passed options.
     * Extends parent to compose a search query when fetching by topic.
     *
     * @param array           $options      an optional array of search conditions and/or options
     *                                      supported options are:
     *                                      FETCH_BY_TOPIC - set to a 'topic' id to limit results
     * @param ConnectionInterface|null $connection the connection to set on the filer
     * @param null            $protects            the users protections list.
     * @return  Iterator                    the list of zero or more matching AiReviewSummary objects
     */
    public function fetchAllWithProtection(
        array $options = [],
        ConnectionInterface $connection = null,
        $protects = null
    ): Iterator {
        // By default this simply passes on to model, with extra option of protections.
        return call_user_func(
            static::MODEL  . '::fetchAll',
            $options,
            $this->getConnection($connection),
            $protects
        );
    }

    /**
     * This function is wrapper to perform to actions one is to fetch all records and then filter out the records
     * @param array $bodyData posted data
     * @return array
     * @throws ForbiddenException
     */
    public function fetchAndFilterAiReviewSummaries(array $bodyData): array
    {
        $reviewId    = null;
        $fromVersion = null;
        $toVersion   = null;
        if ($bodyData) {
            $reviewId    = $bodyData[IAiReviewSummary::REVIEW_ID];
            $fromVersion = $bodyData[IAiReviewSummary::FROM_VERSION];
            $toVersion   = $bodyData[IAiReviewSummary::TO_VERSION];
        }
        $aiReviewSummaries = $this->fetchByTopic(IAiReviewSummary::TOPIC_REVIEW, $reviewId);
        if ($aiReviewSummaries) {
            return $this->filterOutAiReviewSummaries(
                $aiReviewSummaries->toArray(),
                $reviewId,
                $fromVersion,
                $toVersion
            );
        }
        return [];
    }

    /**
     * Filters out unnecessary records for the specific version of review for the current user.
     * @param array $aiReviewSummaries
     * @param int $reviewId
     * @param int $fromVersion
     * @param int $toVersion
     * @return array
     */
    private function filterOutAiReviewSummaries(
        array $aiReviewSummaries,
        int $reviewId,
        int $fromVersion,
        int $toVersion
    ) : array {
        $currentUser = $this->services->get(ConnectionFactory::USER)->getId();
        $versionKey  = "{$fromVersion}_{$toVersion}";

        foreach ($aiReviewSummaries as $fromToVersion => $aiReviewSummary) {
            if (!isset($aiReviewSummary[$reviewId][$versionKey])) {
                unset($aiReviewSummaries[$fromToVersion]);
                continue;
            }
            $diffStartEnd   = current($aiReviewSummary[$reviewId][$versionKey]);
            $recordToVerify = current(current($diffStartEnd)); // Double current to get the innermost array

            // Check if the current user exists in the specified records
            $aiReviewSummaries[$fromToVersion][IAiReviewSummary::IS_VISIBLE] =
                isset($recordToVerify[IAiReviewSummary::USERS]) &&
                in_array($currentUser, $recordToVerify[IAiReviewSummary::USERS]);
        }

        return $aiReviewSummaries;
    }

    /**
     * This function will hard delete the AI review summary records from the p4 keys
     * which are older than the retention tenure
     * @param string $aiDataRetention
     * @return array
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \Exception
     */
    public function removeAiSummaries(string $aiDataRetention): array
    {
        $p4Admin   = $this->services->get(ConnectionFactory::P4_ADMIN);
        $timestamp = (new DateTime())->modify("-$aiDataRetention")->getTimestamp();

        $aiReviewSummaries = AiReviewSummary::fetchRetainedSummaries($p4Admin);
        $deletedSummaries  = [];

        foreach ($aiReviewSummaries as $aiReviewSummary) {
            if ($aiReviewSummary->get('updated') <= $timestamp) {
                $deletedSummaries[] = [
                    'id' => $aiReviewSummary->get('id'),
                    'updated' => date("Y-M-d", $aiReviewSummary->get('updated')),
                ];
                $aiReviewSummary->delete();
            }
        }

        return $deletedSummaries;
    }
}
