<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.2/2785633
 */

namespace Events\Listener;

use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Log\SwarmLogger;
use Laminas\EventManager\Event;
use Laminas\EventManager\EventManagerInterface;
use Laminas\EventManager\ListenerAggregateInterface;
use Laminas\EventManager\ListenerAggregateTrait;
use Laminas\Log\Logger;
use Laminas\ServiceManager\ServiceLocatorInterface as ServiceLocator;

abstract class AbstractEventListener implements ListenerAggregateInterface
{
    use ListenerAggregateTrait;

    protected $services    = null;
    protected $eventConfig = null;
    protected $logger      = null;

    /**
     * Ensure we get a service locator and event config on construction.
     *
     * @param   ServiceLocator  $services       the service locator to use
     * @param   array           $eventConfig    the event config for this listener
     */
    public function __construct(ServiceLocator $services, array $eventConfig)
    {
        $this->services    = $services;
        $this->eventConfig = $eventConfig;
        $this->logger      = $services->get(SwarmLogger::SERVICE);
    }

    /**
     * TODO generic things that handlers want to do with a call to parent::handle($event)
     * @param Event $event
     */
    public function handle(Event $event)
    {
    }

    /**
     * Log event
     * @param Event $event
     * @throws ConfigException
     */
    public function log(Event $event)
    {
        $traceLog = ConfigManager::getValue(
            $this->services->get(IConfigDefinition::CONFIG),
            IConfigDefinition::LOG_EVENT_TRACE,
            false
        );
        if ($traceLog) {
            $function  = debug_backtrace(1)[1]['function'];
            $className = get_class($this);
            $priority  = null;
            foreach ($this->eventConfig as $eventName => $eventDetails) {
                foreach ($eventDetails as $eventDetail) {
                    if ($eventDetail[ListenerFactory::CALLBACK] === $function) {
                        $priority = $eventDetail[ListenerFactory::PRIORITY];
                        break;
                    }
                }
            }
            $this->logger->trace(
                "Handler class [$className], function ['$function']" .
                ($priority ? ", priority [$priority]" : '')
            );
        }
    }

    /**
     * Get the controller if it has an event manager (determined by the route).
     * @param Event $event
     * @return mixed|null the controller from the route if it has an event manager, otherwise null
     */
    public function getControllerFromRoute(Event $event)
    {
        $controller       = null;
        $routeMatch       = $event->getRouteMatch();
        $controllerName   = $routeMatch->getParam('controller');
        $controllerLoader = $event->getApplication()->getServiceManager()->get('ControllerManager');
        if ($controllerLoader->has($controllerName)) {
            try {
                $controller = $controllerLoader->get($controllerName);
            } catch (\Exception $e) {
                // let the dispatch listener handle bad controllers
                $this->logger->trace(
                    "Controller [$controllerName] not found when requesting to attach events"
                );
            }
        }
        return $controller;
    }

    /**
     * Whether the event should be attached. True by default, gives the sub-classes a
     * chance to choose not to attach on a per event basis
     * @param mixed $eventName the event name
     * @param array $eventDetail the event detail
     * @return bool
     */
    protected function shouldAttach($eventName, $eventDetail)
    {
        return true;
    }

    /**
     * @inheritDoc
     * @throws ConfigException
     */
    public function attach(EventManagerInterface $events, $priority = 1)
    {
        $eventManager = $events;
        $traceLog     = ConfigManager::getValue(
            $this->services->get(IConfigDefinition::CONFIG),
            IConfigDefinition::LOG_EVENT_TRACE,
            false
        );
        foreach ($this->eventConfig as $eventName => $eventDetails) {
            foreach ($eventDetails as $eventDetail) {
                if ($this->shouldAttach($eventName, $eventDetail)) {
                    $managerContext = $eventDetail[ListenerFactory::MANAGER_CONTEXT] ?? null;
                    if ($managerContext) {
                        $eventManager =
                            $this->services->get($eventDetail[ListenerFactory::MANAGER_CONTEXT])->getEventManager();
                    }
                    if ($traceLog) {
                        $this->logger->trace(
                            "Attaching [$eventName] to [" . get_class($this) .
                            "] with priority [" . $eventDetail[ListenerFactory::PRIORITY] .
                            "] and callback [" . $eventDetail[ListenerFactory::CALLBACK] .
                            ($managerContext ? "] and context [$managerContext]" : "]")
                        );
                    }
                    $this->listeners[] = $eventManager->attach(
                        $eventName,
                        [$this, $eventDetail[ ListenerFactory::CALLBACK]],
                        $eventDetail[ListenerFactory::PRIORITY]
                    );
                } else {
                    if ($traceLog) {
                        $this->logger->trace(
                            "Listener [" . get_class($this) . "] decided not to attach to [$eventName]" .
                            " for callback [" . $eventDetail[ListenerFactory::CALLBACK] . "]"
                        );
                    }
                }
            }
        }
    }

    /**
     * This is a helper function to check the log level and if its greater than  lowestPriority we process the log
     * message.
     * @param string     $msg            The message we want to append to the class string.
     * @param string|int $priority       The priority of the log from the config.
     * @param string|int $lowestPriority The lowest priority we want to log this message for.
     * @return void
     */
    protected function logMsg(string $msg, $priority, $lowestPriority = Logger::DEBUG)
    {
        if ($priority > $lowestPriority) {
            $this->logger->log($priority, get_class($this) . ": " . $msg);
        }
    }
}
