<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.2/2785633
 */

namespace Changes\Listener;

use Application\Config\ConfigException as ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition as IDef;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Events\Listener\AbstractEventListener;
use P4\Connection\Exception\CommandException;
use Queue\Manager as QueueManager;
use Record\Lock\Lock;
use Reviews\Model\IReview;
use Laminas\EventManager\Event;

class ChangeListener extends AbstractEventListener
{

    const LOG_PREFIX = ChangeListener::class;

    /**
     * When a changelist is saved we go through this event. Then it will generate a changesaved event if required.
     *
     * @param Event $event
     * @return void
     * @throws ConfigException
     */
    public function handleChangeSave(Event $event)
    {
        parent::log($event);
        // We ignore default changelist as we don't need to add a future task for them.
        if ($event->getParam('id') === 'default') {
            return;
        }
        $config = $this->services->get(IDef::CONFIG);
        $delay  = ConfigManager::getValue($config, IDef::QUEUE_WORKER_CHANGE_SAVE_DELAY, 5000) / 1000;
        $this->logger->trace('Delay for task.changesaved is ' . $delay);
        // schedule the task in the future to handle the synchronization
        $this->services->get(QueueManager::SERVICE)->addTask(
            'changesaved',
            $event->getParam('id'),
            $event->getParam('data'),
            time() + $delay
        );
    }

    /**
     * If we have sync description is enabled we will process this. Only fired for changesaved events.
     *
     * @param Event $event
     * @return void
     * @throws ConfigException
     * @throws CommandException
     */
    public function descriptionSync(Event $event)
    {
        parent::log($event);
        $id      = $event->getParam('id');
        $p4Admin = $this->services->get(ConnectionFactory::P4_ADMIN);
        $config  = $this->services->get(IDef::CONFIG);

        $syncDescription = ConfigManager::getValue($config, IDef::REVIEWS_SYNC_DESCRIPTIONS);
        // if we are configured to synchronize descriptions,
        // and there is something to synchronize (it's not a new change)
        if ($syncDescription && $id !== 'default') {
            $lock = new Lock(IReview::LOCK_CHANGE_PREFIX . $id, $p4Admin);
            try {
                $changeDao = $this->services->get(IDao::CHANGE_DAO);
                $change    = $changeDao->fetchById($id, $p4Admin);
                $lock->lock();
                $reviewDao = $this->services->get(IDao::REVIEW_DAO);
                // find any associated reviews with this change, and ensure they are updated
                $reviews  = $reviewDao->fetchAll([IReview::FETCH_BY_CHANGE => $id], $p4Admin);
                $keywords = $this->services->get('review_keywords');
                foreach ($reviews as $review) {
                    $description = rtrim($keywords->filter($review->getDescription()));
                    // An extra newline character is always appear at the end of change description
                    // which results in unnecessary update description activity even when description are same.
                    // To prevent this rtrim is used to trim the extra newline character.
                    $changeDescription = rtrim($keywords->filter($change->getDescription()));
                    if ($description != $changeDescription) {
                        $review->setDescription($changeDescription)->save();
                    } else {
                        continue;
                    }
                    // schedule task.review so @mentions get updated
                    $this->services->get(QueueManager::SERVICE)->addTask(
                        'review',
                        $review->getId(),
                        [
                            'previous'            => [IReview::FIELD_DESCRIPTION => $description],
                            'isDescriptionChange' => true,
                            'user'                => $change->getUser()
                        ]
                    );
                }
            } catch (\Exception $e) {
                $this->logger->err($e);
            } finally {
                $lock->unlock();
            }
        }
    }
}
