<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.2/2785633
 */

namespace Authentication\Service;

use Application\Config\Services;
use Application\Log\SwarmLogger;
use Laminas\Http\Request;
use Laminas\Stdlib\Parameters;
use Laminas\View\Model\JsonModel;
use Authentication\Helper\ILoginHelper;

class SamlLoginAdapter extends AbstractLoginAdapter implements LoginAdapterInterface
{
    const LOG_PREFIX = SamlLoginAdapter::class;
    /**
     * Process login
     * @param mixed $data
     * @return JsonModel
     */
    public function login($data): JsonModel
    {
        $jsonModel = $this->handleSamlLogin($data);
        $data      = $jsonModel->getVariables();
        $jsonModel->clearVariables();
        $jsonModel->setVariables(['data' => $data, 'messages' => [], 'error' => null]);
        return $jsonModel;
    }

    /**
     * Handle saml login request
     *
     * @param array $data
     * @return JsonModel
     */
    public function handleSamlLogin(array $data): JsonModel
    {
        $logger       = $this->services->get(SwarmLogger::SERVICE);
        $serverParams = new Parameters($_SERVER);
        $referer      = $serverParams->get('HTTP_REFERER');
        $logger->debug(ILoginHelper::LOG_ID . 'getting php-saml library');
        $saml = $this->services->get(Services::AUTH_SAML);
        $logger->debug(ILoginHelper::LOG_ID . 'php-saml library success');
        $logger->debug(ILoginHelper::LOG_ID . 'Referrer ' . $referer);
        /* After the IDP has returned we set up to return to the page that we were on originally.
        For example if we login from the reviews page we want to end up on the reviews page */
        if (isset($data[ILoginHelper::PARAM_REDIRECT]) &&
            ($data[ILoginHelper::PARAM_REDIRECT] === false || $data[ILoginHelper::PARAM_REDIRECT] === 'false')) {
            $logger->debug(ILoginHelper::LOG_ID . 'saml login without redirect');
            $url = $saml->login($referer, [], false, false, true);
            return new JsonModel(['url' => $url, 'isValid' => true]);
        } else {
            $logger->debug(ILoginHelper::LOG_ID . 'login with redirect');
            $saml->login($referer);
            return new JsonModel(['isValid' => true]);
        }
    }

    /**
     * Handles the callback from an IDP to authenticate the Saml response
     * with the P4 connection (and hence the configured trigger)
     * @return array results
     */
    public function handleSamlResponse() : array
    {
        $error   = null;
        $userId  = null;
        $adapter = null;
        $logger  = $this->services->get(SwarmLogger::SERVICE);
        try {
            $authHelper = $this->services->get(ILoginHelper::HELPER_NAME);
            $saml       = $this->services->get(Services::AUTH_SAML);
            $logger->debug(sprintf("[%s]: handleSamlResponse processing response", self::LOG_PREFIX));
            // Need to process the response to get the username
            $saml->processResponse(null);
            $userId = $saml->getNameId();
            $logger->debug(sprintf("[%s]: handleSamlResponse user id [%s]", self::LOG_PREFIX, $userId));
            $errors = $saml->getErrors();
            if (empty($errors)) {
                // We still want to authenticate with candidates in case an email was returned
                // by the IDP
                return $authHelper->authenticateCandidates($userId, '', true);
            } else {
                $logger->err(
                    sprintf(
                        "[%s]: handleSamlResponse has errors [%s]",
                        self::LOG_PREFIX,
                        json_encode($errors, true)
                    )
                );
                $error = $saml->getLastErrorReason();
            }
        } catch (\Exception $e) {
            $error = $e->getMessage();
            $logger->err(sprintf("[%s]: An error occurred using Saml login [%s]", self::LOG_PREFIX, $error));
        }
        return [
            ILoginHelper::AUTH_USER => $userId,
            ILoginHelper::USERNAME  => $userId,
            ILoginHelper::ADAPTER   => $adapter,
            ILoginHelper::ERROR     => $error,
            ILoginHelper::IS_VALID  => $error === null
        ];
    }
}
