<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace Users\View\Helper;

use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\View\Helper\AbstractHelper;
use Application\Config\ConfigManager;
use Application\Config\ConfigException;

class UserLink extends AbstractHelper
{
    /**
     * Outputs the user ID and linkifies it if the user exists
     *
     * @param   string  $user       the user id to output and, if able, link to
     * @param   bool    $strong     optional, if false (default) not strong, if true user id wrapped in strong tag
     * @param   string  $baseUrl    optional, if specified, given string will be pre-pended to links
     * @return  string  the user id as a link if the user exists
     * @throws ConfigException
     */
    public function __invoke($user, $strong = false, $baseUrl = null)
    {
        $view     = $this->getView();
        $services = $this->services;
        $p4Admin  = $services->get(ConnectionFactory::P4_ADMIN);
        $label    = $view->escapeHtml($user);
        $config   = $services->get(IConfigDefinition::CONFIG);
        $strict   = isset($config['security']['https_strict']) && $config['security']['https_strict'];
        $uri      = $services->get('request')->getUri();
        $scheme   = $strict ? 'https' : ($uri->getScheme() ?: 'http');
        $hostName = $config['environment']['hostname']     ?: $uri->getHost();
        $hostName = $hostName ?: 'localhost';
        $port     = $uri->getPort() ?: ($scheme == 'https' ? 443 : 80);
        $port     = $scheme == 'https' && isset($config['security']['https_port']) && $config['security']['https_port']
            ? $config['security']['https_port']
            : $port;

        if ($strong) {
            $label = '<strong>' . $label . '</strong>';
        }

        $userDao = $services->get(IDao::USER_DAO);
        if (!$userDao->exists($user, $p4Admin)) {
            return $label;
        }
        $fullName        = $userDao->fetchById($user, $p4Admin)->getFullName();
        $displayFullName = ConfigManager::getValue(
            $services->get(IConfigDefinition::CONFIG),
            ConfigManager::USER_DISPLAY_FULLNAME
        );
        $label           = $displayFullName ? $fullName : $label;
        $title           = $displayFullName ? $fullName . " (" .$user .")" : $fullName;

        // assemble the default origin
        $origin = $scheme . '://' . $hostName . ($port && $port != 80 && $port != 443 ? ':' . $port : '');
        $origin = $view->escapeFullUrl($origin);

        // if no arguments given, return the baseUrl (origin + basePath)
        if (!func_num_args()) {
            return $origin . $view->basePath();
        }

        return '<a alt="'.$title.'" title="'.$title.'" href="' . $origin
            . $view->url('user', ['user' => $user]) . '">'. $label . '</a>';
    }
}
