<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */
namespace Slack\Service;

use Activity\Model\Activity;
use Activity\Model\IActivity;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition as IDef;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Factory\InvokableService;
use Interop\Container\ContainerInterface;
use InvalidArgumentException;
use Mail\MailAction;
use P4\Spec\Change;
use Projects\Model\Project;
use Reviews\Model\Review;
use Users\Model\IUser;

/**
 * SlackUtility function to be used by other services.
 *  * @package Slack\Service
 */
class Utility implements IUtility, InvokableService
{
    private $services;
    private $config;

    /**
     * Slack Utility constructor.
     * @param ContainerInterface $services
     * @param array|null $options
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services = $services;
        $this->config   = $this->services->get(IDef::CONFIG);
    }

    /**
     * @inheritDoc
     */
    public function getFileNames($change) : array
    {
        $limit     = ConfigManager::getValue($this->config, IDef::SLACK_SUMMARY_FILE_LIMIT);
        $fileNames = [];
        $files     = $change->getFileData($change->isPending());
        $count     = count($files);
        if ($count > $limit) {
            $files = array_slice($files, 0, $limit);
        }
        array_map(
            function ($file) use (&$fileNames) {
                $fileNames[] = $file['depotFile'] . '#' . $file['rev'];
            },
            $files
        );
        return [self::FILES => $fileNames, self::COUNT => $count];
    }

    /**
     * @inheritDoc
     */
    public function getHostName(): string
    {
        $hostname    = ConfigManager::getValue($this->config, IDef::ENVIRONMENT_HOSTNAME);
        $externalURL = ConfigManager::getValue($this->config, IDef::ENVIRONMENT_EXTERNAL_URL);

        if ($externalURL !== null) {
            return $externalURL;
        }
        return "http://" . $hostname;
    }

    /**
     * @inheritDoc
     */
    public function buildActivityReply(Activity $activity, array $projects, array $linkedProjects = []): string
    {
        $userDao = $this->services->get(IDao::USER_DAO);
        $p4      = $this->services->get(ConnectionFactory::P4_ADMIN);

        $userId = $activity->getRawValue(IActivity::USER);
        $action = $activity->getRawValue(IActivity::ACTION);
        $target = $activity->getRawValue(IActivity::TARGET);
        // If we are working with a description from a comment
        $rawDescription = $activity->getRawValue(IActivity::DESCRIPTION);
        $description    = '';
        switch ($action) {
            case MailAction::COMMENT_ADDED:
            case MailAction::COMMENT_EDITED:
            case MailAction::COMMENT_REPLY:
            case MailAction::DESCRIPTION_COMMENT_ADDED:
            case MailAction::DESCRIPTION_COMMENT_EDITED:
                $description = $rawDescription;
                break;
        }

        // This is put in as we noticed that we were getting errors when deploy system updated tasks.
        $fullName = "Helix Swarm";
        if ($userDao->exists($userId, $p4)) {
            $user     = $userDao->fetchById($userId, $p4);
            $fullName = $user->get(IUser::FIELD_FULLNAME);
        }

        $projects = count($projects) ? " for" .$this->buildProjectText($projects, false, $linkedProjects) : "";

        return sprintf(
            "%s (%s) %s %s%s%s",
            $userId,
            $fullName,
            $action,
            $target,
            $projects,
            $description ? ":\n" . $description : ""
        );
    }

    /**
     * @inheritDoc
     */
    public function getProjectMappings($projectList, $mappingList, $purePrivate = false): array
    {
        $result = [];

        foreach ($projectList as $project => $branches) {
            foreach ($mappingList as $type => $channels) {
                if ($project === $type) {
                    foreach ($channels as $channel) {
                        if (!isset($result[$channel])) {
                            $result[$channel] = [];
                        }
                        $result[$channel] = array_merge($result[$channel], [$project]);
                    }
                }
            }
        }

        // If we have purePrivate = true, then we shouldn't attempt to include the hardcode Slack notification. If we
        // are false then we can assess if we want to send notifications to one of the three config options.
        if (!$purePrivate) {
            if (empty($result)) {
                if (empty($projectList)) { // The review doesn't have a project, use the *empty* mappings
                    if (array_key_exists(self::NO_ASSOCIATED_PROJECTS_CHANNELS, $mappingList)) {
                        foreach ($mappingList[self::NO_ASSOCIATED_PROJECTS_CHANNELS] as $noAssociatedProjectsChannel) {
                            if (!isset($result[$noAssociatedProjectsChannel])) {
                                $result[$noAssociatedProjectsChannel] = [];
                            }
                            $result[$noAssociatedProjectsChannel] = array_merge(
                                $result[$noAssociatedProjectsChannel],
                                [IUtility::NO_PROJECT_CHANNEL]
                            );
                        }
                    }
                } else { // The review has a project, but it's not specified, use the *default* mappings
                    if (array_key_exists(self::DEFAULT_PROJECTS_CHANNELS, $mappingList)) {
                        foreach ($mappingList[self::DEFAULT_PROJECTS_CHANNELS] as $noMappedProjectsChannel) {
                            if (!isset($result[$noMappedProjectsChannel])) {
                                $result[$noMappedProjectsChannel] = [];
                            }
                            $result[$noMappedProjectsChannel] = array_merge(
                                $result[$noMappedProjectsChannel],
                                [IUtility::NO_PROJECT_CHANNEL]
                            );
                        }
                    }
                }
            }

            if (array_key_exists(self::ALL_NOTIFICATIONS_CHANNELS, $mappingList)) {
                foreach ($mappingList[self::ALL_NOTIFICATIONS_CHANNELS] as $allChannel) {
                    if (!isset($result[$allChannel])) {
                        $result[$allChannel] = [];
                    }
                    // Add '*' mappings for all reviews (regardless of above)
                    $result[$allChannel] = array_merge(
                        $result[$allChannel],
                        [IUtility::NO_PROJECT_CHANNEL]
                    );
                }
            }
        }

        return $result;
    }

    /**
     * @inheritDoc
     */
    public function buildProjectText(?array $projects, $markdown = true, array $linkedProjects = []): string
    {
        $projectDao    = $this->services->get(IDao::PROJECT_DAO);
        $p4            = $this->services->get(ConnectionFactory::P4_ADMIN);
        $text          = '';
        $newLine       = ($markdown ? "\n" : "");
        $totalProjects = count((array) $projects);
        if ($projects && $totalProjects > 0) {
            $text         = $markdown ? "*Projects*: " . "\n" : "";
            $projectCount = 0;
            foreach ($projects as $project => $branches) {
                $projectCount++;
                $projectModel    = $projectDao->fetchById($project, $p4);
                $projectBranches = $projectModel->getBranches();
                $projectName     = $projectModel->getName();
                $isPrivate       = $projectModel->isPrivate();
                // If not private create text.
                if (in_array($project, $linkedProjects) || !$isPrivate) {
                    foreach ($branches as $branch) {
                        try {
                            $branchData = $projectModel->getBranch($branch, $projectBranches);
                            $name       = $branchData[Project::FIELD_NAME];
                        } catch (InvalidArgumentException $e) {
                            $name = $branch;
                        }
                        $text = $text . " " . $projectName . ":" . $name;
                        $text = $text . (!$markdown && $projectCount < $totalProjects ?  "," : $newLine);
                    }
                }
            }
        }
        return $text;
    }

    /**
     * @inheritDoc
     */
    public function buildFileNames(Change $change): string
    {
        $text             = '';
        $summaryFileNames = ConfigManager::getValue($this->config, IDef::SLACK_SUMMARY_FILE_NAMES, false);

        if ($summaryFileNames === true) {
            $fileData = $this->getFileNames($change);
            $files    = $fileData[IUtility::FILES];
            $text     = $text . "*" . $fileData[IUtility::COUNT] . " file"
                . ($fileData[IUtility::COUNT] !== 1 ? "s have" : " has") . " been changed*";
            if ($fileData[IUtility::COUNT] > 0) {
                $text = $text . ":";
                foreach ($files as $file) {
                    $text = $text . "\n" . $file;
                }
                // Total count of files was bigger than the list returned, add an indicator that there are more to list
                if ($fileData[IUtility::COUNT] > count($files)) {
                    $text = $text . "\n...";
                }
            }
        }
        return $text;
    }

    /**
     * @inheritDoc
     */
    public function buildTitle(Change $change, Review $review = null, Activity $activity = null): string
    {
        $user     = $change->getUser();
        $userDao  = $this->services->get(IDao::USER_DAO);
        $title    = '';
        $fullName = $userDao->fetchById($user)->getRawValue(IUser::FIELD_FULLNAME);

        if ($review !== null) {
            $title = sprintf("%s (%s) requested review #%s", $fullName, $user,  $review->getId());
        } else {
            if ($activity && $activity->get('behalfOf')) {
                $activityUser     = $activity->get('user');
                $activityFullName = $userDao->fetchById($activityUser)->getRawValue(IUser::FIELD_FULLNAME);

                $title = sprintf(
                    "%s (%s) committed change #%s on behalf of %s (%s)",
                    $activityFullName,
                    $activityUser,
                    $change->getId(),
                    $fullName,
                    $user
                );
            } else {
                $title = sprintf("%s (%s) committed change #%s", $fullName, $user, $change->getId());
            }
        }
        return $title;
    }

    /**
     * @inheritDoc
     */
    public function buildLink(Change $change, Review $review = null): string
    {
        $basePath = "/".(P4_SERVER_ID ? P4_SERVER_ID. "/" :  '');
        return $basePath . ($review ? "reviews/".$review->getId() : "changes/".$change->getId());
    }

    /**
     * @inheritDoc
     */
    public function buildDescription(Change $change): string
    {
        $description   = $change->getDescription();
        $defaultFormat = "*Description*:\n{description}\n";
        return strtr($defaultFormat, ['{description}' => $description]);
    }
}
