<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace Reviews\Filter;

use Application\Factory\InvokableService;
use Application\I18n\TranslatorFactory;
use Application\InputFilter\DirectInput;
use Application\InputFilter\InputFilter;
use Application\Validator\GreaterThanInt;
use Interop\Container\ContainerInterface;
use Reviews\Validator\States;
use Reviews\ITransition;
use Application\Validator\IsBool;
use Application\Filter\FormBoolean;
use Application\Validator\IsStringOrNull;
use Laminas\Filter\StringTrim;

/**
 * Defines filters for transition.
 * @package Reviews\Filter
 */
class Transition extends InputFilter implements InvokableService
{
    private $translator;
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->translator = $services->get(TranslatorFactory::SERVICE);
        $this->addStateFilter();
        $this->addMaxRevisionFilter();
        $this->addCleanUpFilter();
        $this->addFixStatusFilter();
        $this->addJobsFilter();
        $this->addTextFilter();
    }

    /**
     * Add the state filter to validate the state being used are valid.
     */
    private function addStateFilter()
    {
        $input = new DirectInput(ITransition::TRANSITION);
        $input->setRequired(false);
        $input->getValidatorChain()
            ->attach(
                new States(
                    $this->translator,
                    [
                        States::VALID_STATES => ITransition::ALL_VALID_TRANSITIONS
                    ]
                )
            );
        $this->add($input);
    }

    /**
     * Adds a filter for maxRevision This is an optional field but if provided should be an integer greater than 0
     */
    private function addMaxRevisionFilter()
    {
        $input = new DirectInput(ITransition::MAX_REVISION);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => 0]));
        $this->add($input);
    }

    /**
     * Adds a filter for cleanUp to ensure the value is boolean or can be converted to boolean.
     */
    private function addCleanUpFilter()
    {
        $input = new DirectInput(ITransition::CLEAN_UP);
        $input->setRequired(true);
        $input->getFilterChain()->attach(new FormBoolean());
        $input->getValidatorChain()->attach(new IsBool());
        $this->add($input);
    }

    /**
     * Adds a filter for fixStatus to ensure a value is a string or null.
     */
    private function addFixStatusFilter()
    {
        $input = new DirectInput(ITransition::FIX_STATUS);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new IsStringOrNull());
        $this->add($input);
    }

    /**
     * Add a jobs filter. it is a nan mandatory array so here we just accept a value in the
     * filter, so it is returned by getValues
     */
    private function addJobsFilter()
    {
        $input = new DirectInput(ITransition::JOBS);
        $input->setRequired(false);
        $this->add($input);
    }

    /**
     * Trim the text and make it optional
     * @return void
     */
    private function addTextFilter()
    {
        $input = new DirectInput(ITransition::TEXT);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new StringTrim());
        $this->add($input);
    }
}
