<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */
namespace Projects\Checker;

use Application\Checker;
use Application\Config\ConfigManager as Cm;
use Application\Config\IConfigDefinition as IDef;
use Application\Config\ConfigException;
use Application\Config\Services;
use Application\Permissions\Exception\ForbiddenException;
use Application\Permissions\IPermissions;
use InvalidArgumentException;

/**
 * Checker for project functions
 */
class Project extends Checker
{
    /**
     * Check a project action
     * @throws ConfigException
     * @see Checker
     */
    public function check(string $check, array $options = null): bool
    {
        switch ($check) {
            case IProject::PROJECT_ADD:
                $this->checkAddProject();
                break;
            case IProject::PROJECT_EDIT:
                $this->checkEditProject($options);
                break;
            case IProject::PROJECT_NAME_ADMIN_ONLY:
                $this->checkAdminForConfigSetting(IDef::PROJECTS_EDIT_NAME_ADMIN_ONLY);
                break;
            case IProject::BRANCHES_ADMIN_ONLY:
                $this->checkAdminForConfigSetting(IDef::PROJECTS_EDIT_BRANCHES_ADMIN_ONLY);
                break;
            default:
                throw new InvalidArgumentException(sprintf("Check '%s' is not supported", $check));
        }
        return true;
    }

    /**
     * Check the add project action. Checks the admin only and groups only configuration values to determine if
     * add project is allowed based on the user role and their group membership
     * @throws ConfigException
     */
    protected function checkAddProject()
    {
        $configCheck = $this->services->get(Services::CONFIG_CHECK);
        $config      = $this->services->get(IDef::CONFIG);
        $adminOnly   = Cm::getValue($config, IDef::PROJECTS_ADD_ADMIN_ONLY, false);
        if ($adminOnly) {
            $configCheck->check(IPermissions::ADMIN_CHECKER);
        }
        $addProjectGroups = Cm::getValue($config, IDef::PROJECTS_ADD_GROUPS_ONLY, []);
        if ($addProjectGroups) {
            $checker                   = IPermissions::MEMBER_CHECKER;
            $checker[Checker::OPTIONS] = [self::VALUE => $addProjectGroups];
            $configCheck->check($checker);
        }
    }

    /**
     * Check the edit project action. Checks that a user is either an owner, or, where there are no owners, a
     * member (user/group/project)
     * @throws ConfigException
     */
    protected function checkEditProject($options)
    {
        $project     = $options[Checker::VALUE];
        $configCheck = $this->services->get(Services::CONFIG_CHECK);
        try {
            $configCheck->check(IPermissions::ADMIN_CHECKER);
            return true;
        } catch (ForbiddenException $exception) {
            $configCheck->check(
                ($project->hasOwners() ? IPermissions::OWNER_CHECKER : IPermissions::MEMBER_CHECKER) +
                [Checker::OPTIONS => [Checker::VALUE => $project]]
            );
        }
    }
}
