<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace Application;

use Application\Config\ConfigException;
use Application\Config\ConfigManager as Cm;
use Application\Config\IConfigDefinition as IDef;
use Application\Config\Services;
use Application\Permissions\Exception\Exception as PermissionsException;
use Application\Permissions\IPermissions;
use Interop\Container\ContainerInterface;
use Application\Factory\InvokableService;

/**
 * Class Checker. Abstract checker for subclassing
 * @package Application
 */
abstract class Checker implements InvokableService
{
    const CHECKERS     = 'checkers';
    const NAME         = 'name';
    const VALUE        = 'value';
    const OPTIONS      = 'options';
    const RETURN_VALUE = 'returnValue';

    protected $services;

    /**
     * @inheritDoc
     */
    public function __construct(?ContainerInterface $services, ?array $options = null)
    {
        $this->services = $services;
    }

    /**
     * Carry out a check
     * @param string            $check      name of the check
     * @param array|null        $options    optional data to assist the check
     */
    abstract protected function check(string $check, array $options = null);

    /**
     * Gets the configuration setting specified and if it is true performs an admin check for the current user. If the
     * admin check is performed and fails the exception is thrown.
     *
     * @return void
     * @throws ConfigException|PermissionsException
     *
     */
    protected function checkAdminForConfigSetting(string $configSetting)
    {
        $config      = $this->services->get(IDef::CONFIG);
        $configValue = Cm::getValue($config, $configSetting, false);
        if ($configValue) {
            try {
                $configCheck = $this->services->get(Services::CONFIG_CHECK);
                $configCheck->check(IPermissions::ADMIN_CHECKER);
            } catch (PermissionsException $e) {
                // return new exception
                throw $e;
            }
        }
    }
}
