<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace AiAnalysis\Service;

use AiAnalysis\Filter\IAiAnalysis;
use AiAnalysis\Helper\IAiAnalysisHelper;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Log\SwarmLogger;
use Laminas\Http\Response;
use Laminas\Http\Client\Adapter\Exception\RuntimeException;
use Laminas\Http\Client\Adapter\Exception\TimeoutException;

class LMStudioAIAdapter extends AbstractAiAdapter implements AiServiceInterface
{

    const LOG_PREFIX = LMStudioAIAdapter::class;

    /**
     * This will prepare the request as Open AI format
     * @param mixed $data
     * @return array
     */
    public function executeAIRequest(mixed $data): array
    {
        $logger = $this->services->get(SwarmLogger::SERVICE);
        try {
            $fileName    = ($data[IAiAnalysis::FILE_NAME]) ?
                'of the file: ' . $data[IAiAnalysis::FILE_NAME] . ' ' : ' ';
            $content     = $data[IConfigDefinition::AI_PACKAGE_TYPE] . $fileName .
                $data[IAiAnalysisHelper::CONTENT_TO_ANALYZE];
            $requestBody = ['messages' => [
                [
                    'role' => 'user',
                    'content' => $content
                ],
            ]];
            $result      = $this->request($this->getApiUrl(), false, $requestBody);
            if ($this->validateResult($result)) {
                $logger->trace(sprintf("[%s]: Content generation successful ", self::LOG_PREFIX));
                return [IAiAnalysisHelper::CONTENT_GENERATED => $result->choices[0]->message->content,
                    self::ERROR => null,
                    self::CODE => Response::STATUS_CODE_200];
            } else {
                $logger->trace(sprintf("[%s]: Failed at content generation ", self::LOG_PREFIX));
                return [
                    IAiAnalysisHelper::CONTENT_GENERATED => '',
                    self::ERROR => is_object($result) && property_exists($result, 'error') ? $result->error :
                        'Response not received from AI Vendor',
                    self::CODE => is_object($result) && property_exists($result, 'code') ? $result->code :
                        Response::STATUS_CODE_500
                ];
            }
        } catch (TimeoutException $exception) {
            $logger->debug(
                sprintf(
                    "[%s]: Error occurred at LM Studio AI Adapter %s",
                    self::LOG_PREFIX,
                    $exception->getMessage()
                )
            );
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                'error' => 'Request to AI vendor timed out.',
                'code' => Response::STATUS_CODE_504
            ];
        } catch (RuntimeException $exception) {
            $logger->debug(
                sprintf(
                    "[%s]: Error occurred at LM Studio AI Adapter %s",
                    self::LOG_PREFIX,
                    $exception->getMessage()
                )
            );
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                'error' => 'Unable to connect, connection refused',
                'code' => Response::STATUS_CODE_502
            ];
        } catch (ConfigException $exception) {
            $logger->debug(
                sprintf(
                    "[%s]: Error occurred at LM Studio AI Adapter %s",
                    self::LOG_PREFIX,
                    $exception->getMessage()
                )
            );
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                'error' => 'Config issue: ' . $exception->getMessage(),
                'code' => Response::STATUS_CODE_500
            ];
        } catch (\Exception | \Throwable $exception) {
            $logger->debug(
                sprintf(
                    "[%s]: Error occurred at LM Studio AI Adapter %s",
                    self::LOG_PREFIX,
                    $exception->getMessage()
                )
            );
            $statusCode = $exception->getCode();
            // Ensure only valid HTTP status codes (100-599) are used
            if (!is_int($statusCode) || $statusCode < 100 || $statusCode > 599) {
                $statusCode = Response::STATUS_CODE_500;
            }
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                'error' => $exception->getMessage(),
                'code' => $statusCode,
            ];
        }
    }
}
