<?php

/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace AiAnalysis\Model;

use Application\Model\ServicesModelTrait;
use Application\Permissions\Protections;
use Comments\Filter\ContextAttributes;
use P4\Connection\ConnectionInterface as Connection;
use P4\Model\Fielded\Iterator as ModelIterator;
use Record\Exception\NotFoundException as RecordNotFoundException;
use Record\Exception\Exception;
use Record\Key\AbstractKey as KeyRecord;

/**
 * Provides persistent storage and indexing of generated AI summary .
 */
class AiReviewSummary extends KeyRecord implements IAiReviewSummary
{
    use ServicesModelTrait;
    const KEY_PREFIX  = 'swarm-ai-review-summary-';
    const KEY_COUNT   = 'swarm-ai-review-summary:count';
    const COUNT_INDEX = 1101;

    protected $fields = [
        self::TOPIC     => [               // object being commented on (e.g. changes/1)
                                       'index'     => 1102             // note we index by topic for later retrieval
        ],
        self::CONTEXT   => [               // specific context e.g.:
                                       'accessor'  => 'getContext',// {file: '//depot/foo', leftLine: 85,
                                                                   // rightLine: null}
                                       'mutator'   => 'setContext'
        ],
        self::FLAGS     => [               // list of flags
                                       'accessor'  => 'getFlags',
                                       'mutator'   => 'setFlags'
        ],
        self::TIME,                             // timestamp when the AI summary was created
        self::UPDATED,                          // timestamp when the AI summary was last updated
        self::EDITED,                           // timestamp when the AI summary was added as comment body
    ];

    public function getContext(): array
    {
        return (array) $this->getRawValue(self::CONTEXT);
    }

    /**
     * Gets whether the context contains the value in the 'attribute'
     * @param mixed $attribute the attribute
     * @return bool true if the context has the attribute
     */
    public function hasAttribute($attribute): bool
    {
        $context = $this->getContext();
        return isset($context[ContextAttributes::ATTRIBUTE]) && $context[ContextAttributes::ATTRIBUTE] === $attribute;
    }

     /**
     * Returns an array of flags set on this comment.
     * @return  array   flag names for all flags current set on this comment
     */
    public function getFlags(): array
    {
        return (array) $this->getRawValue(self::FLAGS);
    }

    /**
     * Set an array of active flag names on this comment.
     *
     * @param   array|null  $flags    an array of flags or null
     * @return  AiReviewSummary    to maintain a fluent interface
     */
    public function setFlags(array $flags = null): KeyRecord
    {
        // only grab unique flags, and reset the index
        $flags = array_values(array_unique((array) $flags));

        return $this->setRawValue(self::FLAGS, $flags);
    }

    /**
     * Adds the flags names in passed array to the existing flags on this data
     *
     * @param array|string|null $flags an array of flags to add, an individual flag string or null
     * @return KeyRecord|AiReviewSummary to maintain a fluent interface
     */
    public function addFlags(array|string $flags = null): KeyRecord|AiReviewSummary
    {
        $flags = array_merge($this->getFlags(), (array) $flags);
        return $this->setFlags($flags);
    }

    /**
     * Retrieves all records that match the passed options.
     * Extends parent to compose a search query when fetching by topic.
     *
     * @param   array               $options       an optional array of search conditions and/or options
     *                                             supported options are:
     *                                             FETCH_MAXIMUM - set to integer value to limit to the first
     *                                                             'max' number of entries.
     *                                             FETCH_AFTER - set to an id _after_ which we start collecting
     *                                             FETCH_BY_TOPIC - set to a 'topic' id to limit results
     *                                             FETCH_BY_IDS - provide an array of ids to fetch.
     *                                                            not compatible with FETCH_SEARCH or FETCH_AFTER.
     *                                             FETCH_BY_READ_BY - includes comments that have been read by the users
     *                                             FETCH_BY UNREAD_BY - includes comments that have not been read by the
     *                                                                  users
     * @param   Connection          $p4            the perforce connection to use
     * @param   Protections|null    $protections   optional - if set, comments associated with files the user cannot
     *                                             read according to given protections will be removed before returning
     * @return  ModelIterator                      the list of zero or more matching activity objects
     * @throws \Exception
     */
    public static function fetchAll(array $options, Connection $p4, Protections $protections = null): ModelIterator
    {
        // normalize options
        $options += [
            static::FETCH_BY_TOPIC        => null,
            static::FETCH_BY_CONTEXT      => null,
            static::FETCH_BY_USER         => null
        ];

        // build a search expression for topic.
        $options[static::FETCH_SEARCH] = static::makeSearchExpression(
            [self::TOPIC => $options[ static::FETCH_BY_TOPIC]]
        );
        return parent::fetchAll($options, $p4);
    }

    /**
     * Retrieves all records that match the passed options.
     * Extends parent to compose a search query when fetching by updated.
     *
     * @param   Connection          $p4            the perforce connection to use
     * @param   array               $options       an optional array of search conditions and/or options
     *                                             supported options are:
     *                                             FETCH_MAXIMUM - set to integer value to limit to the first
     *                                                             'max' number of entries.
     * @return  ModelIterator                      the list of zero or more matching activity objects
     * @throws \Exception
     */
    public static function fetchRetainedSummaries(
        Connection $p4,
        array $options = []
    ): ModelIterator {
        return parent::fetchAll($options, $p4);
    }

    /**
     * Saves the records values and updates indexes as needed.
     * Extends the basic save behavior to also:
     * - set timestamp to current time if one isn't already set
     * - remove existing count indices for this topic and add a current one
     *
     * @return  AiReviewSummary   to maintain a fluent interface
     * @throws Exception if no topic is set or an id is present but the record was not fetched
     * @throws \Exception
     */
    public function save(): static
    {
        if (!strlen($this->get(self::TOPIC))) {
            throw new Exception('Cannot save, no topic has been set.');
        }

        // always set update time to now
        $this->set(self::UPDATED, time());

        // if no time is already set, use now as a default
        $this->set(self::TIME, $this->get(self::TIME) ?: $this->get(self::UPDATED));

        // let parent actually save before we go about indexing
        parent::save();

        $this->updateCountIndex();

        return $this;
    }

    /**
     * Delete this AiReviewSummary record.
     * Extends parent to update topic count index.
     *
     * @return AiReviewSummary  provides fluent interface
     * @throws \Exception
     */
    public function delete(): AiReviewSummary
    {
        parent::delete();
        $this->updateCountIndex();

        return $this;
    }

    /**
     * For each topic we maintain an index of the number of
     * aiSummaries in that topic. This makes it easy to fetch the number of
     * aiSummaries in arbitrary topics with one call to p4 search.
     *
     * @return AiReviewSummary  provides fluent interface
     * @throws \Exception
     */
    protected function updateCountIndex(): AiReviewSummary
    {
        // retrieve all existing indices for this topic count and delete them
        $p4      = $this->getConnection();
        $topic   = static::encodeIndexValue($this->get(self::TOPIC));
        $query   = static::COUNT_INDEX . '=' . $topic;
        $indices = $p4->run('search', $query)->getData();
        foreach ($indices as $index) {
            $p4->run(
                'index',
                ['-a', static::COUNT_INDEX, '-d', $index],
                $topic
            );
        }

        // read out all comments for this topic so we can get the new counts
        // we try and do this as close to writing out the index to improve
        // our chances of getting an accurate count.
        $aiReviewSummaries = static::fetchAll(
            [static::FETCH_BY_TOPIC => $this->get(self::TOPIC)],
            $p4
        );

        // early exit if no ai review summaries (zero count)
        if (!count($aiReviewSummaries)) {
            return $this;
        }

        // write out our current count to the index.
        // we include the encoded topic id in the key so that we can tell
        // which topic each count is for when searching for multiple topics.
        $p4->run(
            'index',
            ['-a', static::COUNT_INDEX, $topic],
            $topic
        );

        return $this;
    }

    /**
     * @inheritDoc
     * Override for public access
     */
    public static function encodeIndexValue($value): string
    {
        return parent::encodeIndexValue($value);
    }

    /**
     * @inheritDoc
     * Override for public access
     */
    public static function decodeIndexValue($value): string
    {
        return parent::decodeIndexValue($value);
    }

    /**
     * Fetch a comment by its id
     * @param mixed         $id     the id
     * @param Connection    $p4     the connection
     * @return KeyRecord
     * @throws RecordNotFoundException
     */
    public static function fetchById($id, Connection $p4): KeyRecord
    {
        return AiReviewSummary::fetch($id, $p4);
    }

    /**
     * This function will create the new record for ai review summary
     * @param array $filterData
     * @param Connection $p4
     * @return AiReviewSummary
     */
    public static function createAiReviewSummary(array $filterData, Connection $p4): AiReviewSummary
    {
        $aiReviewSummaryModel = new static($p4);
        return $aiReviewSummaryModel->set($filterData);
    }
}
