<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */
namespace Projects\Validator;

use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Validator\AbstractValidator;
use Application\Validator\ConnectedAbstractValidator;
use Record\Exception\NotFoundException;

/**
 * Validator for the workflow field on projects and branches
 */
class Workflow extends AbstractValidator
{
    const CHECK_WORKFLOW_PERMISSION = 'checkWorkflowPermission';
    private $p4;
    private $p4User;
    private $services;
    private $checkWorkflowPermission;
    const NOT_FOUND             = 'notFound';
    protected $messageTemplates = [];
    protected $validateOwner    = true;

    /**
     * Construct with options
     * - connection: required connection to look up workflow
     * - services: required services
     * - p4_user: optional user to test permission, will default to ConnectionFactory::P4_USER fron services
     * - checkWorkflowPermission: should permission be checked if the workflow is not shared?
     * @param $options
     */
    public function __construct($options = null)
    {
        $this->services                = $options[self::SERVICES];
        $this->p4                      = $options[ConnectedAbstractValidator::CONNECTION]
            ?? $this->services->get(ConnectionFactory::P4_ADMIN);
        $this->p4User                  = $options[ConnectionFactory::P4_USER]
            ?? $this->services->get(ConnectionFactory::P4_USER);
        $this->checkWorkflowPermission = $options[self::CHECK_WORKFLOW_PERMISSION] ?? true;
        $this->messageTemplates        = [
            self::NOT_FOUND => $this->tGen("Workflow with id '%value%' was not found.")
        ];
        $this->validateOwner           = $options["validateWorkflowOwner"] ?? true;
        parent::__construct($options);
    }

    /**
     * Validate workflow. If invalid will set an error message:
     *     'notFound' => "Workflow with id '%value%' was not found."
     * An invalid result can occur for any of these conditions:
     * - a workflow with the id is not found
     * - the workflow is found, it is not shared, we are not checking permission
     * - the workflow is found, it is not shared, we are checking permission and the p4User is not the owner
     * @param mixed $value workflow value (id)
     * @return bool
     */
    public function isValid($value): bool
    {
        $valid = true;
        if (!empty($value)) {
            $wfDao = $this->services->get(IDao::WORKFLOW_DAO);
            try {
                $matchingWorkflow = $wfDao->fetch($value, $this->p4);
                if (!$this->p4User->isSuperUser() &&
                    !$matchingWorkflow->isShared()
                    && ($this->checkWorkflowPermission === true ||
                            (is_array($this->checkWorkflowPermission) &&
                                in_array($matchingWorkflow->getId(), $this->checkWorkflowPermission)
                            )
                        )
                    && $this->validateOwner && !$matchingWorkflow->isOwner($this->p4User->getUser(), $this->p4)) {
                    $valid = false;
                }
            } catch (NotFoundException $e) {
                // Workflow id does not exist at all
                $valid = false;
            }
        }
        if (!$valid) {
            $this->error(self::NOT_FOUND, $value);
        }
        return $valid;
    }
}
