<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */
namespace Attachments\Model;

use Application\Factory\InvokableService;
use P4\Connection\ConnectionInterface;
use P4\Connection\Exception\CommandException;
use P4\File\Exception\Exception as P4FileException;
use Record\Exception\NotFoundException;
use Record\Key\AbstractKey;

interface IAttachmentDAO extends InvokableService
{
    // Field definitions for key values that are set on $_FILES for file upload
    const FILE     = 'file';
    const TMP_NAME = 'tmp_name';

    const LOCK_ATTACHMENT_PREFIX = 'attachment-comment-';

    const THUMBNAIL_SIZE = 240;
    // File attribute name for thumbnails
    const THUMB_ATTRIBUTE = 'thumb';

    /**
     * Fetches an attachment based on the id and the file. The file value is expected to be an encoded value that when
     * decoded should match the 'name' value on the attachment. In order for a NotFoundException to not be thrown the
     * following criteria must be met:
     *      - id must match an attachment in key data
     *      - decoded file must match the 'name' value from the attachment
     *      - the depot file from the attachment must exist
     * @param mixed                     $id         the attachment id
     * @param string                    $file       the encoded file name (compared to 'name' from the attachment)
     * @param ConnectionInterface|null  $connection connection to use
     * @return AbstractKey|callable the attachment
     * @throws CommandException
     * @throws NotFoundException
     * @throws P4FileException
     */
    public function fetchAttachment($id, string $file, ConnectionInterface $connection);

    /**
     * Create attachment key data and save the file to the depot. On completion the temporary file
     * referenced in the data 'tmp_name' will be deleted.
     * A clean-up operation is placed in the task queue for processing 24 hours later so that if the comment
     * associated with an attachment is not posted the attachment will be cleaned up
     * @param array $data   data containing file information
     */
    public function createAttachment(array $data) : Attachment;

    /**
     * Gets the thumbnail from the attachment if it exists.
     * The file value is expected to be an encoded value that when decoded should match the 'name' value on the
     * attachment. In order for a NotFoundException to not be thrown the following criteria must be met:
     *      - id must match an attachment in key data
     *      - decoded file must match the 'name' value from the attachment
     *      - the depot file from the attachment must exist
     * @param mixed                     $id         the attachment id
     * @param string                    $file       the encoded file name (compared to 'name' from the attachment)
     * @param ConnectionInterface|null  $connection connection to use
     * @return AbstractKey|callable the attachment thumbnail in base64 encoded format, or null if there is no thumbnail
     * @throws CommandException
     * @throws NotFoundException
     * @throws P4FileException
     */
    public function getThumbnail($id, string $file, ConnectionInterface $connection);
}
