<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace Application\Filter;

use Application\Config\ConfigManager;

class Emojify extends ServiceAwareFilter
{
    private $services;
    private $emojiRoot = BASE_PATH;
    public $emoji      = [
        ':+1:'                              => '&#x1F44D;',
        ':-1:'                              => '&#x1F44E;',
        ':100:'                             => '&#x1F4AF;',
        ':1234:'                            => '&#x1F522;',
        ':8ball:'                           => '&#x1F3B1;',
        ':a:'                               => '&#x1F170;',
        ':ab:'                              => '&#x1F18E;',
        ':abc:'                             => '&#x1F524;',
        ':abcd:'                            => '&#x1F521;',
        ':accept:'                          => '&#x1F251;',
        ':aerial_tramway:'                  => '&#x1F6A1;',
        ':airplane:'                        => '&#x2708;',
        ':alarm_clock:'                     => '&#x23F0;',
        ':alien:'                           => '&#x1F47D;',
        ':ambulance:'                       => '&#x1F691;',
        ':anchor:'                          => '&#x2693;',
        ':angel:'                           => '&#x1F47C;',
        ':anger:'                           => '&#x1F4A2;',
        ':angry:'                           => '&#x1F620;',
        ':anguished:'                       => '&#x1F627;',
        ':ant:'                             => '&#x1F41C;',
        ':apple:'                           => '&#x1F34E;',
        ':aquarius:'                        => '&#x2652;',
        ':aries:'                           => '&#x2648;',
        ':arrow_backward:'                  => '&#x25C0;',
        ':arrow_double_down:'               => '&#x23EC;',
        ':arrow_double_up:'                 => '&#x23EB;',
        ':arrow_down:'                      => '&#x2B07;',
        ':arrow_down_small:'                => '&#x1F53D;',
        ':arrow_forward:'                   => '&#x25B6;',
        ':arrow_heading_down:'              => '&#x2935;',
        ':arrow_heading_up:'                => '&#x2934;',
        ':arrow_left:'                      => '&#x2B05;',
        ':arrow_lower_left:'                => '&#x2199;',
        ':arrow_lower_right:'               => '&#x2198;',
        ':arrow_right:'                     => '&#x27A1;',
        ':arrow_right_hook:'                => '&#x21AA;',
        ':arrow_up:'                        => '&#x2B06;',
        ':arrow_up_down:'                   => '&#x2195;',
        ':arrow_up_small:'                  => '&#x1F53C;',
        ':arrow_upper_left:'                => '&#x2196;',
        ':arrow_upper_right:'               => '&#x2197;',
        ':arrows_clockwise:'                => '&#x1F503;',
        ':arrows_counterclockwise:'         => '&#x1F504;',
        ':art:'                             => '&#x1F3A8;',
        ':articulated_lorry:'               => '&#x1F69B;',
        ':astonished:'                      => '&#x1F632;',
        ':atm:'                             => '&#x1F3E7;',
        ':b:'                               => '&#x1F171;',
        ':baby:'                            => '&#x1F476;',
        ':baby_bottle:'                     => '&#x1F37C;',
        ':baby_chick:'                      => '&#x1F424;',
        ':baby_symbol:'                     => '&#x1F6BC;',
        ':baggage_claim:'                   => '&#x1F6C4;',
        ':balloon:'                         => '&#x1F388;',
        ':ballot_box_with_check:'           => '&#x2611;',
        ':bamboo:'                          => '&#x1F38D;',
        ':banana:'                          => '&#x1F34C;',
        ':bangbang:'                        => '&#x203C;',
        ':bank:'                            => '&#x1F3E6;',
        ':bar_chart:'                       => '&#x1F4CA;',
        ':barber:'                          => '&#x1F488;',
        ':baseball:'                        => '&#x26BE;',
        ':basketball:'                      => '&#x1F3C0;',
        ':bath:'                            => '&#x1F6C0;',
        ':bathtub:'                         => '&#x1F6C1;',
        ':battery:'                         => '&#x1F50B;',
        ':bear:'                            => '&#x1F43B;',
        ':beer:'                            => '&#x1F37A;',
        ':beers:'                           => '&#x1F37B;',
        ':beetle:'                          => '&#x1F41E;',
        ':beginner:'                        => '&#x1F530;',
        ':bell:'                            => '&#x1F514;',
        ':bento:'                           => '&#x1F371;',
        ':bicyclist:'                       => '&#x1F6B4;',
        ':bike:'                            => '&#x1F6B2;',
        ':bikini:'                          => '&#x1F459;',
        ':bird:'                            => '&#x1F426;',
        ':birthday:'                        => '&#x1F382;',
        ':black_circle:'                    => '&#x26AB;',
        ':black_joker:'                     => '&#x1F0CF;',
        ':black_nib:'                       => '&#x2712;',
        ':black_square_button:'             => '&#x1F532;',
        ':blossum:'                         => '&#x1F33C;',
        ':blowfish:'                        => '&#x1F421;',
        ':blue_book:'                       => '&#x1F4D8;',
        ':blue_car:'                        => '&#x1F699;',
        ':blue_heart:'                      => '&#x1F499;',
        ':blush:'                           => '&#x1F60A;',
        ':boar:'                            => '&#x1F417;',
        ':boat:'                            => '&#x26F5;',
        ':bomb:'                            => '&#x1F4A3;',
        ':bookmark:'                        => '&#x1F516;',
        ':bookmark_tabs:'                   => '&#x1F4D1;',
        ':books:'                           => '&#x1F4DA;',
        ':boom:'                            => '&#x1F4A5;',
        ':boot:'                            => '&#x1F462;',
        ':bouquet:'                         => '&#x1F490;',
        ':bow:'                             => '&#x1F647;',
        ':bowling:'                         => '&#x1F3B3;',
        ':boy:'                             => '&#x1F466;',
        ':bread:'                           => '&#x1F35E;',
        ':bride_with_veil:'                 => '&#x1F470;',
        ':bridge_at_night:'                 => '&#x1F309;',
        ':briefcase:'                       => '&#x1F4BC;',
        ':broken_heart:'                    => '&#x1F494;',
        ':bug:'                             => '&#x1F41B;',
        ':bulb:'                            => '&#x1F4A1;',
        ':bullettrain_front:'               => '&#x1F685;',
        ':bullettrain_side:'                => '&#x1F684;',
        ':bus:'                             => '&#x1F68C;',
        ':busstop:'                         => '&#x1F68F;',
        ':bust_in_silhouette:'              => '&#x1F464;',
        ':busts_in_silhouette:'             => '&#x1F465;',
        ':cactus:'                          => '&#x1F335;',
        ':cake:'                            => '&#x1F370;',
        ':calendar:'                        => '&#x1F4C6;',
        ':calling:'                         => '&#x1F4F2;',
        ':camel:'                           => '&#x1F42B;',
        ':camera:'                          => '&#x1F4F7;',
        ':cancer:'                          => '&#x264B;',
        ':candy:'                           => '&#x1F36C;',
        ':capital_abcd:'                    => '&#x1F520;',
        ':capricorn:'                       => '&#x2651;',
        ':car:'                             => '&#x1F697;',
        ':card_index:'                      => '&#x1F4C7;',
        ':carousel_horse:'                  => '&#x1F3A0;',
        ':cat2:'                            => '&#x1F408;',
        ':cat:'                             => '&#x1F431;',
        ':cd:'                              => '&#x1F4BF;',
        ':chart:'                           => '&#x1F4B9;',
        ':chart_with_downwards_trend:'      => '&#x1F4C9;',
        ':chart_with_upwards_trend:'        => '&#x1F4C8;',
        ':checkered_flag:'                  => '&#x1F3C1;',
        ':cherries:'                        => '&#x1F352;',
        ':cherry_blossom:'                  => '&#x1F338;',
        ':chestnut:'                        => '&#x1F330;',
        ':chicken:'                         => '&#x1F414;',
        ':children_crossing:'               => '&#x1F6B8;',
        ':chocolate_bar:'                   => '&#x1F36B;',
        ':christmas_tree:'                  => '&#x1F384;',
        ':church:'                          => '&#x26EA;',
        ':cinema:'                          => '&#x1F3A6;',
        ':circus_tent:'                     => '&#x1F3AA;',
        ':city_sunrise:'                    => '&#x1F307;',
        ':city_sunset:'                     => '&#x1F306;',
        ':cl:'                              => '&#x1F191;',
        ':clap:'                            => '&#x1F44F;',
        ':clapper:'                         => '&#x1F3AC;',
        ':clipboard:'                       => '&#x1F4CB;',
        ':clock1030:'                       => '&#x1F565;',
        ':clock10:'                         => '&#x1F559;',
        ':clock1130:'                       => '&#x1F566;',
        ':clock11:'                         => '&#x1F55A;',
        ':clock1230:'                       => '&#x1F567;',
        ':clock12:'                         => '&#x1F55B;',
        ':clock130:'                        => '&#x1F55C;',
        ':clock1:'                          => '&#x1F550;',
        ':clock230:'                        => '&#x1F55D;',
        ':clock2:'                          => '&#x1F551;',
        ':clock330:'                        => '&#x1F55E;',
        ':clock3:'                          => '&#x1F552;',
        ':clock430:'                        => '&#x1F55F;',
        ':clock4:'                          => '&#x1F553;',
        ':clock530:'                        => '&#x1F560;',
        ':clock5:'                          => '&#x1F554;',
        ':clock630:'                        => '&#x1F561;',
        ':clock6:'                          => '&#x1F555;',
        ':clock730:'                        => '&#x1F562;',
        ':clock7:'                          => '&#x1F556;',
        ':clock830:'                        => '&#x1F563;',
        ':clock8:'                          => '&#x1F557;',
        ':clock930:'                        => '&#x1F564;',
        ':clock9:'                          => '&#x1F558;',
        ':closed_book:'                     => '&#x1F4D5;',
        ':closed_lock_with_key:'            => '&#x1F510;',
        ':closed_umbrella:'                 => '&#x1F302;',
        ':cloud:'                           => '&#x2601;',
        ':clubs:'                           => '&#x2663;',
        ':cn:'                              => '&#x1F1E8;&#x1F1F3;',
        ':cocktail:'                        => '&#x1F378;',
        ':coffee:'                          => '&#x2615;',
        ':cold_sweat:'                      => '&#x1F630;',
        ':collision:'                       => '&#x1F4A5;',
        ':computer:'                        => '&#x1F4BB;',
        ':confetti_ball:'                   => '&#x1F38A;',
        ':confounded:'                      => '&#x1F616;',
        ':confused:'                        => '&#x1F615;',
        ':congratulations:'                 => '&#x3297;',
        ':construction:'                    => '&#x1F6A7;',
        ':construction_worker:'             => '&#x1F477;',
        ':convenience_store:'               => '&#x1F3EA;',
        ':cookie:'                          => '&#x1F36A;',
        ':cool:'                            => '&#x1F192;',
        ':cop:'                             => '&#x1F46E;',
        ':copyright:'                       => '&#x00A9;',
        ':corn:'                            => '&#x1F33D;',
        ':couple:'                          => '&#x1F46B;',
        ':couple_with_heart:'               => '&#x1F491;',
        ':couplekiss:'                      => '&#x1F48F;',
        ':cow2:'                            => '&#x1F404;',
        ':cow:'                             => '&#x1F42E;',
        ':credit_card:'                     => '&#x1F4B3;',
        ':crocodile:'                       => '&#x1F40A;',
        ':crossed_flags:'                   => '&#x1F38C;',
        ':crown:'                           => '&#x1F451;',
        ':cry:'                             => '&#x1F622;',
        ':crying_cat_face:'                 => '&#x1F63F;',
        ':crystal_ball:'                    => '&#x1F52E;',
        ':cupid:'                           => '&#x1F498;',
        ':curly_loop:'                      => '&#x27B0;',
        ':currency_exchange:'               => '&#x1F4B1;',
        ':curry:'                           => '&#x1F35B;',
        ':custard:'                         => '&#x1F36E;',
        ':customs:'                         => '&#x1F6C3;',
        ':cyclone:'                         => '&#x1F300;',
        ':dancer:'                          => '&#x1F483;',
        ':dancers:'                         => '&#x1F46F;',
        ':dango:'                           => '&#x1F361;',
        ':dart:'                            => '&#x1F3AF;',
        ':dash:'                            => '&#x1F4A8;',
        ':date:'                            => '&#x1F4C5;',
        ':de:'                              => '&#x1F1E9;&#x1F1EA;',
        ':deciduous_tree:'                  => '&#x1F333;',
        ':department_store:'                => '&#x1F3EC;',
        ':diamond_shape_with_a_dot_inside:' => '&#x1F4A0;',
        ':diamonds:'                        => '&#x2666;',
        ':disappointed:'                    => '&#x1F61E;',
        ':dizzy:'                           => '&#x1F4AB;',
        ':dizzy_face:'                      => '&#x1F635;',
        ':do_not_litter:'                   => '&#x1F6AF;',
        ':dog2:'                            => '&#x1F415;',
        ':dog:'                             => '&#x1F436;',
        ':dollar:'                          => '&#x1F4B5;',
        ':dolls:'                           => '&#x1F38E;',
        ':dolphin:'                         => '&#x1F42C;',
        ':door:'                            => '&#x1F6AA;',
        ':doughnut:'                        => '&#x1F369;',
        ':dragon:'                          => '&#x1F409;',
        ':dragon_face:'                     => '&#x1F432;',
        ':dress:'                           => '&#x1F457;',
        ':dromedary_camel:'                 => '&#x1F42A;',
        ':droplet:'                         => '&#x1F4A7;',
        ':dvd:'                             => '&#x1F4C0;',
        ':e-mail:'                          => '&#x1F4E7;',
        ':ear:'                             => '&#x1F442;',
        ':ear_of_rice:'                     => '&#x1F33E;',
        ':earth_africa:'                    => '&#x1F30D;',
        ':earth_americas:'                  => '&#x1F30E;',
        ':earth_asia:'                      => '&#x1F30F;',
        ':egg:'                             => '&#x1F373;',
        ':eggplant:'                        => '&#x1F346;',
        ':eight:'                           => '&#x0038;',
        ':eight_pointed_black_star:'        => '&#x2734;',
        ':eight_spoked_asterisk:'           => '&#x2733;',
        ':electric_plug:'                   => '&#x1F50C;',
        ':elephant:'                        => '&#x1F418;',
        ':email:'                           => '&#x2709;',
        ':end:'                             => '&#x1F51A;',
        ':envelope:'                        => '&#x2709;',
        ':es:'                              => '&#x1F1EA;&#x1F1F8;',
        ':euro:'                            => '&#x1F4B6;',
        ':european_castle:'                 => '&#x1F3F0;',
        ':european_post_office:'            => '&#x1F3E4;',
        ':evergreen_tree:'                  => '&#x1F332;',
        ':exclamation:'                     => '&#x2757;',
        ':expressionless:'                  => '&#x1F611;',
        ':eyeglasses:'                      => '&#x1F453;',
        ':eyes:'                            => '&#x1F440;',
        ':facepunch:'                       => '&#x1F44A;',
        ':factory:'                         => '&#x1F3ED;',
        ':fallen_leaf:'                     => '&#x1F342;',
        ':family:'                          => '&#x1F46A;',
        ':fast_forward:'                    => '&#x23E9;',
        ':fax:'                             => '&#x1F4E0;',
        ':fearful:'                         => '&#x1F628;',
        ':feet:'                            => '&#x1F463;',
        ':ferris_wheel:'                    => '&#x1F3A1;',
        ':file_folder:'                     => '&#x1F4C1;',
        ':fire:'                            => '&#x1F525;',
        ':fire_engine:'                     => '&#x1F692;',
        ':fireworks:'                       => '&#x1F386;',
        ':first_quarter_moon:'              => '&#x1F313;',
        ':first_quarter_moon_with_face:'    => '&#x1F31B;',
        ':fish:'                            => '&#x1F41F;',
        ':fish_cake:'                       => '&#x1F365;',
        ':fishing_pole_and_fish:'           => '&#x1F3A3;',
        ':fist:'                            => '&#x270A;',
        ':five:'                            => '&#x0035;',
        ':flags:'                           => '&#x1F38F;',
        ':flashlight:'                      => '&#x1F526;',
        ':floppy_disk:'                     => '&#x1F4BE;',
        ':flower_playing_cards:'            => '&#x1F3B4;',
        ':flushed:'                         => '&#x1F633;',
        ':foggy:'                           => '&#x1F301;',
        ':football:'                        => '&#x1F3C8;',
        ':fork_and_knife:'                  => '&#x1F374;',
        ':fountain:'                        => '&#x26F2;',
        ':four:'                            => '&#x0034;',
        ':four_leaf_clover:'                => '&#x1F340;',
        ':fr:'                              => '&#x1F1EB;&#x1F1F7;',
        ':free:'                            => '&#x1F193;',
        ':fried_shrimp:'                    => '&#x1F364;',
        ':fries:'                           => '&#x1F35F;',
        ':frog:'                            => '&#x1F438;',
        ':frowning:'                        => '&#x1F626;',
        ':fuelpump:'                        => '&#x26FD;',
        ':full_moon:'                       => '&#x1F315;',
        ':full_moon_with_face:'             => '&#x1F31D;',
        ':game_die:'                        => '&#x1F3B2;',
        ':gb:'                              => '&#x1F1EC;&#x1F1E7;',
        ':gem:'                             => '&#x1F48E;',
        ':gemini:'                          => '&#x264A;',
        ':ghost:'                           => '&#x1F47B;',
        ':gift:'                            => '&#x1F381;',
        ':gift_heart:'                      => '&#x1F49D;',
        ':girl:'                            => '&#x1F467;',
        ':globe_with_meridians:'            => '&#x1F310;',
        ':goat:'                            => '&#x1F410;',
        ':golf:'                            => '&#x26F3;',
        ':grapes:'                          => '&#x1F347;',
        ':green_apple:'                     => '&#x1F34F;',
        ':green_book:'                      => '&#x1F4D7;',
        ':green_heart:'                     => '&#x1F49A;',
        ':grey_exclamation:'                => '&#x2755;',
        ':grey_question:'                   => '&#x2754;',
        ':grimacing:'                       => '&#x1F62C;',
        ':grin:'                            => '&#x1F601;',
        ':grinning:'                        => '&#x1F600;',
        ':guardsman:'                       => '&#x1F482;',
        ':guitar:'                          => '&#x1F3B8;',
        ':gun:'                             => '&#x1F52B;',
        ':haircut:'                         => '&#x1F487;',
        ':hamburger:'                       => '&#x1F354;',
        ':hammer:'                          => '&#x1F528;',
        ':hamster:'                         => '&#x1F439;',
        ':hand:'                            => '&#x270B;',
        ':handbag:'                         => '&#x1F45C;',
        ':hankey:'                          => '&#x1F4A9;',
        ':hash:'                            => '&#x0023;',
        ':hatched_chick:'                   => '&#x1F425;',
        ':hatching_chick:'                  => '&#x1F423;',
        ':headphones:'                      => '&#x1F3A7;',
        ':hear_no_evil:'                    => '&#x1F649;',
        ':heart:'                           => '&#x2764;',
        ':heart_decoration:'                => '&#x1F49F;',
        ':heart_eyes:'                      => '&#x1F60D;',
        ':heart_eyes_cat:'                  => '&#x1F63B;',
        ':heartbeat:'                       => '&#x1F493;',
        ':heartpulse:'                      => '&#x1F497;',
        ':hearts:'                          => '&#x2665;',
        ':heavy_check_mark:'                => '&#x2714;',
        ':heavy_division_sign:'             => '&#x2797;',
        ':heavy_dollar_sign:'               => '&#x1F4B2;',
        ':heavy_minus_sign:'                => '&#x2796;',
        ':heavy_multiplication_x:'          => '&#x2716;',
        ':heavy_plus_sign:'                 => '&#x2795;',
        ':helicopter:'                      => '&#x1F681;',
        ':herb:'                            => '&#x1F33F;',
        ':hibiscus:'                        => '&#x1F33A;',
        ':high_brightness:'                 => '&#x1F506;',
        ':high_heel:'                       => '&#x1F460;',
        ':hocho:'                           => '&#x1F52A;',
        ':honey_pot:'                       => '&#x1F36F;',
        ':honeybee:'                        => '&#x1F41D;',
        ':horse:'                           => '&#x1F434;',
        ':horse_racing:'                    => '&#x1F3C7;',
        ':hospital:'                        => '&#x1F3E5;',
        ':hotel:'                           => '&#x1F3E8;',
        ':hotsprings:'                      => '&#x2668;',
        ':hourglass:'                       => '&#x231B;',
        ':hourglass_flowing_sand:'          => '&#x23F3;',
        ':house:'                           => '&#x1F3E0;',
        ':house_with_garden:'               => '&#x1F3E1;',
        ':hushed:'                          => '&#x1F62F;',
        ':ice_cream:'                       => '&#x1F368;',
        ':icecream:'                        => '&#x1F366;',
        ':id:'                              => '&#x1F194;',
        ':ideograph_advantage:'             => '&#x1F250;',
        ':imp:'                             => '&#x1F47F;',
        ':inbox_tray:'                      => '&#x1F4E5;',
        ':incoming_envelope:'               => '&#x1F4E8;',
        ':information_desk_person:'         => '&#x1F481;',
        ':information_source:'              => '&#x2139;',
        ':innocent:'                        => '&#x1F607;',
        ':interrobang:'                     => '&#x2049;',
        ':iphone:'                          => '&#x1F4F1;',
        ':it:'                              => '&#x1F1EE;&#x1F1F9;',
        ':jack_o_lantern:'                  => '&#x1F383;',
        ':japan:'                           => '&#x1F5FE;',
        ':japanese_castle:'                 => '&#x1F3EF;',
        ':japanese_goblin:'                 => '&#x1F47A;',
        ':japanese_ogre:'                   => '&#x1F479;',
        ':jeans:'                           => '&#x1F456;',
        ':joy:'                             => '&#x1F602;',
        ':joy_cat:'                         => '&#x1F639;',
        ':jp:'                              => '&#x1F1EF;&#x1F1F5;',
        ':key:'                             => '&#x1F511;',
        ':keycap_ten:'                      => '&#x1F51F;',
        ':kimono:'                          => '&#x1F458;',
        ':kiss:'                            => '&#x1F48B;',
        ':kissing:'                         => '&#x1F617;',
        ':kissing_cat:'                     => '&#x1F63D;',
        ':kissing_closed_eyes:'             => '&#x1F61A;',
        ':kissing_heart:'                   => '&#x1F618;',
        ':kissing_smiling_eyes:'            => '&#x1F619;',
        ':koala:'                           => '&#x1F428;',
        ':koko:'                            => '&#x1F201;',
        ':kr:'                              => '&#x1F1F0;&#x1F1F7;',
        ':large_blue_circle:'               => '&#x1F535;',
        ':large_blue_diamond:'              => '&#x1F537;',
        ':large_orange_diamond:'            => '&#x1F536;',
        ':last_quarter_moon:'               => '&#x1F317;',
        ':last_quarter_moon_with_face:'     => '&#x1F31C;',
        ':laughing:'                        => '&#x1F606;',
        ':leaves:'                          => '&#x1F343;',
        ':ledger:'                          => '&#x1F4D2;',
        ':left_luggage:'                    => '&#x1F6C5;',
        ':left_right_arrow:'                => '&#x2194;',
        ':leftwards_arrow_with_hook:'       => '&#x21A9;',
        ':lemon:'                           => '&#x1F34B;',
        ':leo:'                             => '&#x264C;',
        ':leopard:'                         => '&#x1F406;',
        ':libra:'                           => '&#x264E;',
        ':light_rail:'                      => '&#x1F688;',
        ':link:'                            => '&#x1F517;',
        ':lips:'                            => '&#x1F444;',
        ':lipstick:'                        => '&#x1F484;',
        ':lock:'                            => '&#x1F512;',
        ':lock_with_ink_pen:'               => '&#x1F50F;',
        ':lollipop:'                        => '&#x1F36D;',
        ':loop:'                            => '&#x27BF;',
        ':loudspeaker:'                     => '&#x1F4E2;',
        ':love_hotel:'                      => '&#x1F3E9;',
        ':love_letter:'                     => '&#x1F48C;',
        ':low_brightness:'                  => '&#x1F505;',
        ':m:'                               => '&#x24C2;',
        ':mag:'                             => '&#x1F50D;',
        ':mag_right:'                       => '&#x1F50E;',
        ':mahjong:'                         => '&#x1F004;',
        ':mailbox:'                         => '&#x1F4EB;',
        ':mailbox_closed:'                  => '&#x1F4EA;',
        ':mailbox_with_mail:'               => '&#x1F4EC;',
        ':mailbox_with_no_mail:'            => '&#x1F4ED;',
        ':man:'                             => '&#x1F468;',
        ':man_with_gua_pi_mao:'             => '&#x1F472;',
        ':man_with_turban:'                 => '&#x1F473;',
        ':mans_shoe:'                       => '&#x1F45E;',
        ':maple_leaf:'                      => '&#x1F341;',
        ':mask:'                            => '&#x1F637;',
        ':massage:'                         => '&#x1F486;',
        ':meat_on_bone:'                    => '&#x1F356;',
        ':mega:'                            => '&#x1F4E3;',
        ':melon:'                           => '&#x1F348;',
        ':memo:'                            => '&#x1F4DD;',
        ':mens:'                            => '&#x1F6B9;',
        ':metro:'                           => '&#x1F687;',
        ':microphone:'                      => '&#x1F3A4;',
        ':microscope:'                      => '&#x1F52C;',
        ':milky_way:'                       => '&#x1F30C;',
        ':minibus:'                         => '&#x1F690;',
        ':minidisc:'                        => '&#x1F4BD;',
        ':mobile_phone_off:'                => '&#x1F4F4;',
        ':money_with_wings:'                => '&#x1F4B8;',
        ':moneybag:'                        => '&#x1F4B0;',
        ':monkey:'                          => '&#x1F412;',
        ':monkey_face:'                     => '&#x1F435;',
        ':monorail:'                        => '&#x1F69D;',
        ':moon:'                            => '&#x1F319;',
        ':mortar_board:'                    => '&#x1F393;',
        ':mount_fuji:'                      => '&#x1F5FB;',
        ':mountain_bicyclist:'              => '&#x1F6B5;',
        ':mountain_cableway:'               => '&#x1F6A0;',
        ':mountain_railway:'                => '&#x1F69E;',
        ':mouse2:'                          => '&#x1F401;',
        ':mouse:'                           => '&#x1F42D;',
        ':movie_camera:'                    => '&#x1F3A5;',
        ':moyai:'                           => '&#x1F5FF;',
        ':muscle:'                          => '&#x1F4AA;',
        ':mushroom:'                        => '&#x1F344;',
        ':musical_keyboard:'                => '&#x1F3B9;',
        ':musical_note:'                    => '&#x1F3B5;',
        ':musical_score:'                   => '&#x1F3BC;',
        ':mute:'                            => '&#x1F507;',
        ':nail_care:'                       => '&#x1F485;',
        ':name_badge:'                      => '&#x1F4DB;',
        ':necktie:'                         => '&#x1F454;',
        ':negative_squared_cross_mark:'     => '&#x274E;',
        ':neutral_face:'                    => '&#x1F610;',
        ':new:'                             => '&#x1F195;',
        ':new_moon:'                        => '&#x1F311;',
        ':new_moon_with_face:'              => '&#x1F31A;',
        ':newspaper:'                       => '&#x1F4F0;',
        ':ng:'                              => '&#x1F196;',
        ':nine:'                            => '&#x0039;',
        ':no_bell:'                         => '&#x1F515;',
        ':no_bicycles:'                     => '&#x1F6B3;',
        ':no_entry:'                        => '&#x26D4;',
        ':no_entry_sign:'                   => '&#x1F6AB;',
        ':no_good:'                         => '&#x1F645;',
        ':no_mobile_phones:'                => '&#x1F4F5;',
        ':no_mouth:'                        => '&#x1F636;',
        ':no_pedestrians:'                  => '&#x1F6B7;',
        ':no_smoking:'                      => '&#x1F6AD;',
        ':non-potable_water:'               => '&#x1F6B1;',
        ':nose:'                            => '&#x1F443;',
        ':notebook:'                        => '&#x1F4D3;',
        ':notebook_with_decorative_cover:'  => '&#x1F4D4;',
        ':notes:'                           => '&#x1F3B6;',
        ':nut_and_bolt:'                    => '&#x1F529;',
        ':o2:'                              => '&#x1F17E;',
        ':o:'                               => '&#x2B55;',
        ':ocean:'                           => '&#x1F30A;',
        ':octocat:'                         => '&#x1f431;',
        ':octopus:'                         => '&#x1F419;',
        ':oden:'                            => '&#x1F362;',
        ':office:'                          => '&#x1F3E2;',
        ':ok:'                              => '&#x1F197;',
        ':ok_hand:'                         => '&#x1F44C;',
        ':ok_woman:'                        => '&#x1F646;',
        ':older_man:'                       => '&#x1F474;',
        ':older_woman:'                     => '&#x1F475;',
        ':on:'                              => '&#x1F51B;',
        ':oncoming_automobile:'             => '&#x1F698;',
        ':oncoming_bus:'                    => '&#x1F68D;',
        ':oncoming_police_car:'             => '&#x1F694;',
        ':oncoming_taxi:'                   => '&#x1F696;',
        ':one:'                             => '&#x0031;',
        ':open_file_folder:'                => '&#x1F4C2;',
        ':open_hands:'                      => '&#x1F450;',
        ':open_mouth:'                      => '&#x1F62E;',
        ':ophiuchus:'                       => '&#x26CE;',
        ':orange_book:'                     => '&#x1F4D9;',
        ':outbox_tray:'                     => '&#x1F4E4;',
        ':ox:'                              => '&#x1F402;',
        ':page_facing_up:'                  => '&#x1F4C4;',
        ':page_with_curl:'                  => '&#x1F4C3;',
        ':pager:'                           => '&#x1F4DF;',
        ':palm_tree:'                       => '&#x1F334;',
        ':panda_face:'                      => '&#x1F43C;',
        ':paperclip:'                       => '&#x1F4CE;',
        ':parking:'                         => '&#x1F17F;',
        ':part_alternation_mark:'           => '&#x303D;',
        ':partly_sunny:'                    => '&#x26C5;',
        ':passport_control:'                => '&#x1F6C2;',
        ':paw_prints:'                      => '&#x1F43E;',
        ':peach:'                           => '&#x1F351;',
        ':pear:'                            => '&#x1F350;',
        ':pencil2:'                         => '&#x270F;',
        ':pencil:'                          => '&#x1F4DD;',
        ':penguin:'                         => '&#x1F427;',
        ':pensive:'                         => '&#x1F614;',
        ':performing_arts:'                 => '&#x1F3AD;',
        ':persevere:'                       => '&#x1F623;',
        ':person_frowning:'                 => '&#x1F64D;',
        ':person_with_blond_hair:'          => '&#x1F471;',
        ':person_with_pouting_face:'        => '&#x1F64E;',
        ':phone:'                           => '&#x260E;',
        ':pig2:'                            => '&#x1F416;',
        ':pig:'                             => '&#x1F437;',
        ':pig_nose:'                        => '&#x1F43D;',
        ':pill:'                            => '&#x1F48A;',
        ':pineapple:'                       => '&#x1F34D;',
        ':pisces:'                          => '&#x2653;',
        ':pizza:'                           => '&#x1F355;',
        ':point_down:'                      => '&#x1F447;',
        ':point_left:'                      => '&#x1F448;',
        ':point_right:'                     => '&#x1F449;',
        ':point_up:'                        => '&#x261D;',
        ':point_up_2:'                      => '&#x1F446;',
        ':police_car:'                      => '&#x1F693;',
        ':poodle:'                          => '&#x1F429;',
        ':poop:'                            => '&#x1F4A9;',
        ':post_office:'                     => '&#x1F3E3;',
        ':postal_horn:'                     => '&#x1F4EF;',
        ':postbox:'                         => '&#x1F4EE;',
        ':potable_water:'                   => '&#x1F6B0;',
        ':pouch:'                           => '&#x1F45D;',
        ':poultry_leg:'                     => '&#x1F357;',
        ':pound:'                           => '&#x1F4B7;',
        ':pouting_cat:'                     => '&#x1F63E;',
        ':pray:'                            => '&#x1F64F;',
        ':princess:'                        => '&#x1F478;',
        ':punch:'                           => '&#x1F44A;',
        ':purple_heart:'                    => '&#x1F49C;',
        ':purse:'                           => '&#x1F45B;',
        ':pushpin:'                         => '&#x1F4CC;',
        ':put_litter_in_its_place:'         => '&#x1F6AE;',
        ':question:'                        => '&#x2753;',
        ':rabbit2:'                         => '&#x1F407;',
        ':rabbit:'                          => '&#x1F430;',
        ':racehorse:'                       => '&#x1F40E;',
        ':radio:'                           => '&#x1F4FB;',
        ':radio_button:'                    => '&#x1F518;',
        ':rage:'                            => '&#x1F621;',
        ':railway_car:'                     => '&#x1F683;',
        ':rainbow:'                         => '&#x1F308;',
        ':raised_hand:'                     => '&#x1F64B;',
        ':raised_hands:'                    => '&#x1F64C;',
        ':ram:'                             => '&#x1F40F;',
        ':ramen:'                           => '&#x1F35C;',
        ':rat:'                             => '&#x1F400;',
        ':recycle:'                         => '&#x267B;',
        ':red_car:'                         => '&#x1F697;',
        ':red_circle:'                      => '&#x1F534;',
        ':registered:'                      => '&#x00AE;',
        ':relaxed:'                         => '&#x263A;',
        ':relieved:'                        => '&#x1F625;',
        ':repeat:'                          => '&#x1F501;',
        ':repeat_one:'                      => '&#x1F502;',
        ':restroom:'                        => '&#x1F6BB;',
        ':revolving_hearts:'                => '&#x1F49E;',
        ':rewind:'                          => '&#x23EA;',
        ':ribbon:'                          => '&#x1F380;',
        ':rice:'                            => '&#x1F35A;',
        ':rice_ball:'                       => '&#x1F359;',
        ':rice_cracker:'                    => '&#x1F358;',
        ':rice_scene:'                      => '&#x1F391;',
        ':ring:'                            => '&#x1F48D;',
        ':rocket:'                          => '&#x1F680;',
        ':roller_coaster:'                  => '&#x1F3A2;',
        ':rooster:'                         => '&#x1F413;',
        ':rose:'                            => '&#x1F339;',
        ':rotating_light:'                  => '&#x1F6A8;',
        ':round_pushpin:'                   => '&#x1F4CD;',
        ':rowboat:'                         => '&#x1F6A3;',
        ':ru:'                              => '&#x1F1F7;&#x1F1FA;',
        ':rugby_football:'                  => '&#x1F3C9;',
        ':runner:'                          => '&#x1F3C3;',
        ':running:'                         => '&#x1F3C3;',
        ':running_shirt_with_sash:'         => '&#x1F3BD;',
        ':sa:'                              => '&#x1F202;',
        ':sagittarius:'                     => '&#x2650;',
        ':sailboat:'                        => '&#x26F5;',
        ':sake:'                            => '&#x1F376;',
        ':sandal:'                          => '&#x1F461;',
        ':santa:'                           => '&#x1F385;',
        ':satellite:'                       => '&#x1F4E1;',
        ':satisfied:'                       => '&#x1F60C;',
        ':saxophone:'                       => '&#x1F3B7;',
        ':school:'                          => '&#x1F3EB;',
        ':school_satchel:'                  => '&#x1F392;',
        ':scissors:'                        => '&#x2702;',
        ':scorpius:'                        => '&#x264F;',
        ':scream:'                          => '&#x1F631;',
        ':scream_cat:'                      => '&#x1F640;',
        ':scroll:'                          => '&#x1F4DC;',
        ':seat:'                            => '&#x1F4BA;',
        ':secret:'                          => '&#x3299;',
        ':see_no_evil:'                     => '&#x1F648;',
        ':seedling:'                        => '&#x1F331;',
        ':seven:'                           => '&#x0037;',
        ':shaved_ice:'                      => '&#x1F367;',
        ':sheep:'                           => '&#x1F411;',
        ':shell:'                           => '&#x1F41A;',
        ':ship:'                            => '&#x1F6A2;',
        ':shirt:'                           => '&#x1F455;',
        ':shit:'                            => '&#x1F4A9;',
        ':shoe:'                            => '&#x1F45E;',
        ':shower:'                          => '&#x1F6BF;',
        ':signal_strength:'                 => '&#x1F4F6;',
        ':six:'                             => '&#x0036;',
        ':six_pointed_star:'                => '&#x1F52F;',
        ':ski:'                             => '&#x1F3BF;',
        ':skull:'                           => '&#x1F480;',
        ':sleeping:'                        => '&#x1F634;',
        ':sleepy:'                          => '&#x1F62A;',
        ':slot_machine:'                    => '&#x1F3B0;',
        ':small_blue_diamond:'              => '&#x1F539;',
        ':small_orange_diamond:'            => '&#x1F538;',
        ':small_red_triangle:'              => '&#x1F53A;',
        ':small_red_triangle_down:'         => '&#x1F53B;',
        ':smile:'                           => '&#x1F604;',
        ':smile_cat:'                       => '&#x1F638;',
        ':smiley:'                          => '&#x1F603;',
        ':smiley_cat:'                      => '&#x1F63A;',
        ':smiling_imp:'                     => '&#x1F608;',
        ':smirk:'                           => '&#x1F60F;',
        ':smirk_cat:'                       => '&#x1F63C;',
        ':smoking:'                         => '&#x1F6AC;',
        ':snail:'                           => '&#x1F40C;',
        ':snake:'                           => '&#x1F40D;',
        ':snowboarder:'                     => '&#x1F3C2;',
        ':snowflake:'                       => '&#x2744;',
        ':snowman:'                         => '&#x26C4;',
        ':sob:'                             => '&#x1F62D;',
        ':soccer:'                          => '&#x26BD;',
        ':soon:'                            => '&#x1F51C;',
        ':sos:'                             => '&#x1F198;',
        ':sound:'                           => '&#x1F509;',
        ':space_invader:'                   => '&#x1F47E;',
        ':spades:'                          => '&#x2660;',
        ':spaghetti:'                       => '&#x1F35D;',
        ':sparkler:'                        => '&#x1F387;',
        ':sparkles:'                        => '&#x2728;',
        ':sparkling_heart:'                 => '&#x1F496;',
        ':speak_no_evil:'                   => '&#x1F64A;',
        ':speaker:'                         => '&#x1F50A;',
        ':speech_balloon:'                  => '&#x1F4AC;',
        ':speedboat:'                       => '&#x1F6A4;',
        ':star2:'                           => '&#x1F31F;',
        ':star:'                            => '&#x2B50;',
        ':stars:'                           => '&#x1F303;',
        ':station:'                         => '&#x1F689;',
        ':statue_of_liberty:'               => '&#x1F5FD;',
        ':steam_locomotive:'                => '&#x1F682;',
        ':stew:'                            => '&#x1F372;',
        ':straight_ruler:'                  => '&#x1F4CF;',
        ':strawberry:'                      => '&#x1F353;',
        ':stuck_out_tongue:'                => '&#x1F61B;',
        ':stuck_out_tongue_closed_eyes:'    => '&#x1F61D;',
        ':stuck_out_tongue_winking_eye:'    => '&#x1F61C;',
        ':sun_with_face:'                   => '&#x1F31E;',
        ':sunflower:'                       => '&#x1F33B;',
        ':sunglasses:'                      => '&#x1F60E;',
        ':sunny:'                           => '&#x2600;',
        ':sunrise:'                         => '&#x1F305;',
        ':sunrise_over_mountains:'          => '&#x1F304;',
        ':surfer:'                          => '&#x1F3C4;',
        ':sushi:'                           => '&#x1F363;',
        ':suspension_railway:'              => '&#x1F69F;',
        ':sweat:'                           => '&#x1F613;',
        ':sweat_drops:'                     => '&#x1F4A6;',
        ':sweat_smile:'                     => '&#x1F605;',
        ':sweet_potato:'                    => '&#x1F360;',
        ':swimmer:'                         => '&#x1F3CA;',
        ':symbols:'                         => '&#x1F523;',
        ':syringe:'                         => '&#x1F489;',
        ':tada:'                            => '&#x1F389;',
        ':tanabata_tree:'                   => '&#x1F38B;',
        ':tangerine:'                       => '&#x1F34A;',
        ':taurus:'                          => '&#x2649;',
        ':taxi:'                            => '&#x1F695;',
        ':tea:'                             => '&#x1F375;',
        ':telephone:'                       => '&#x260E;',
        ':telephone_receiver:'              => '&#x1F4DE;',
        ':telescope:'                       => '&#x1F52D;',
        ':tennis:'                          => '&#x1F3BE;',
        ':tent:'                            => '&#x26FA;',
        ':thought_balloon:'                 => '&#x1F4AD;',
        ':three:'                           => '&#x0033;',
        ':thumbsdown:'                      => '&#x1F44E;',
        ':thumbsup:'                        => '&#x1F44D;',
        ':ticket:'                          => '&#x1F3AB;',
        ':tiger2:'                          => '&#x1F405;',
        ':tiger:'                           => '&#x1F42F;',
        ':tired_face:'                      => '&#x1F62B;',
        ':tm:'                              => '&#x2122;',
        ':toilet:'                          => '&#x1F6BD;',
        ':tokyo_tower:'                     => '&#x1F5FC;',
        ':tomato:'                          => '&#x1F345;',
        ':tongue:'                          => '&#x1F445;',
        ':top:'                             => '&#x1F51D;',
        ':tophat:'                          => '&#x1F3A9;',
        ':tractor:'                         => '&#x1F69C;',
        ':traffic_light:'                   => '&#x1F6A5;',
        ':train2:'                          => '&#x1F686;',
        ':train:'                           => '&#x1F683;',
        ':tram:'                            => '&#x1F68A;',
        ':triangular_flag_on_post:'         => '&#x1F6A9;',
        ':triangular_ruler:'                => '&#x1F4D0;',
        ':trident:'                         => '&#x1F531;',
        ':triumph:'                         => '&#x1F624;',
        ':trolleybus:'                      => '&#x1F68E;',
        ':trophy:'                          => '&#x1F3C6;',
        ':tropical_drink:'                  => '&#x1F379;',
        ':tropical_fish:'                   => '&#x1F420;',
        ':truck:'                           => '&#x1F69A;',
        ':trumpet:'                         => '&#x1F3BA;',
        ':tshirt:'                          => '&#x1F455;',
        ':tulip:'                           => '&#x1F337;',
        ':turtle:'                          => '&#x1F422;',
        ':tv:'                              => '&#x1F4FA;',
        ':twisted_rightwards_arrows:'       => '&#x1F500;',
        ':two:'                             => '&#x0032;',
        ':two_hearts:'                      => '&#x1F495;',
        ':two_men_holding_hands:'           => '&#x1F46C;',
        ':two_women_holding_hands:'         => '&#x1F46D;',
        ':u5272:'                           => '&#x1F239;',
        ':u5408:'                           => '&#x1F234;',
        ':u55b6:'                           => '&#x1F23A;',
        ':u6307:'                           => '&#x1F22F;',
        ':u6708:'                           => '&#x1F237;',
        ':u6709:'                           => '&#x1F236;',
        ':u6e80:'                           => '&#x1F235;',
        ':u7121:'                           => '&#x1F21A;',
        ':u7533:'                           => '&#x1F238;',
        ':u7981:'                           => '&#x1F232;',
        ':u7a7a:'                           => '&#x1F233;',
        ':uk:'                              => '&#x1F1EC;&#x1F1E7;',
        ':umbrella:'                        => '&#x2614;',
        ':unamused:'                        => '&#x1F612;',
        ':underage:'                        => '&#x1F51E;',
        ':unlock:'                          => '&#x1F513;',
        ':up:'                              => '&#x1F199;',
        ':us:'                              => '&#x1F1FA;&#x1F1F8;',
        ':v:'                               => '&#x270C;',
        ':vertical_traffic_light:'          => '&#x1F6A6;',
        ':vhs:'                             => '&#x1F4FC;',
        ':vibration_mode:'                  => '&#x1F4F3;',
        ':video_camera:'                    => '&#x1F4F9;',
        ':video_game:'                      => '&#x1F3AE;',
        ':violin:'                          => '&#x1F3BB;',
        ':virgo:'                           => '&#x264D;',
        ':volcano:'                         => '&#x1F30B;',
        ':vs:'                              => '&#x1F19A;',
        ':walking:'                         => '&#x1F6B6;',
        ':waning_crescent_moon:'            => '&#x1F318;',
        ':waning_gibbous_moon:'             => '&#x1F316;',
        ':warning:'                         => '&#x26A0;',
        ':watch:'                           => '&#x231A;',
        ':water_buffalo:'                   => '&#x1F403;',
        ':watermelon:'                      => '&#x1F349;',
        ':wave:'                            => '&#x1F44B;',
        ':wavy_dash:'                       => '&#x3030;',
        ':waxing_crescent_moon:'            => '&#x1F312;',
        ':waxing_gibbous_moon:'             => '&#x1F314;',
        ':wc:'                              => '&#x1F6BE;',
        ':weary:'                           => '&#x1F629;',
        ':wedding:'                         => '&#x1F492;',
        ':whale2:'                          => '&#x1F40B;',
        ':whale:'                           => '&#x1F433;',
        ':wheelchair:'                      => '&#x267F;',
        ':white_check_mark:'                => '&#x2705;',
        ':white_circle:'                    => '&#x26AA;',
        ':white_flower:'                    => '&#x1F4AE;',
        ':white_square_button:'             => '&#x1F533;',
        ':wind_chime:'                      => '&#x1F390;',
        ':wine_glass:'                      => '&#x1F377;',
        ':wink2:'                           => '&#x1F61C;',
        ':wink:'                            => '&#x1F609;',
        ':wolf:'                            => '&#x1F43A;',
        ':woman:'                           => '&#x1F469;',
        ':womans_clothes:'                  => '&#x1F45A;',
        ':womans_hat:'                      => '&#x1F452;',
        ':womens:'                          => '&#x1F6BA;',
        ':worried:'                         => '&#x1F61F;',
        ':wrench:'                          => '&#x1F527;',
        ':x:'                               => '&#x274C;',
        ':yellow_heart:'                    => '&#x1F49B;',
        ':yen:'                             => '&#x1F4B4;',
        ':yum:'                             => '&#x1F60B;',
        ':zap:'                             => '&#x26A1;',
        ':zero:'                            => '&#x0030;',
        ':zzz:'                             => '&#x1F4A4;'
    ];

    /**
     * Emojify constructor.
     * @param mixed $services application services
     */
    public function __construct($services)
    {
        $this->setServices($services);
    }

    /**
     * Sets the root of the emojis. By default this is BASE_PATH so that BASE_PATH/public is used.
     * @param string $emojiRoot the root.
     */
    public function setEmojiRoot(string $emojiRoot)
    {
        $this->emojiRoot = $emojiRoot;
    }

    /**
     * Replace occurrences of emoji short-hand (e.g. :smile:) with html-entities
     * for the corresponding Unicode codepoints (e.g. &#x1F604).
     *
     * If we detect a copy of Gemoji, favor its PNGs because they support more
     * emojis and work cross-browser/platform more reliably.
     *
     * http://www.emoji-cheat-sheet.com/
     * https://github.com/github/gemoji
     *
     * @param  string $value
     * @return string
     */
    public function filter($value)
    {
        // We have no need to worry about html because html is not really supported with markdown
        // and the markdown filter is only applied after the emojify filter. So we keep it simple
        // here. We will also take the opportunity here to not render the emoji when it is in quotes
        // or back tics.
        $filter = $this;

        return preg_replace_callback(
            '/(?<!"|\'|`|```|&quot;|\w)(\:[a-z0-9\+\-_]+\:)(?!"|\'|`|```|&quot;|\w)/',
            function ($matches) use ($filter) {
                if (count($matches) !== 2) {
                    return $matches[0];
                }

                // automatically upgrade to gemoji images if present
                // we support two forms of emoji-code to filename conventions
                // 1. the emoji short name is the filename
                // 2. the corresponding unicode codepoint is the filename
                $path        = '/' . trim(
                    ConfigManager::getValue(
                        $filter->getServices()->get('config'),
                        ConfigManager::ENVIRONMENT_EMOJI_URL,
                        'vendor/gemoji/images'
                    ),
                    "/"
                );
                $gemojiPath  = $this->emojiRoot . '/public' . $path;
                $namedFile   = $gemojiPath . '/' . trim($matches[1], ':') . '.png';
                $unicodeFile = isset($filter->emoji[$matches[1]]) ? $filter->emoji[$matches[1]] : null;
                $unicodeFile = strtolower(str_replace([';&#x', '&#x', ';'], ['-', '', ''], $unicodeFile));
                $unicodeFile = $gemojiPath . '/unicode/' . $unicodeFile . '.png';
                if (is_dir($gemojiPath) && (file_exists($namedFile) || file_exists($unicodeFile))) {
                    $fileUrl = file_exists($namedFile)
                        ? $path . '/' . basename($namedFile)
                        : $path . '/unicode/' . basename($unicodeFile);
                    return '<img class="emoji" title="' . $matches[1] . '" alt="' . $matches[1] . '"'
                        . ' width="18" height="18"' . ' src="' . $fileUrl . '">';
                }

                // no gemoji, use unicode codepoints
                if (isset($filter->emoji[$matches[1]])) {
                    return '<span class="emoji" title="' . $matches[1] . '">'
                        . $filter->emoji[$matches[1]] . '</span>';
                }

                return $matches[0];
            },
            $value
        );
    }
}
