<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace AiAnalysis\Service;

use AiAnalysis\Helper\IAiAnalysisHelper;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Log\SwarmLogger;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use Laminas\Mvc\Controller\AbstractRestfulController;
use Laminas\Http\Client as HttpClient;
use Laminas\Http\Request;

abstract class AbstractAiAdapter extends AbstractRestfulController
{
    const API_BASE = '/api';
    const ERROR    = 'error';
    const MESSAGES = 'messages';
    const DATA     = 'data';
    const CODE     = 'code';
    const TEXT     = 'text';

    const DATA_DIFF_CONTENT = 'diffContent';
    const LOG_PREFIX        = AbstractAiAdapter::class;

    protected $services = null;

    /**
     * IndexController constructor.
     * @param $services
     */
    public function __construct($services)
    {
        $this->services = $services;
    }
    /**
     * Build a message with the code and message text
     * @param mixed  $code      message code
     * @param string $text      message text
     * @return array with self::CODE => $code and self::TEXT => $text
     */
    public function buildMessage($code, string $text): array
    {
        return [self::CODE => $code, self::TEXT => $text];
    }

    /**
     * Return Success Response
     * @param mixed $data           Data that will be returned
     * @param array $messages       optional messages, defaults to empty array
     * @return JsonModel
     */
    public function success($data, array $messages = []): JsonModel
    {
        return $this->buildResponse($data, $messages);
    }

    /**
     * Return Error Response
     * @param  array    $messages       messages
     * @param  mixed    $errorCode      error code
     * @param  mixed    $data           optional data
     * @return JsonModel
     */
    public function error(array $messages, $errorCode, $data = null): JsonModel
    {
        return $this->buildResponse($data, $messages, $errorCode);
    }

    /**
     * Prepares Json Response
     * @param  array|null   $data           Data that will be returned
     * @param  array        $messages       messages to return
     * @param  int|null     $errorCode      error code
     * @return JsonModel
     */
    private function buildResponse($data, array $messages, $errorCode = null): JsonModel
    {
        $returnResponse = [
            self::ERROR     => $errorCode,
            self::MESSAGES  => $messages,
            self::DATA      => $data
        ];
        return new JsonModel($returnResponse);
    }

    /**
     * Function that make the GET or POST requests
     * @param string $url Url in which we want to make our request to
     * @param bool $fetch Set to true for a GET request, otherwise will do a POST.
     * @param array $args These are the parameter that we pass the GET or POST request to filter the reviews
     * @param array $headers These are the parameter that we pass the GET or POST request to filter the reviews
     * @param string $mimeType mimeType for the given request
     * @throws ConfigException
     */
    protected function request(
        string $url,
        bool   $fetch = false,
        array  $args = array(),
        array  $headers = [],
        string $mimeType = 'application/json'
    ) {
        $logger          = $this->services->get(SwarmLogger::SERVICE);
        $client          = new HttpClient;
        $timeout         = $this->getTimeout();
        $request         = new Request();
        $responseContent = '';
        try {
            $request->setMethod(Request::METHOD_POST);
            $request->setUri($url);
            $request->getHeaders()->addHeaders($headers);
            $request->setContent(json_encode($args));

            $logger->debug(sprintf("%s: request headers %s", self::LOG_PREFIX, json_encode($headers)));
            $logger->debug(sprintf("%s: request body %s", self::LOG_PREFIX, json_encode($args)));

            if (!is_int($timeout)) {
                $logger->trace(
                    sprintf(
                        "[%s]: Invalid ai_timeout value set [%s]. Please set only integer value",
                        self::LOG_PREFIX,
                        $timeout
                    )
                );
                return new JsonModel(
                    [self::ERROR => 'Invalid ai_timeout config value, Please set only integer value',
                        self::CODE => Response::STATUS_CODE_500]
                );
            }
            $client->setOptions(
                [
                    'maxredirects' => 0,
                    'timeout' => $timeout,
                ]
            );
            $client->setUri($url);
            if ($headers) {
                $client->setHeaders($headers);
            }
            $client->setMethod(Request::METHOD_POST);
            $client->setRawBody(json_encode($args));
            $client->setEncType($mimeType);

            $response = $client->dispatch($request);
            if (!$response) {
                $logger->trace(sprintf("[%s]: Failed to get response back. ", self::LOG_PREFIX));
            }

            $encodedContent = $response->getBody();
            if (!$encodedContent) {
                $logger->trace(sprintf("[%s]: Failed to get response content. ", self::LOG_PREFIX));
            }
            $responseContent = json_decode($encodedContent);
            if (!$responseContent) {
                $logger->trace(
                    sprintf(
                        "[%s]: Failed at json decoding of response content. ",
                        self::LOG_PREFIX
                    )
                );
            }
        } catch (Exception $e) {
            $logger->err(sprintf("%s: unexpected error %s", self::LOG_PREFIX, $e->getMessage()));
        }
        // We are intentionally keeping the check of data since few clients have
        // implemented it in this way
        return is_object($responseContent) && property_exists($responseContent, 'data')
            ? $responseContent->data : $responseContent;
    }

    /**
     * Get the timeout value for the ai review config.
     * @return int
     * @throws ConfigException
     */
    protected function getTimeout(): int
    {
        $config = $this->services->get(IConfigDefinition::CONFIG);
        return ConfigManager::getValue($config, IConfigDefinition::AI_REVIEW_AI_TIMEOUT);
    }

    /**
     * This function will validate the result returned from AI vendor
     * @param object|mixed $result
     * @return bool
     */
    protected function validateResult(mixed $result): bool
    {
        $result = $this->normalizeApiResponse($result);
        if (!property_exists($result, 'error') &&
            property_exists($result, 'choices') &&
            isset($result->choices[0]) && is_object($result->choices[0]) &&
            property_exists($result->choices[0], 'message') &&
            is_object($result->choices[0]->message) &&
            property_exists($result->choices[0]->message, 'content')) {
            return true;
        }
        return false;
    }

    /**
     * Function to normalizeApiResponse
     * @param mixed $response
     * @return object|string
     */
    protected function normalizeApiResponse(mixed $response): object|string
    {
        // Case 1: already an object → return as is
        if (is_object($response)) {
            return $response;
        }

        // Case 2: array → encode to JSON string
        if (is_array($response)) {
            return json_encode($response, JSON_UNESCAPED_UNICODE);
        }

        // Case 3: XML string → leave it as string
        // Case 4: plain string → leave as is
        // Case 5: null/empty → return empty string
        if (is_null($response) || $response === '') {
            return '';
        }

        // Case 6: numbers, bool → cast to string
        return (string)$response;
    }

    /**
     * This method will fetch the API key from config, required to execute the open AI request
     * @return string
     * @throws ConfigException
     */
    protected function getApiKey(): string
    {
        $config = $this->services->get(IConfigDefinition::CONFIG);
        return ConfigManager::getValue($config, IConfigDefinition::AI_REVIEW_AI_VENDORS_AI_MODEL1_API_KEY);
    }

    /**
     * This method will fetch the API key from config, required to execute the generic AI request
     * @throws ConfigException
     * @return string
     */
    protected function getApiUrl() : string
    {
        $config = $this->services->get(IConfigDefinition::CONFIG);
        return ConfigManager::getValue($config, IConfigDefinition::AI_REVIEW_AI_VENDORS_AI_MODEL1_API_END_POINT);
    }

    /**
     * This function will work to send the call to AI Vendor and fetch improved comment
     * @param mixed $data
     * @return array
     */
    public function getImprovedCommentByAi(mixed $data): array
    {
        $logger = $this->services->get(SwarmLogger::SERVICE);
        try {
            $requestBody = ['messages' => [
                [
                    'role' => 'user',
                    'content' => $data['statement']
                ],
            ]];
            $headers     = [];
            $apiKey      = $this->getApiKey();
            $apiEndPoint = $this->getApiUrl();
            if ($apiKey) {
                if (!$apiEndPoint) {
                    $logger->trace(
                        sprintf(
                            "[%s]: Failed in establishing connection as end point is invalid ",
                            self::LOG_PREFIX
                        )
                    );

                    return [IAiAnalysisHelper::CONTENT_GENERATED => '',
                        self::ERROR => 'Unable to process request: API END POINT not provided.',
                        self::CODE => Response::STATUS_CODE_404];
                }
                $headers = [
                    "Content-Type: application/json",
                    "Authorization: Bearer " . $apiKey
                ];

                $requestBody = [
                    'model' => $data[IConfigDefinition::AI_MODEL],
                    'messages' => [
                        [
                            'role' => 'user',
                            'content' => $data['statement']
                        ],
                    ],
                ];
            }

            $result = $this->request($this->getApiUrl(), false, $requestBody, $headers);

            if ($this->validateResult($result)) {
                $logger->trace(sprintf("[%s]: Content generation successful ", self::LOG_PREFIX));
                return [IAiAnalysisHelper::CONTENT_GENERATED => $result->choices[0]->message->content,
                    self::ERROR => null,
                    self::CODE => Response::STATUS_CODE_200];
            } else {
                $logger->trace(sprintf("[%s]: Failed at content generation ", self::LOG_PREFIX));
                return [IAiAnalysisHelper::CONTENT_GENERATED => '',
                    self::ERROR => is_object($result) && property_exists($result, 'error') ? $result->error :
                        'Response not received from AI Vendor',
                    self::CODE => is_object($result) && property_exists($result, 'code') ? $result->code :
                        Response::STATUS_CODE_500];
            }
        } catch (\Exception | \Throwable $exception) {
            $logger->debug(
                sprintf(
                    "[%s]: Error occurred at LM Studio AI Adapter %s",
                    self::LOG_PREFIX,
                    $exception->getMessage()
                )
            );
            $statusCode = $exception->getCode();
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                'error' => $exception->getMessage(),
                'code' => $statusCode === 0 ? Response::STATUS_CODE_500 : $statusCode,
            ];
        }
    }
}
