<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.5/2869592
 */

namespace AiAnalysis;

use Application\Checker;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Config\ConfigException;
use InvalidArgumentException;
use Psr\Container\ContainerExceptionInterface;

/**
 * Class AiAnalysisCharLimitChecker. Checks for character limits present in AI config if input adheres them.
 * @package AiAnalysis
 */
class AiAnalysisCharLimitChecker extends Checker
{
    /**
     * Performs a check to see that input data adheres character limits present in AI config.
     * @param string            $check      the name of the check
     * @param array|null        $options    optional data to assist the check
     * @throws InvalidArgumentException if input data failed to adhere character limits
     * @throws ConfigException
     * @throws ContainerExceptionInterface
     */
    public function check(string $check, array $options = null): void
    {
        // Check if input data adheres character limits present in AI config
        $AiMinCharLimit = ConfigManager::getValue(
            $this->services->get(IConfigDefinition::CONFIG),
            IConfigDefinition::AI_REVIEW_AI_VENDORS_AI_MODEL1_AI_MIN_CHAR_LIMIT
        );
        $AiMaxCharLimit = ConfigManager::getValue(
            $this->services->get(IConfigDefinition::CONFIG),
            IConfigDefinition::AI_REVIEW_AI_VENDORS_AI_MODEL1_AI_MAX_CHAR_LIMIT
        );
        $inputData      = $options[Checker::VALUE];
        $sizeInputData  = strlen($inputData);
        if ($sizeInputData < $AiMinCharLimit || $sizeInputData > $AiMaxCharLimit) {
            throw new InvalidArgumentException(
                sprintf(
                    "Input Diff content character count is not between minimum %d and maximum %d characters",
                    $AiMinCharLimit,
                    $AiMaxCharLimit
                )
            );
        }
    }
}
