<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Users\Listener;

use Application\Model\IModelDAO;
use Events\Listener\AbstractEventListener;
use Events\Listener\ListenerFactory;
use Laminas\EventManager\Event;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Users\Service\CleanUpDeletedUsers;
use Users\Service\ICleanUpDeletedUsers;

class UserEventListener extends AbstractEventListener
{
    public function onUser(Event $event)
    {
        $name = $event->getName();
        $id   = $event->getParam('id');
        $this->logger->trace(self::class." processing event $name for id $id");
        // ignore git-fusion-reviews-* users - these are regularly updated
        // and used internally by git-fusion in ways that don't concern us
        if (strpos($id, 'git-fusion-reviews-') === 0) {
            $this->logger->trace(self::class." Ignoring git-fusion-review users");
            return;
        }

        try {
            $userDao = $this->services->get(IModelDAO::USER_DAO);
            // This will remove from the cache first if set and then add it again which will
            // handle both new users and deletes
            $userDao->fetchByIdAndSet($event->getParam('id'));
            $this->logger->trace(self::class." processed event $name for id $id");
        } catch (\Exception $e) {
            $this->logger->err($e);
        }
    }

    /**
     * Listener responsible for handling the user deletion event.
     *
     * @param Event $event
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function processUserDelete(Event $event)
    {
        $id   = $event->getParam('id');
        $name = $event->getName();
        try {
            /** @var CleanUpDeletedUsers $cleanUpDeletedUsers */
            $cleanUpDeletedUsers = $this->services->get(ICleanUpDeletedUsers::SERVICE);
            $cleanUpDeletedUsers->handleCleanUpOfDeletedUsersData($id);
            $this->logger->trace(self::class." processed event $name for id $id");
        } catch (\Exception $e) {
            $this->logger->err($e);
        }
    }

    /**
     * Listener responsible for processing cleanup when users are deleted.
     *
     * @param Event $event
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function processCleanupOfDeletedUsers(Event $event)
    {
        $id   = $event->getParam('id') ?? null;
        $name = $event->getName();
        $data = $event->getParam('data');
        $type = $data['type'] ?? '';

        if (!$id || !$type) {
            return;
        }

        try {
            /** @var CleanUpDeletedUsers $cleanUpDeletedUsers */
            $cleanUpDeletedUsers = $this->services->get(ICleanUpDeletedUsers::SERVICE);
            match ($type) {
                ListenerFactory::FOLLOWERS       => $cleanUpDeletedUsers->cleanUpFollowersForDeletedUsers($event),
                ListenerFactory::WORKFLOW        => $cleanUpDeletedUsers->cleanUpWorkflowForDeletedUsers($event),
                ListenerFactory::TEST_DEFINITION => $cleanUpDeletedUsers->cleanUpTestDefinitionsForDeletedUsers($event),
                ListenerFactory::GROUP           => $cleanUpDeletedUsers->cleanUpGroupsForDeletedUsers($event),
                ListenerFactory::PROJECT         => $cleanUpDeletedUsers->cleanUpProjectsForDeletedUsers($event),
                ListenerFactory::CONFIG          => $cleanUpDeletedUsers->cleanUpConfigForDeletedUsers($id),
            };
            $this->logger->trace(self::class." processed event $name for id $id");
        } catch (\Exception $exception) {
            $this->logger->trace(
                sprintf(
                    '%s-Cannot remove user %s from %s due to following exception: %s',
                    self::class,
                    $id,
                    $type,
                    $exception->getMessage()
                )
            );
        }
    }
}
