<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Projects\Validator;

use Application\Config\IConfigDefinition;
use Application\Permissions\Exception\ForbiddenException;
use Application\Permissions\IPermissions;
use Application\Validator\ConnectedAbstractValidator;
use P4\Connection\AbstractConnection;
use P4\Connection\Exception\CommandException;

/**
 * Check if the given path is valid branch path.
 */
class BranchPath extends ConnectedAbstractValidator
{
    const UNSUPPORTED_WILDCARDS = 'unsupportedWildcards';
    const UNSUPPORTED_START     = 'unsupportedStart';
    const UNSUPPORTED_TEXT_PART = 'unsupportedTextPart';
    const RELATIVE_PATH         = 'relativePath';
    const INVALID_DEPOT         = 'invalidDepot';
    const UNFOLLOWED_DEPOT      = 'unfollowedDepot';
    const NULL_DIRECTORY        = 'nullDirectory';
    const NO_PATHS              = 'noPaths';
    const UNSUPPORTED_OVERLAY   = 'unsupportedOverlay';
    const CLIENT_MAP_TWISTED    = 'clientMapTwisted';
    const NO_PERMISSION         = 'noPermission';

    protected $messageTemplates = [
        self::UNSUPPORTED_WILDCARDS => "The only permitted wildcard is trailing '...' & '*' in the middle & end.",
        self::UNSUPPORTED_START     => "The '*' is not allowed at the start of path. it is allowed in the middle & end",
        self::UNSUPPORTED_TEXT_PART => "The '*' is only allowed at the end of text (like mai*).",
        self::RELATIVE_PATH         => "Relative paths (., ..) are not allowed.",
        self::INVALID_DEPOT         => "The first path component must be a valid depot name.",
        self::UNFOLLOWED_DEPOT      => "Depot name must be followed by a path or '/...'.",
        self::NULL_DIRECTORY        => "The path cannot contain null directories ('//') or end with a '/'.",
        self::NO_PATHS              => "No depot paths specified.",
        self::UNSUPPORTED_OVERLAY   => "Overlay '+' mappings are not supported",
        self::CLIENT_MAP_TWISTED    => "Client map too twisted for directory list.",
        self::NO_PERMISSION         => "No Permission to one or more branch paths."
    ];

    /**
     * In-memory cache of existing depots in Perforce (per connection).
     */
    protected $depots = null;
    protected $services;

    /**
     * Construct with options
     * - services: required services
     * @param $options
     */
    public function __construct($options = null)
    {
        $this->services = $options[self::SERVICES];
        parent::__construct($options);
    }

    /**
     * Extend parent to also clear in-memory cache for depots.
     *
     * @param  mixed        $connection
     * @return ConnectedAbstractValidator   provides a fluent interface
     */
    public function setConnection(AbstractConnection $connection)
    {
        $this->depots = null;
        return parent::setConnection($connection);
    }

    /**
     * Returns true if $value is a valid branch path or a list of valid branch paths.
     *
     * @param   string|array    $value  value or list of values to check for
     * @return  boolean         true if value is valid branch path, false otherwise
     */
    public function isValid($value)
    {
        // normalize to an array and knock out whitespace
        $value = array_filter(array_map('trim', (array)$value), 'strlen');

        // value must contain at least one path
        if (!count($value)) {
            $this->error(self::NO_PATHS);
            return false;
        }

        // The branch path permissions are only checked if the 'permission_check' configuration is set to true
        $config          = $this->services->get(IConfigDefinition::CONFIG);
        $checkPermission = isset($config[IConfigDefinition::PROJECTS][IConfigDefinition::PERMISSION_CHECK]) &&
            (bool)$config[IConfigDefinition::PROJECTS][IConfigDefinition::PERMISSION_CHECK];

        $depots = $this->getDepots();
        foreach ($value as $path) {
            // check for the allowed wild cards
            if (!$this->checkAllowedWildCards($path)) {
                return false;
            }

            // verify that the first path component is an existing depot
            preg_match('#^(-){0,1}//([^/]+)#', $path, $match);
            if (!isset($match[2]) || !in_array($match[2], $depots)) {
                preg_match('#^(\+){1}//([^/]+)#', $path, $plusMatch);
                if (isset($plusMatch) && count($plusMatch) > 0) {
                    $this->error(self::UNSUPPORTED_OVERLAY);
                    return false;
                }
                $this->error(self::INVALID_DEPOT);
                return false;
            }

            // check that depot name is followed by something ('//depot' or '//depot/'
            // are not permitted paths)
            if (!preg_match('#^(-){0,1}//[^/]+/[^/]+#', $path)) {
                $this->error(self::UNFOLLOWED_DEPOT);
                return false;
            }

            // check for existence of relative paths ('.', '..') which are not allowed
            // (i.e //depot/.. and //depot/../folder are not permitted, but //depot/a..b/folder is permitted)
            if (preg_match('#/\.\.?(/|$)#', $path)) {
                $this->error(self::RELATIVE_PATH);
                return false;
            }

            // ensure that the path doesn't end with a slash or contain null directories
            // as such paths are not allowed in client view mappings
            if (substr($path, -1) === '/' || preg_match('#.+.(-){0,1}//+#', $path)) {
                $this->error(self::NULL_DIRECTORY);
                return false;
            }

            // check if the logged-in user has correct permission for the every branch path
            if ($checkPermission && !$this->checkPathPermission($path)) {
                return false;
            }
        }

        return true;
    }

    /**
     * Method to check the allowed wildcards in the branch paths and where they are allowed to inside the path.
     * The only allowed wild cards inside the branch paths are * & ...
     * If the * is present at the start in path, it is not allowed it is allowed in between or at the end.
     * When embedded inside directory name, * is allowed only at the end, not at the start or in between.
     * If the ... is present in between (embedded) in paths, it is not allowed, it is only allowed at the end.
     * @param string $path a string path to a depot location
     * @return bool
     */
    protected function checkAllowedWildCards(string $path): bool
    {
        $segments = explode('/', trim($path, '/'));

        // Disallow '*' at start or end
        if ($segments[0] === '*') {
            $this->error(self::UNSUPPORTED_START);
            return false;
        }

        // Disallow segments that contain '*' as part of text at the start or in between
        // Format allowed -: * or MAIN*
        // Format not allowed -: *MAIN, MA*N
        foreach ($segments as $segment) {
            if ($segment !== '*' && str_contains($segment, '*') && !str_ends_with($segment, '*')) {
                $this->error(self::UNSUPPORTED_TEXT_PART);
                return false;
            }
        }

        // Disallow embedded '...'
        if (preg_match('#\.\.\..+#', $path)) {
            $this->error(self::UNSUPPORTED_WILDCARDS);
            return false;
        }
        return true;
    }

    /**
     * Method to check if the user has write permission or more to the given path.
     * @param string $path a string path to a depot location
     * @return bool
     */
    protected function checkPathPermission(string $path): bool
    {
        // Check if the user has permission to the given path we are checking.
        try {
            $path       = ltrim($path, '-'); // We need to remove minus prefix from exclusionary branch path
            $permission = $this->services->get(IPermissions::PERMISSIONS)->getMaxPathPermission($path);
            switch ($permission) {
                case IPermissions::SUPER:
                case IPermissions::ADMIN:
                case IPermissions::WRITE:
                case IPermissions::OWNER:
                    // User has access.
                    return true;
                    break;
                default:
                    // User has no access.
                    throw new ForbiddenException(self::NO_PERMISSION);
            }
        } catch (CommandException $e) {
            if (str_contains($e->getMessage(), '- must refer to client')) {
                $this->error(self::NO_PERMISSION);
                return false;
            }
            $this->error($e->getMessage());
            return false;
        } catch (ForbiddenException $forbidden) {
            $this->error($forbidden->getMessage());
            return false;
        }
    }

    /**
     * Truncate a path, or some paths, at the first Perforce wildcard; i.e. ... or *
     * @param $paths
     * @return string|string[]|null
     */
    public static function trimWildcards($paths)
    {
        return preg_replace('/[^\/]*(\.[^a-zA-Z0-9]|\*).*/', '', $paths);
    }

    /**
     * Returns list of existing depots in Perforce based on the connection set on this instance.
     * Supports in-memory cache, so 'p4 depots' doesn't run every time this function is called
     * (as long as connection hasn't changed).
     */
    protected function getDepots()
    {
        if ($this->depots === null) {
            $this->depots = array_map('current', $this->getConnection()->run('depots')->getData());
        }

        return $this->depots;
    }
    /**
     * Split the depot file path
     * @param $path
     * @return array
     */
    public static function splitPath($path)
    {
        return preg_split(
            '/([^\/]*[\/])/',
            str_replace('//', '', $path),
            0,
            PREG_SPLIT_NO_EMPTY|PREG_SPLIT_DELIM_CAPTURE
        );
    }
}
