<?php

namespace Jobs\Controller;

use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Filter\Preformat;
use Jobs\Filter\IGetJobs;
use Jobs\Model\IJob;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use P4\Filter\Utf8 as Utf8Filter;
use P4\Spec\Job;
use Exception;
use P4\Spec\PluralAbstract;

/**
 * Trait to carry out common functions for jobs to be used by classes that have services in context
 */
trait JobTrait
{
    /**
     * Iterate the jobs to filter out any invalid characters from the description
     * @param mixed     $jobs   jobs in array from
     * @return mixed jobs in array form
     */
    public function filterDescriptions($jobs)
    {
        $preFormat = new Preformat($this->services, $this->getRequest()->getBaseUrl());
        $utf8      = new Utf8Filter;
        foreach ($jobs as &$job) {
            $job[IJob::FIELD_DESCRIPTION_MARKDOWN] = $preFormat->filter($job[IJob::FIELD_DESCRIPTION]);
            // Filter out any problem characters that may be in the job description
            $job[IJob::FIELD_DESCRIPTION] = $utf8->filter($job[IJob::FIELD_DESCRIPTION]);
        }
        return $jobs;
    }

    /**
     * Get a list of jobs. Supports 'max' and 'filter' query parameters to limit the number of jobs returned
     *
     * Example response:
     *   {
     *       "error": null,
     *       "messages": [],
     *       "data": {
     *           "jobs": [
     *               {
     *                   "job": "job000020",
     *                   "link": "/jobs/job000020",
     *                   "fixStatus": "open",
     *                   "description": "Need Project files\n",
     *                   "descriptionMarkdown": "<span class=\"first-line\">Need Project files</span>"
     *              },
     *              ...
     *              ...
     *       ]
     *   }
     *
     * Example error response:
     *   {
     *       "error": <code>,
     *       "messages": [
     *           {
     *               "code": <code>,
     *               "text": "<message>"
     *           }
     *       ],
     *       "data": null
     *   }
     *
     * @param mixed $fetchByFilter  if specified the value will be used to filter fetched jobs,
     *                              for example 'subsystem=swarm'
     * @return JsonModel
     */
    public function getJobsByFilter($fetchByFilter) : JsonModel
    {
        $errors  = null;
        $jobs    = [];
        $request = $this->getRequest();
        $query   = $request->getQuery();
        try {
            $services = $this->services;
            $p4Admin  = $services->get(ConnectionFactory::P4_ADMIN);
            $jobDao   = $services->get(IDao::JOB_DAO);
            $filter   = $services->get(IGetJobs::FILTER);
            $options  = $query->toArray();
            if ($fetchByFilter) {
                $options[IGetJobs::FILTER_PARAMETER] = $fetchByFilter;
            }
            $filter->setData($options);
            if ($filter->isValid()) {
                $options  = $filter->getValues();
                $options += [
                    Job::FETCH_REVERSE => true,
                    Job::FETCH_BY_FILTER => $options[IGetJobs::FILTER_PARAMETER] ?? null,
                    PluralAbstract::FETCH_MAXIMUM => $options[IGetJobs::FETCH_MAX] ?? 50
                ];
                unset($options[IGetJobs::FETCH_MAX]);
                $jobs = $jobDao->buildJobsArray($jobDao->fetchAll($options, $p4Admin));
                $jobs = $this->filterDescriptions($jobs);
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $errors = $filter->getMessages();
            }
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors !== null) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success(
                [
                    'jobs' => $jobs
                ]
            );
        }
        return $json;
    }
}
