<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Authentication\Service;

use Application\Connection\ConnectionFactory;
use Application\Model\IModelDAO;
use Application\Permissions\Exception\UnauthorizedException;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use Users\Authentication\IHelper;
use P4\Exception;
use Laminas\Mvc\Controller\AbstractRestfulController;

abstract class AbstractLoginAdapter extends AbstractRestfulController
{
    const API_BASE = '/api';
    const ERROR    = 'error';
    const MESSAGES = 'messages';
    const DATA     = 'data';
    const CODE     = 'code';
    const TEXT     = 'text';

    protected $services = null;

    /**
     * IndexController constructor.
     * @param $services
     */
    public function __construct($services)
    {
        $this->services = $services;
    }

    /**
     * Build a message with the code and message text
     * @param mixed  $code      message code
     * @param string $text      message text
     * @return array with self::CODE => $code and self::TEXT => $text
     */
    public function buildMessage($code, string $text)
    {
        return [self::CODE => $code, self::TEXT => $text];
    }

    /**
     * Check if the user is logged in and then return the user object
     * @param string|null $ticket
     * @return JsonModel
     * @throws UnauthorizedException
     * @throws Exception
     */
    protected function currentUser(string $ticket = null)
    {
        $userDao = $this->services->get(IModelDAO::USER_DAO);
        try {
            $p4User = $this->services->get(ConnectionFactory::P4_USER);
            if ($ticket) {
                $p4User->setPassword($ticket);
            }
            if ($p4User->isAuthenticated()) {
                $authUser = $userDao->fetchAuthUser($p4User->getUser(), $p4User);
            } else {
                throw new UnauthorizedException(IHelper::TEXT_NOT_LOGGED_IN, Response::STATUS_CODE_401);
            }
        } catch (\Exception $e) {
            if (strpos($e->getMessage(), IHelper::SERVICE_P4_USER_NOT_CREATED) === 0
                || $e instanceof UnauthorizedException) {
                throw new UnauthorizedException(IHelper::TEXT_NOT_LOGGED_IN, Response::STATUS_CODE_401);
            } else {
                throw new \Exception($e->getMessage(), Response::STATUS_CODE_500);
            }
        }
        return $authUser;
    }

    /**
     * Return Success Response
     * @param mixed $data           Data that will be returned
     * @param array $messages       optional messages, defaults to empty array
     * @return JsonModel
     */
    public function success($data, array $messages = [])
    {
        return $this->buildResponse($data, $messages);
    }

    /**
     * Return Error Response
     * @param  array    $messages       messages
     * @param  mixed    $errorCode      error code
     * @param  mixed    $data           optional data
     * @return JsonModel
     */
    public function error(array $messages, $errorCode, $data = null)
    {
        return $this->buildResponse($data, $messages, $errorCode);
    }

    /**
     * Prepares Json Response
     * @param  array|null   $data           Data that will be returned
     * @param  array        $messages       messages to return
     * @param  int|null     $errorCode      error code
     * @return JsonModel
     */
    private function buildResponse($data, array $messages, $errorCode = null)
    {
        $returnResponse = [
            self::ERROR     => $errorCode,
            self::MESSAGES  => $messages,
            self::DATA      => $data
        ];
        return new JsonModel($returnResponse);
    }
}
