<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Authentication\Helper;

use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Factory\InvokableService;
use Application\Http\SwarmRequest;
use Application\Log\SwarmLogger;
use Application\Model\ServicesModelTrait;
use Application\Session\SwarmSession;
use Authentication\Service\Auth;
use Interop\Container\ContainerInterface;
use Laminas\Http\Response;
use Record\Exception\NotFoundException;

class LogoutHelper implements InvokableService, ILogoutHelper
{
    private $services;

    /**
     * @param ContainerInterface $services
     * @param array|null $options
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services = $services;
    }

    /**
     * @param SwarmRequest $request current request
     * @param Response $response current response
     * @return array|Response|mixed
     * @throws ConfigException
     */
    public function logout(SwarmRequest $request, Response $response)
    {
        /** @var SwarmLogger $logger */
        $logger = $this->services->get(SwarmLogger::SERVICE);
        /** @var SwarmSession $session */
        $session = $this->services->get(self::SESSION);
        /** @var Auth $service */
        $service = $this->services->get(Auth::class);
        // clear identity and all other session data on logout
        // note we need to explicitly restart the session (it's closed by default)
        $session->start();
        $service->clearIdentity();
        $session->destroy(['send_expire_cookie' => true, 'clear_storage' => true]);
        $session->writeClose();
        $redirect = $request->getQuery(self::PARAM_REDIRECT);
        if ($redirect === false || $redirect === 'false') {
            $logger->debug(self::LOG_ID . 'logged out without redirect');
            return $response;
        } else {
            $logger->debug(self::LOG_ID . 'logging out with redirect');
            return $this->customRedirect($request, $response);
        }
    }

    /**
     * @param SwarmRequest $request current request
     * @param Response $response current response
     * @return array|Response|mixed
     * @throws ConfigException
     */
    private function customRedirect(SwarmRequest $request, Response $response)
    {
        $services     = $this->services;
        $logger       = $services->get(SwarmLogger::SERVICE);
        $config       = $services->get(IConfigDefinition::CONFIG);
        $customLogout = ConfigManager::getValue($config, IConfigDefinition::ENVIRONMENT_LOGOUT_URL);
        if ($customLogout) {
            return $this->redirect($customLogout, $response, $request);
        } else {
            $projectDAO = ServicesModelTrait::getProjectDao();
            // If a referrer is set, and it appears to point at us; I want to go there
            $referrer = $request->getServer('HTTP_REFERER');
            if (strpos($referrer, $request->getUriString()) !== false) {
                // Redirecting to here would result in a circular redirect error
                $logger->warn(self::LOG_ID . "Averting circular redirect to  $referrer");
                return $response;
            }
            $scheme = $request->getUri()->getScheme();
            $host   = $scheme ? $scheme : 'http://' . $request->getUri()->getHost();
            // Check the url address for project in it if found check project really exist and then check if its private
            $urlParts  = explode('/', parse_url($referrer)['path']);
            $projectID = sizeof($urlParts) >= 3 && $urlParts[1] === 'projects' ? $urlParts[2] : null;
            if ($projectID) {
                $p4Admin = $services->get('p4_admin');
                try {
                    // May contain encoded UTF-8 characters, so we need to get the decoded id
                    $projectID = urldecode($projectID);
                    // If we are on project settings we want to direct to '/' as you must be logged in
                    // If the project is private we do not want to redirect to the project on logout
                    $referrer = (isset($urlParts[3]) && $urlParts[3] === 'settings')
                    || $projectDAO->fetchById($projectID, $p4Admin)->isPrivate() ? '/' : $referrer;
                } catch (NotFoundException|\InvalidArgumentException $e) {
                    $referrer = '/';
                }
            }
            $logger->debug(self::LOG_ID . 'redirect referrer [' . $referrer . ']');
            if ($referrer !== '/' && stripos($referrer, $host) === 0) {
                return $this->redirect($referrer, $response, $request);
            }
        }
        return $this->redirect($request->getBaseUrl() . '/', $response, $request);
    }

    /**
     * Build an array that can be used in response details with the URL to redirect to for API v10 onwards. For other
     * cases do the redirect as before to preserve backward compatibility
     * @param Response $response the response
     * @param SwarmRequest $request the current request
     * @param mixed $url the url
     * @return mixed a redirected response for v9 or with a key of 'url' and a value of the URL to redirect to
     */
    private function redirect($url, Response $response, SwarmRequest $request)
    {
        $logger = $this->services->get(SwarmLogger::SERVICE);
        $logger->debug(self::LOG_ID . 'log out redirect url [' . $url . ']');
        if ($this->isLegacyRedirect($request)) {
            $response->getHeaders()->addHeaderLine('Location', $url);
            $response->setStatusCode(302);
            return $response;
        } else {
            return [self::URL => $url];
        }
    }

    /**
     * Tests if the request should be redirected (legacy)
     * @param SwarmRequest $request
     * @return bool true if an API call prior to v10 or from a non-API call (web controller)
     */
    public function isLegacyRedirect(SwarmRequest $request) : bool
    {
        $legacyApiVersions = ['v1', 'v1.1', 'v1.2', 'v2', 'v3', 'v4', 'v5', 'v6', 'v7', 'v8', 'v9'];
        $parts             = explode('/', parse_url($request->getUriString())['path']);
        // Api string could be in two positions depending on whether this is single or multiple P4D
        $isApi = sizeof($parts) >= 3 && ($parts[1] === 'api' || $parts[2] === 'api');
        // Version string could be in two positions depending on whether this is single or multiple P4D
        $isLegacyApi = $isApi
            && (in_array($parts[2], $legacyApiVersions) || in_array($parts[3], $legacyApiVersions));
        // Detect a web controller logout, this will not be an API call and will end in 'logout'
        $isLogout = !$isApi && end($parts) === 'logout';
        return $isLegacyApi || $isLogout;
    }
}
