<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Attachments\Listener;

use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\Helper\SwarmBlurHash;
use Attachments\Model\Attachment;
use Attachments\Model\IAttachmentDAO;
use Events\Listener\AbstractEventListener;
use Files\MimeType;
use Laminas\EventManager\Event;
use Exception;
use P4\Connection\Exception\CommandException;
use Record\Lock\Lock;
use Throwable;
use RuntimeException;

class AttachmentsListener extends AbstractEventListener
{
    const LOG_PREFIX = AttachmentsListener::class;

    /**
     * Generate a thumbnail image from a file associated with an attachment. If the file is a simple image type the
     * PHP gd library (if present) will be used for thumbnail generation, otherwise this is delegated to ImageMagick.
     * Once generated this image will be set as an attribute on the depot file associated with the attachment
     * @param Event     $event      the thumbnail event, should have the attachment id as data
     * @return void
     */
    public function generateThumbnail(Event $event)
    {
        $p4Admin = $this->services->get(ConnectionFactory::P4_ADMIN);
        $id      = $event->getParam(Attachment::FIELD_ID);
        $dao     = $this->services->get(IDao::ATTACHMENT_DAO);
        $lock    = new Lock(IAttachmentDAO::LOCK_ATTACHMENT_PREFIX . $id, $this->services->get('p4_admin'));
        $lock->lock();
        try {
            $attachment        = $dao->fetch($id, $p4Admin);
            $isSimpleImageType = MimeType::isSimpleImageType(
                MimeType::getTypeFromName($attachment->get(Attachment::FIELD_NAME))
            );
            if ($isSimpleImageType) {
                $this->generateSimpleImageThumbnail($attachment);
            } else {
                $this->generateImageMagickThumbnail($attachment);
            }
        } catch (Throwable $e) {
            $this->logger->warn(
                sprintf(
                    "%s: Error in thumbnail generation for attachment with id '%s'",
                    self::LOG_PREFIX,
                    $id
                )
            );
            $this->logger->warn($e);
        } finally {
            $lock->unlock();
        }
    }

    /**
     * Use ImageMagick (if available) to create a thumbnail and set it as an attribute on the file in the depot
     * referenced by the attachment
     * @param Attachment $attachment the attachment
     * @return void
     * @throws \ImagickException
     */
    private function generateImageMagickThumbnail(Attachment $attachment)
    {
        $this->logger->info(sprintf("%s: Generate thumbnail with ImageMagick", self::LOG_PREFIX));
        $depotStorage = $this->services->get(IConfigDefinition::DEPOT_STORAGE);
        $attDepotFile = $attachment->get(Attachment::FIELD_DEPOT_FILE);
        $p4Admin      = $this->services->get(ConnectionFactory::P4_ADMIN);
        $fileService  = $this->services->get(Services::FILE_SERVICE);
        $dao          = $this->services->get(IDao::ATTACHMENT_DAO);
        $cacheDir     = $this->getCacheDir();
        // Write the depot file to a temporary location so that ImageMagick can read it. It might have been better
        // to use $image->readImageBlob from getData(1) from the print command result instead of writing to a temp
        // file but Imagick does not seem to recognise the data as valid when this is done, even when using
        // setFormat beforehand
        $file            = $depotStorage->getFile($attDepotFile);
        $tempName        = tempnam($cacheDir, 'depot');
        $cachedDepotFile = $tempName . '.' . $file->getExtension();
        $p4Admin->run('print', ['-o', $cachedDepotFile, $file->getFilespec()]);
        $oldHome = getenv('HOME');
        try {
            // soffice conversions will try to use /var/www as a temp directory with the www-data user. We will
            // temporarily move the directory to avoid permissions issues
            putenv('HOME=' . $cacheDir);
            // By default, Imagick uses the first page of types such as PDF so do not need to specify it
            $image = new \Imagick($cachedDepotFile);
            $image->mergeImageLayers(\Imagick::LAYERMETHOD_FLATTEN);
            $image->setFormat('png');
            // thumbnailImage results in smaller image files than scaleImage
            $image->thumbnailImage(IAttachmentDAO::THUMBNAIL_SIZE, IAttachmentDAO::THUMBNAIL_SIZE);
            $imageBlob = $image->getImageBlob();
            $fileService->attribute(
                $p4Admin,
                IAttachmentDAO::THUMB_ATTRIBUTE,
                bin2hex($imageBlob),
                $attDepotFile,
                ['flags' => ['-e', '-f', '-p']]
            );
            $this->logger->info(sprintf("%s: Generated thumbnail with ImageMagick", self::LOG_PREFIX));
            if (extension_loaded('gd')) {
                $attachment->set(Attachment::FIELD_BLUR, SwarmBlurHash::createImageBlurFromString($imageBlob));
                $dao->save($attachment, $p4Admin);
                $this->logger->info(sprintf("%s: Generated blur from ImageMagick thumbnail", self::LOG_PREFIX));
            } else {
                $this->logger->warn(
                    sprintf("%s: PHP gd extension not found", self::LOG_PREFIX)
                );
            }
        } finally {
            unlink($cachedDepotFile);
            unlink($tempName);
            putenv('HOME=' . $oldHome);
        }
    }

    /**
     * Use PHP gd (if available) to create a thumbnail and set it as an attribute on the file in the depot
     * referenced by the attachment
     * @param Attachment $attachment    the attachment
     * @return void
     */
    private function generateSimpleImageThumbnail(Attachment $attachment)
    {
        $p4Admin      = $this->services->get(ConnectionFactory::P4_ADMIN);
        $dao          = $this->services->get(IDao::ATTACHMENT_DAO);
        $fileService  = $this->services->get(Services::FILE_SERVICE);
        $depotStorage = $this->services->get(IConfigDefinition::DEPOT_STORAGE);
        $depotFile    = $attachment->get(Attachment::FIELD_DEPOT_FILE);
        $this->logger->info(sprintf("%s: Generate thumbnail with gd extension", self::LOG_PREFIX));
        if (extension_loaded('gd')) {
            $image       = imagecreatefromstring($depotStorage->read($depotFile));
            $scaledImage = imagescale($image, IAttachmentDAO::THUMBNAIL_SIZE);
            ob_start();
            imagepng($scaledImage);
            $image = ob_get_clean();
            $fileService->attribute(
                $p4Admin,
                IAttachmentDAO::THUMB_ATTRIBUTE,
                bin2hex($image),
                $depotFile,
                ['flags' => ['-e', '-f', '-p']]
            );
            $this->logger->info(
                sprintf(
                    "%s: File attribute '%s' set on '%s'",
                    self::LOG_PREFIX,
                    IAttachmentDAO::THUMB_ATTRIBUTE,
                    $depotFile
                )
            );
            if ($attachment->get(Attachment::FIELD_BLUR) === null) {
                $attachment->set(Attachment::FIELD_BLUR, SwarmBlurHash::createImageBlurFromImage($scaledImage));
                $dao->save($attachment, $p4Admin);
            }
            imagedestroy($scaledImage);
        } else {
            $this->logger->warn(sprintf("%s: PHP gd extension not found", self::LOG_PREFIX));
        }
    }

    public function cleanUp(Event $event)
    {
        $p4Admin = $this->services->get(ConnectionFactory::P4_ADMIN);
        $id      = $event->getParam(Attachment::FIELD_ID);
        $dao     = $this->services->get(IDao::ATTACHMENT_DAO);
        $lock    = new Lock(IAttachmentDAO::LOCK_ATTACHMENT_PREFIX . $id, $this->services->get('p4_admin'));
        $lock->lock();
        try {
            $attachment = $dao->fetch($id, $p4Admin);
            if (!$attachment->getReferences()) {
                $attachment->delete();
            }
        } catch (Exception $e) {
            $this->logger->err($e);
        } finally {
            $lock->unlock();
        }
    }

    /**
     * Get a location to use as a temporary store for image files
     * @return string   directory location, the Swarm install path cache/attachments
     */
    private function getCacheDir() : string
    {
        $dir = DATA_PATH . '/cache/attachments';
        if (!is_dir($dir)) {
            @mkdir($dir, 0700, true);
        }
        if (!is_writable($dir)) {
            @chmod($dir, 0700);
        }
        if (!is_dir($dir) || !is_writable($dir)) {
            throw new RuntimeException(
                "Cannot write to cache directory ('" . $dir . "'). Check permissions."
            );
        }
        return $dir;
    }
}
