<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace AiAnalysis\Service;

use AiAnalysis\Filter\IAiAnalysis;
use AiAnalysis\Helper\IAiAnalysisHelper;
use Application\Config\IConfigDefinition;
use Application\Log\SwarmLogger;
use OpenAI\Exceptions\TransporterException;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Component\HttpClient\Psr18Client;
use Laminas\Http\Response;
use OpenAI;

class OpenAIAdapter extends AbstractAiAdapter implements AiServiceInterface
{

    const LOG_PREFIX = OpenAIAdapter::class;

    /**
     * performAI, note this uses the content body and ignores any data provided
     * Login to Swarm using the credentials provided
     *
     * @param mixed $data
     * @return array
     */
    public function executeAIRequest(mixed $data): array
    {
        $fileName = ($data[IAiAnalysis::FILE_NAME]) ? 'of the file: ' . $data[IAiAnalysis::FILE_NAME] : '';
        $payload  = [
                'model' => $data[IConfigDefinition::AI_MODEL],
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => $data[IConfigDefinition::AI_PACKAGE_TYPE] . ' ' . $fileName . ' ' .
                            $data[IAiAnalysisHelper::CONTENT_TO_ANALYZE]
                    ],
                ],
        ];
        return $this->performChatCompletion($payload, true);
    }

    /**
     * This function will work to send the call to AI Vendor and fetch improved comment
     * @param mixed $data
     * @return array
     */
    public function getImprovedCommentByAi(mixed $data): array
    {
        $payload =  [
                'model' => $data[IConfigDefinition::AI_MODEL],
                'messages' => [
                    [
                        'role' => 'user',
                        'content' => $data['statement']
                    ],
                ],
            ];
        return $this->performChatCompletion($payload, true);
    }

    /**
     * A single, private helper to handle all AI chat requests.
     * This centralizes the try-catch logic and response formatting.
     *
     * @param array $payload The payload for the OpenAI API call.
     * @param bool $validate Whether to perform extra validation on the result.
     * @return array The structured response.
     */
    private function performChatCompletion(array $payload, bool $validate): array
    {
        $logger = $this->services->get(SwarmLogger::SERVICE);
        try {
            $httpClient = new Psr18Client(
                HttpClient::create(
                    ['timeout' => $this->getTimeout(),]
                )
            );
            $client     = ($this->getApiUrl()) ? OpenAI::factory()
                ->withApiKey($this->getApiKey())
                ->withHttpClient($httpClient)
                ->withBaseUri($this->getApiUrl())
                ->make() : OpenAI::factory()
                ->withApiKey($this->getApiKey())
                ->withHttpClient($httpClient)
                ->make();

            $result = $client->chat()->create($payload);

            if ($validate && !$this->validateResult($result)) {
                $logger->trace(
                    sprintf(
                        "[%s]: Failed at content generation in open AI adapter",
                        self::LOG_PREFIX
                    )
                );
                return [
                    IAiAnalysisHelper::CONTENT_GENERATED => '',
                    self::ERROR => is_object($result) && property_exists($result, 'error')
                        ? $result->error : 'Response not received from AI Vendor',
                    self::CODE => is_object($result) && property_exists($result, 'code')
                        ? $result->code : Response::STATUS_CODE_500
                ];
            }

            $logger->trace(sprintf("[%s]: Content generation successful.", self::LOG_PREFIX));
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => $result->choices[0]->message->content,
                self::ERROR => null,
                self::CODE => Response::STATUS_CODE_200
            ];
        } catch (TransporterException $e) {
            $logger->debug(
                sprintf(
                    "[%s]: Connection error at Open AI Adapter: %s",
                    self::LOG_PREFIX,
                    $e->getMessage()
                )
            );
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                self::ERROR => "Connection error or timeout occurred with " . $e->getMessage(),
                self::CODE => Response::STATUS_CODE_408,
            ];
        } catch (\Exception | \Throwable $e) {
            $logger->debug(sprintf("[%s]: General error at Open AI Adapter: %s", self::LOG_PREFIX, $e->getMessage()));
            $statusCode = $e->getCode();
            return [
                IAiAnalysisHelper::CONTENT_GENERATED => null,
                self::ERROR => $e->getMessage(),
                self::CODE => $statusCode === 0 ? Response::STATUS_CODE_500 : $statusCode,
            ];
        }
    }
}
