<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Activity\Listener;

use Activity\Model\Activity;
use Application\Config\IConfigDefinition as IConDef;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Events\Listener\AbstractEventListener;
use P4\Key\Key;
use P4\Spec\Group as GroupSpec;
use P4\Spec\Job;
use Record\Key\AbstractKey;
use Reviews\Model\Review;
use Users\Model\Config;
use Laminas\EventManager\Event;
use P4\Spec\Definition as SpecDefinition;

class ActivityListener extends AbstractEventListener
{
    /**
     * Connect to all tasks and write activity data we do this late (low-priority) so all handlers have
     * a chance to influence the activity model.
     * @param Event $event
     * @throws \Exception
     */
    public function createActivity(Event $event)
    {
        $config   = $this->services->get(IConDef::CONFIG);
        $priority = $config[IConDef::LOG][IConDef::PRIORITY];
        $this->logMsg("Activity Listener starting here", $priority);

        $model = $event->getParam('activity');
        if (!$model instanceof Activity) {
            return;
        }

        $activityConfig = $config[IConDef::ACTIVITY];
        // We don't want to build any activity.
        $skipActivity = $activityConfig[IConDef::SKIP_ACTIVITY] ?? false;
        if ($skipActivity) {
            $this->logMsg("Skipping All activity", $priority);
            return;
        }

        // ignore 'quiet' events.
        $data  = (array) $event->getParam('data') + ['quiet' => null];
        $quiet = $event->getParam('quiet', $data['quiet']);
        if ($quiet === true || in_array('activity', (array) $quiet)) {
            $this->logMsg("Activity is set to quiet", $priority);
            return;
        }

        // don't record activity by users we ignore.
        $ignore = isset($activityConfig[IConDef::IGNORED_USERS])
            ? (array) $activityConfig[IConDef::IGNORED_USERS]
            : [];
        $userID = $model->get('user');
        if (in_array($userID, $ignore)) {
            $this->logMsg(sprintf("User %s is in ignore_users list", $userID), $priority);
            return;
        }

        $streams = (array) $model->getStreams();

        // all activity should appear in the activity streams
        // of the user that initiated the activity.
        $streams[] = 'user-'     . $userID;
        $streams[] = 'personal-' . $userID;

        $skipFollowers = $activityConfig[IConDef::SKIP_FOLLOWERS] ?? false;
        // add anyone who follows the user that initiated this activity
        $p4Admin   = $this->services->get(ConnectionFactory::P4_ADMIN);
        $followers = !$skipFollowers ? Config::fetchFollowerIds($userID, 'user', $p4Admin) : [];

        $skipGroups   = $activityConfig[IConDef::SKIP_GROUPS] ?? false;
        $skipProjects = $activityConfig[IConDef::SKIP_PROJECTS] ?? false;
        if (!$skipProjects) {
            // projects that are affected should also get the activity
            // and, by extension, project members should see it too.
            if ($model->getProjects()) {
                $projectDAO = $this->services->get(IDao::PROJECT_DAO);
                $projects   = $projectDAO->fetchAll(
                    [AbstractKey::FETCH_BY_IDS => array_keys($model->getProjects())],
                    $p4Admin
                );
                foreach ($projects as $project) {
                    $projectID = $project->getId();
                    $streams[] = 'project-' . $projectID;
                    $followers = array_merge(
                        $followers,
                        (array) ($skipGroups ? $project->getMembers() : $project->getAllMembers(false))
                    );
                    $this->logMsg(
                        sprintf("Added all members for project with id [%s] as followers", $projectID),
                        $priority
                    );
                }
            }
        } else {
            $this->logMsg("Skipping Projects activity", $priority);
        }

        if (!$skipGroups) {
            // ensure groups the user is member of get the activity
            if ($userID) {
                $groupFetchOptions = [
                    GroupSpec::FETCH_BY_USER => $userID,
                    GroupSpec::FETCH_INDIRECT => true
                ];
                $groups            = $this->services->get(IDao::GROUP_DAO)->fetchAll($groupFetchOptions, $p4Admin);
                $this->logMsg(sprintf("Adding the activity for user %s's groups", $userID), $priority);
                $streams = array_merge(
                    $streams,
                    (array)preg_filter('/^/', 'group-', $groups->invoke('getId'))
                );
            }
        } else {
            $this->logMsg("Skipping Groups activity", $priority);
        }

        $skipReview = $activityConfig[IConDef::SKIP_REVIEW] ?? false;
        if (!$skipReview) {
            // activity related to a review should include review participants
            // and should appear in the activity stream for the review itself
            $review = $event->getParam('review');
            if ($review instanceof Review) {
                $followers = array_merge($followers, (array)$review->getParticipants());
                $streams[] = 'review-' . $review->getId();
            }
        } else {
            $this->logMsg("Skipping Review activity", $priority);
        }
        // Now add the followers to the model.
        $model->addFollowers($followers);
        $this->logMsg(sprintf("Adding Personal Followers, %s",  implode(", ", $followers)), $priority);
        // this all the followers that have been added by other listeners.
        $modelFollowers = $model->getFollowers();
        // we don't want the group, so we filter them out.
        if ($skipGroups) {
            $this->logMsg(
                sprintf("The Followers before filtering to users only, %s", implode(", ", $modelFollowers)),
                $priority
            );

            $services       = $this->services;
            $userDAO        = $services->get(IDao::USER_DAO);
            $modelFollowers = $userDAO->filter($modelFollowers, $p4Admin, []);
        }
        $streams = array_merge(
            $streams,
            (array) preg_filter('/^/', 'personal-', $modelFollowers)
        );
        $this->logMsg(sprintf("Setting the streams, %s", implode(", ", $streams)), $priority);
        $model->setStreams($streams);

        try {
            $model->setConnection($p4Admin)->save();
        } catch (\Exception $e) {
            $this->logger->err($e);
        }
    }

    /**
     * Connect to worker startup to check if we need to prime activity
     * data (i.e. this is a first run against an existing server).
     * @param Event $event
     * @throws \P4\Counter\Exception\NotFoundException
     */
    public function prePopulateActivity(Event $event)
    {
        $manager = $this->services->get('queue');
        $events  = $manager->getEventManager();
        if ($event->getParam('slot') !== 1) {
            return;
        }

        // if we already have an event counter, nothing to do.
        $p4Admin = $this->services->get('p4_admin');
        if (Key::exists(Activity::KEY_COUNT, $p4Admin)) {
            return;
        }

        // initialize count to zero so we exit early next time.
        $key = new Key($p4Admin);
        $key->setId(Activity::KEY_COUNT)
            ->set(0);

        // looks like we're going to do the initial import, tie up as many
        // worker slots as we can to minimize concurrency/out-of-order issues
        // (if other workers were already running, we won't get all the slots)
        // release these slots on shutdown - only really needed when testing
        $slots = [];
        while ($slot = $manager->getWorkerSlot()) {
            $slots[] = $slot;
        }
        $events->attach(
            'worker.shutdown',
            function () use ($slots, $manager) {
                foreach ($slots as $slot) {
                    $manager->releaseWorkerSlot($slot);
                }
            }
        );

        // grab the last 10k changes and get ready to queue them.
        $queue   = [];
        $changes = $p4Admin->run('changes', ['-m10000', '-s', 'submitted']);
        foreach ($changes->getData() as $change) {
            $queue[] = [
                'type' => 'commit',
                'id'   => $change['change'],
                'time' => (int) $change['time']
            ];
        }

        // grab the last 10k jobs and get ready to queue them.
        // note, jobspec is mutable so we get the date via its code
        try {
            // use modified date field if available, falling-back to the default date field.
            // often this will be the same field, by default the date field is a modified date.
            $job  = new Job($p4Admin);
            $spec = SpecDefinition::fetch('job', $p4Admin);
            $date = $job->hasModifiedDateField()
                ? $job->getModifiedDateField()
                : $spec->fieldCodeToName(104);

            $jobs = $p4Admin->run('jobs', ['-m10000', '-r']);
            foreach ($jobs->getData() as $job) {
                if (isset($job[$date])) {
                    $queue[] = [
                        'type' => 'job',
                        'id'   => $job['Job'],
                        'time' => strtotime($job[$date])
                    ];
                }
            }
        } catch (\Exception $e) {
            $this->services->get('logger')->err($e);
        }

        // sort items by time so they are processed in order
        // if other workers are already pulling tasks from the queue.
        usort(
            $queue,
            function ($a, $b) {
                return $a['time'] - $b['time'];
            }
        );

        // we don't want to duplicate activity
        // it's possible there are already tasks in the queue
        // (imagine the trigger was running, but the workers were not),
        // if there are >10k abort; else fetch them so we can skip them.
        if ($manager->getTaskCount() > 10000) {
            return;
        }
        $skip = [];
        foreach ($manager->getTaskFiles() as $file) {
            $task = $manager->parseTaskFile($file);
            if ($task) {
                $skip[$task['type'] . ',' . $task['id']] = true;
            }
        }

        // again, we don't want to duplicate activity
        // if there is any activity at this point, abort.
        if (Key::fetch(Activity::KEY_COUNT, $p4Admin)->get()) {
            return;
        }

        // add jobs and changes to the queue
        foreach ($queue as $task) {
            if (!isset($skip[$task['type'] . ',' . $task['id']])) {
                $manager->addTask($task['type'], $task['id'], null, $task['time']);
            }
        }
    }
}
