<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */

namespace Users\Service;

use Activity\Model\Activity;
use Application\Cache\AbstractCacheService;
use Application\Config\ConfigException;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Factory\InvokableService;
use Application\I18n\TranslatorFactory;
use Application\Log\SwarmLogger;
use Application\Model\IModelDAO;
use Events\Listener\ListenerFactory;
use Groups\Model\Group;
use Groups\Model\IGroup;
use Interop\Container\ContainerInterface;
use Laminas\EventManager\Event;
use P4\Exception;
use Projects\Model\IProject;
use Projects\Model\Project as ProjectModel;
use Psr\Container\ContainerExceptionInterface;
use Psr\Container\NotFoundExceptionInterface;
use Queue\Manager;
use Redis\RedisService;
use TestIntegration\Model\ITestDefinition;
use TestIntegration\Model\TestDefinition;
use Users\Model\IConfig;
use Users\Model\IUser;
use Users\Model\Config;
use Workflow\Model\IWorkflow;
use Users\Model\User;
use Workflow\Model\Workflow;

/**
 * Service to implement clean up of deleted users.
 *
 * @package Users\Service
 */
class CleanUpDeletedUsers implements ICleanUpDeletedUsers, InvokableService
{
    public const CLEANUP_QUEUED_MSG  = 'Your request to clean up %s for the deleted user %s has been queued';
    public const USER_NOT_EXISTS_MSG = 'The requested user %s does not exist in any %s';
    public const ENTITIES            = [
        ListenerFactory::TEST_DEFINITION,
        ListenerFactory::WORKFLOW,
        ListenerFactory::PROJECT,
        ListenerFactory::GROUP,
        ListenerFactory::FOLLOWERS,
    ];
    public const CACHE_KEY_PREFIX    = IUser::USER . RedisService::SEPARATOR;
    private $services;
    private $logger;
    private $p4Admin;
    private $adminUser;
    private $events;

    /**
     * CleanUpDeletedUsers constructor.
     *
     * @param ContainerInterface $services
     * @param array|null         $options
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services  = $services;
        $this->logger    = $services->get(SwarmLogger::class);
        $this->p4Admin   = $services->get(ConnectionFactory::P4_ADMIN);
        $this->adminUser = $this->p4Admin ? $this->p4Admin->getUser() : '';
        $this->events    = $this->services->get('queue')->getEventManager();
    }

    /**
     * Handle cleanup of deleted users data
     *
     * @param $id User id being deleted
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function handleCleanUpOfDeletedUsersData($id): void
    {
        $this->logger->trace(self::class." processing cleanup of deleted user for id $id");
        if (empty($id)) {
            $this->logger->debug(self::class." id $id is empty exit early");
            return;
        }

        // Config isn't here as it comes after the followers.
        $queue     = $this->services->get(Manager::SERVICE);
        $queuedMsg = '%s Queued task for cleanup of deleted users from %s';
        foreach (self::ENTITIES as $entity) {
            $queue->addTask(
                ListenerFactory::CLEANUP_DELETED_USER . '.' . $entity,
                $id,
                ['type' => $entity] /* type - Entity type - workflow, testdefinition, project etc. */
            );
            $this->logger->trace(sprintf($queuedMsg, ListenerFactory::class, $entity));
        }

        $this->services->get(AbstractCacheService::CACHE_SERVICE)->delete(self::CACHE_KEY_PREFIX . $id);
    }

    /**
     * Handle cleanup of config for deleted user
     * @param $id
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws \Record\Exception\NotFoundException
     */
    public function cleanUpConfigForDeletedUsers($id): void
    {
        $this->logger->trace(self::class." processing Config cleanup of deleted user for id $id");

        $config = Config::fetch($id, $this->services->get(ConnectionFactory::P4_ADMIN));

        if ($config) {
            $config->setUserSettings(null);
            $config->setUserNotificationSettings(null);
            try {
                $config->save();
                $this->logger->trace(self::class." Removed user settings $id");
            } catch (Exception $exception) {
                $this->logger->trace(
                    sprintf(
                        '%s - Cannot remove config for user %s due to following exception: %s',
                        self::class,
                        $id,
                        $exception->getMessage()
                    )
                );
            }
        }
    }

    /**
     * Cleanup projects for the deleted user.
     *
     * @param Event $event Event to cleanup deleted users from projects
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function cleanUpProjectsForDeletedUsers(Event $event): void
    {
        $id = $event->getParam('id');
        $this->logger->trace(self::class." processing projects cleanup of deleted user for id $id");

        $projectDAO = $this->services->get(IDao::PROJECT_DAO);
        $projects   = $projectDAO->fetchAll(
            [ProjectModel::FETCH_BY_ALL_MEMBERSHIP_LEVELS => $id, IModelDAO::FETCH_SUMMARY => []],
            $this->p4Admin
        );
        if ($projects->count() === 0) {
            return;
        }
        /*
        * Different membership levels of the user considered are owner, member, follower, default reviewer
        * moderator for branches, default reviewers for branches.
        */
        foreach ($projects as $project) {
            $projectID     = $project->getId();
            $isUserRemoved = false;
            $projectName   = $project->getName();
            $role          = [];
            // Owner's block
            if ($project->isOwner($id)) {
                $project->removeOwner($id);
                $this->logger->trace(self::class . " removing $id from the list of project owners.");
                $isUserRemoved = true;
                $role[]        = IProject::OWNERS;
            }
            // Member's block
            if ($project->isMember($id)) {
                $project->removeMember($id);
                // If the deleted user is sole member of a project, add the Swarm user as the member.
                if ($project->getMembers() === []) {
                    $project->setMembers([$this->adminUser]);
                    $projectDAO->save($project);
                } else {
                    $isUserRemoved = true;
                }
                $this->logger->trace(self::class . " removing $id from the list of project members.");
                $role[] = IProject::MEMBERS;
            }
            // Default reviewer's block
            if ($project->isDefaultReviewer($id)) {
                $project->removeDefaultReviewer($id);
                $this->logger->trace(self::class . " removing $id from the list of default reviewers of project.");
                $isUserRemoved = true;
                $role[]        = 'default reviewers';
            }

            // Branch moderator's block
            if ($project->isModerator($id)) {
                $project->removeBranchModerator($id);
                $this->logger->trace(self::class . " removing $id from the list of branch moderators of project.");
                $isUserRemoved = true;
                $role[]        = IProject::MODERATORS;
            }
            // Branch default reviewer's block
            if ($project->isDefaultReviewerForBranches($id)) {
                $project->removeBranchDefaultReviewer($id);
                $this->logger->trace(
                    self::class . " removing $id from the list of branch default reviewers of project."
                );
                $isUserRemoved = true;
                $role[]        = 'branch default reviewers';
            }
            if ($isUserRemoved) {
                $projectDAO->save($project);
            }
            if ($role) {
                $this->createCleanupActivity(
                    $event,
                    ListenerFactory::PROJECT,
                    $projectName,
                    implode(', ', $role),
                    [IProject::PROJECT. '-' . $projectName]
                );
            }
        }
    }

    /**
     * Cleanup groups for the deleted user.
     *
     * @param Event $event Event to cleanup deleted users from groups
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function cleanUpGroupsForDeletedUsers(Event $event): void
    {
        $id = $event->getParam('id');
        $this->logger->trace(self::class." processing groups cleanup of deleted user for id $id");

        $groupDAO = $this->services->get(IDao::GROUP_DAO);
        $groups   = $groupDAO->fetchAll(
            [Group::FETCH_BY_USER => $id, Group::FETCH_BY_USER_MODE => Group::USER_MODE_ALL],
            $this->p4Admin
        );

        foreach ($groups as $group) {
            $groupId   = $group->getId();
            $owners    = $group->getOwners() ?? [];
            $members   = $group->getUsers() ?? [];
            $groupName = $group->getConfig()->getName();
            $role      = [];
            if (in_array($this->adminUser, $owners) || $this->p4Admin->isSuperUser()) {
                $isOwner = array_search($id, $owners);
                if ($isOwner !== false) {
                    array_splice($owners, $isOwner, 1);
                    $owners === [] ? $group->setOwners([$this->adminUser]) : $group->setOwners($owners);
                    $this->logger->trace(
                        self::class." Removing user $id from group '$groupId' owner's list " .
                        json_encode($owners, true)
                    );
                    $role[] = IGroup::OWNERS;
                }
                $isMember = array_search($id, $members);
                if ($isMember !== false) {
                    array_splice($members, $isMember, 1);
                    $members === [] && $owners === [] ? $group->setUsers([$this->adminUser]) :
                        $group->setUsers($members);
                    $this->logger->trace(
                        self::class." Removing user $id from '$groupId' group's  member list" .
                        json_encode($members, true)
                    );
                    $role[] = IGroup::MEMBERS;
                }
                try {
                    $editAsOwner = in_array($this->adminUser, $owners);
                    $groupDAO->save($group, $editAsOwner, false);
                    $this->logger->trace(self::class." Removed user $id from group $groupId");
                    $this->createCleanupActivity(
                        $event,
                        ListenerFactory::GROUP,
                        $groupName,
                        implode(', ', $role),
                        [IGroup::GROUP . '-' . $groupName]
                    );
                } catch (Exception $exception) {
                    $this->logger->trace(
                        sprintf(
                            '%s - Cannot remove user %s from group %s due to following exception: %s',
                            self::class,
                            $id,
                            $groupId,
                            $exception->getMessage()
                        )
                    );
                }
            } else {
                $this->logger->trace(
                    sprintf(
                        '%s - Cannot remove user %s from group %s due 
                        to Swarm user permissions being only Admin and is not an owner / super user.',
                        self::class,
                        $id,
                        $groupId
                    )
                );
            }
        }
    }

    /**
     * Cleanup workflow for deleted users.
     *
     * @param Event $event Event to cleanup deleted users from workflow
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function cleanUpWorkflowForDeletedUsers(Event $event): void
    {
        $id = $event->getParam('id');
        $this->logger->trace(self::class." processing workflows cleanup of deleted user for id $id");

        $workflowDao = $this->services->get(IDao::WORKFLOW_DAO);
        $workflows   = $workflowDao->fetchAll([Workflow::FETCH_BY_OWNER => $id], $this->p4Admin);
        /** @var Workflow $workflow */
        foreach ($workflows as $workflow) {
            $workflowId = $workflow->getId();
            $owners     = $workflow->getOwners();
            $isOwner    = array_search($id, $owners);
            if ($isOwner === false) {
                continue;
            }
            array_splice($owners, $isOwner, 1);
            $owners === [] ? $workflow->setOwners([$this->adminUser]) : $workflow->setOwners($owners);
            try {
                $workflowDao->save($workflow);
                $this->logger->trace(self::class." Removed user $id from workflow $workflowId");
                $this->createCleanupActivity(
                    $event,
                    ListenerFactory::WORKFLOW,
                    $workflow->getName(),
                    IWorkflow::OWNERS
                );
            } catch (Exception $exception) {
                $this->logger->trace(
                    sprintf(
                        '%s - Cannot remove user %s from workflow %s due to following exception: %s',
                        self::class,
                        $id,
                        $workflowId,
                        $exception->getMessage()
                    )
                );
            }
        }
    }

    /**
     * Cleanup test-defintions for deleted users.
     * @param Event $event Event to cleanup deleted users from test-defintions
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    public function cleanUpTestDefinitionsForDeletedUsers(Event $event): void
    {
        $id = $event->getParam('id');
        $this->logger->trace(self::class." processing test definitions cleanup of deleted user for id $id");

        $testDefinitionDao = $this->services->get(IDao::TEST_DEFINITION_DAO);
        $testDefinitions   = $testDefinitionDao->fetchAll([TestDefinition::FETCH_BY_OWNER => $id], $this->p4Admin);
        /** @var TestDefinition $testDefinitions */
        foreach ($testDefinitions as $testDefinition) {
            $testDefinitionId = $testDefinition->getId();
            $owners           = $testDefinition->getOwners();
            $isOwner          = array_search($id, $owners);
            if ($isOwner === false) {
                continue;
            }
            array_splice($owners, $isOwner, 1);
            $owners === [] ? $testDefinition->setOwners([$this->adminUser]) : $testDefinition->setOwners($owners);
            try {
                $testDefinitionDao->save($testDefinition);
                $this->logger->trace(self::class." Removed user $id from test definition $testDefinitionId");
                $this->createCleanupActivity(
                    $event,
                    ListenerFactory::TEST_DEFINITION,
                    $testDefinition->getName(),
                    ITestDefinition::FIELD_OWNERS
                );
            } catch (Exception $exception) {
                $exceptionMessage = $exception->getMessage();
                $this->logger->trace(
                    sprintf(
                        '%s: Cannot remove user %s from test definition %s due to following exception %s',
                        self::class,
                        $id,
                        $testDefinitionId,
                        $exceptionMessage
                    )
                );
            }
        }
    }

    /**
     * Cleans up deleted users from the specified entity.
     *
     * @param string      $userId User Id being cleaned up
     * @param string|null $entity Entities - workflow, test definition, project, group, config
     * @return array|string
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws Exception
     */
    public function cleanUpDeletedUsersFromEntity(string $userId, ?string $entity): array | string
    {
        return match ($entity) {
            ListenerFactory::TEST_DEFINITION,
            ListenerFactory::WORKFLOW,
            ListenerFactory::GROUP,
            ListenerFactory::PROJECT,
            ListenerFactory::FOLLOWERS => $this->handleEntityCleanup(
                $this->isUserInEntity($userId, $entity),
                $userId,
                $entity,
            ),
            default => $this->handleCleanupOfDeletedUsersFromAllEntities($userId),
        };
    }

    /**
     * Function that handles the cleanup of the concerned deleted user from all entities.
     *
     * @param string $userId User id of the deleted user
     */
    private function handleCleanupOfDeletedUsersFromAllEntities(string $userId): array | string
    {
        try {
            $messages = [];
            foreach (self::ENTITIES as $type) {
                $messages[] = $this->handleEntityCleanup($this->isUserInEntity($userId, $type), $userId, $type);
            }
            return array_combine(self::ENTITIES, $messages);
        } catch (\Exception | \Throwable $exception) {
            return $exception->getMessage();
        }
    }

    /**
     * Function that handles the cleanup of deleted user from the concerned entity.
     *
     * @param bool   $userExistsInEntity Flag indicating user existence in the entity
     * @param string $userId             User id being cleaned up
     * @param string $entity             Entity from which the user is being cleaned up
     * @return string
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    private function handleEntityCleanup(
        bool   $userExistsInEntity,
        string $userId,
        string $entity
    ): string {
        if ($userExistsInEntity) {
            $queue = $this->services->get(Manager::SERVICE);
            $queue->addTask(
                ListenerFactory::CLEANUP_DELETED_USER . '.' . $entity,
                $userId,
                ['type' => $entity]
            );
        }
        $translator = $this->services->get(TranslatorFactory::SERVICE);
        return $userExistsInEntity ? $translator->t(self::CLEANUP_QUEUED_MSG, [$entity, $userId]) :
            $translator->t(self::USER_NOT_EXISTS_MSG, [$userId, $entity]);
    }

    /**
     * Checks if a user exists within a specified entity.
     *
     * @param string $userId user id
     * @param string $entity Entity - workflow, test definition, project, group, follower etc
     * @return bool
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws Exception
     */
    private function isUserInEntity(string $userId, string $entity): bool
    {
        return match ($entity) {
            ListenerFactory::WORKFLOW => !empty(
                $this->services->get(IDao::WORKFLOW_DAO)
                ->fetchAll([TestDefinition::FETCH_BY_OWNER => $userId], $this->p4Admin)->toArray()
            ),
            ListenerFactory::TEST_DEFINITION => !empty(
                $this->services->get(IDao::TEST_DEFINITION_DAO)
                ->fetchAll([TestDefinition::FETCH_BY_OWNER => $userId], $this->p4Admin)->toArray()
            ),
            ListenerFactory::GROUP => !empty(
                $this->services->get(IDao::GROUP_DAO)->fetchAll(
                    [Group::FETCH_BY_USER => $userId, Group::FETCH_BY_USER_MODE => Group::USER_MODE_ALL],
                    $this->p4Admin
                )->toArray()
            ),
            ListenerFactory::PROJECT => !empty(
                $this->services->get(IDao::PROJECT_DAO)->fetchAll(
                    [ProjectModel::FETCH_BY_ALL_MEMBERSHIP_LEVELS => $userId, IModelDAO::FETCH_SUMMARY => []],
                    $this->p4Admin
                )->toArray()
            ),
            ListenerFactory::FOLLOWERS => $this->hasFollowData($userId),
            default => false,
        };
    }

    /**
     * Get the follow data (followers/following) of the concerned users.
     *
     * @param string $id User id being cleaned up.
     * @return bool
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    private function hasFollowData(string $id): bool
    {
        $user = $this->services->get(IUser::USER);
        $user->setId($id);
        $user->setEmail($id.'@tmp');
        $user->setFullName($id);
        $delConfig         = $user->getConfig();
        $followers         = $delConfig->getFollowers();
        $following         = $delConfig->getFollows();
        $followingUsers    = $following[IConfig::USER_FOLLOW_TYPE] ?? [];
        $followingProjects = $following[IConfig::PROJECT_FOLLOW_TYPE] ?? [];

        return !empty($followers) || !empty($followingUsers) || !empty($followingProjects);
    }

    /**
     * We should clean up the followers for the user being passed in.
     *
     * @param Event $event
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     * @throws ConfigException
     */
    public function cleanUpFollowersForDeletedUsers(Event $event): void
    {
        $id = $event->getParam('id');
        $this->logger->trace(self::class." processing followers cleanup of deleted user for id $id");
        $this->unfollowProcess($id, $event);
        $config             = $this->services->get(IConfigDefinition::CONFIG);
        $cleanupConfigTimer = (int) ConfigManager::getValue($config, IConfigDefinition::USERS_CLEANUP_CONFIG_TIMER, 10);
        $queue              = $this->services->get(Manager::SERVICE);
        $queue->addTask(
            ListenerFactory::CLEANUP_DELETED_USER . '.' . ListenerFactory::CONFIG,
            $id,
            ['type' => ListenerFactory::CONFIG],
            time() + $cleanupConfigTimer*60,
        );
        $queuedMsg = '%s Queued task for cleanup of deleted users for %s';
        $this->logger->trace(sprintf($queuedMsg, ListenerFactory::class, ListenerFactory::CONFIG));
    }

    /**
     * This function is going to take a userid and process all the users and projects it is following and remove then.
     * It's also going to deal with the users following this user.
     *
     * @param string $id The user ID that we want to use to remove their followers
     * @param Event  $event
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    protected function unfollowProcess(string $id, Event $event): void
    {
        $this->logger->trace(self::class." processing unfollow cleanup of deleted user for id $id");
        try {
            $userDao = $this->services->get(IDao::USER_DAO);
            // We need to create a temp module so we can access the config for that model.
            $user = new User($this->p4Admin);
            $user->setId($id);
            $user->setEmail($id.'@tmp');
            $user->setFullName($id);
            $delConfig         = $user->getConfig();
            $followers         = $delConfig->getFollowers();
            $following         = $delConfig->getFollows();
            $followingUsers    = $following[IConfig::USER_FOLLOW_TYPE] ?? [];
            $followingProjects = $following[IConfig::PROJECT_FOLLOW_TYPE] ?? [];
            $this->logger->trace(sprintf('%s: Starting the unfollow of users for user %s', self::class, $id));
            foreach ($followingUsers as $followingUser) {
                try {
                    $delConfig->removeFollow($followingUser);
                    $this->createCleanupActivity(
                        $event,
                        IUser::USER,
                        $followingUser,
                        'followers',
                        [IUser::USER . '-' . $followingUser]
                    );
                } catch (Exception $error) {
                    $this->logger->debug(
                        sprintf('%s: failed to remove user %s for user %s', self::class, $followingUser, $id)
                    );
                    $this->logger->trace(
                        sprintf('%s: Remove followed user error: %s', self::class, $error->getMessage())
                    );
                }
            }

            $this->logger->trace(sprintf('%s: Starting the unfollow of project for user %s', self::class, $id));
            foreach ($followingProjects as $followingProject) {
                try {
                    $delConfig->removeFollow($followingProject, IConfig::PROJECT_FOLLOW_TYPE);
                    $this->createCleanupActivity(
                        $event,
                        IProject::PROJECT,
                        $followingProject,
                        'followers',
                        [IProject::PROJECT. '-' . $followingProject]
                    );
                } catch (Exception $error) {
                    $this->logger->debug(
                        sprintf('%s: failed to remove project %s for user %s', self::class, $followingProject, $id)
                    );
                    $this->logger->trace(
                        sprintf('%s: Remove followed project error: %s', self::class, $error->getMessage())
                    );
                }
            }

            $this->logger->trace(sprintf('%s: Starting the removal of followers for user %s', self::class, $id));
            foreach ($followers as $followerUser) {
                try {
                    $follower       = $userDao->fetchById($followerUser);
                    $followerConfig = $follower->getConfig();
                    $followerConfig->removeFollow($id);
                    $followerConfig->save();
                    $this->createCleanupActivity(
                        $event,
                        IUser::USER,
                        $followerUser,
                        'following',
                        [IUser::USER. '-' . $followerUser]
                    );
                } catch (Exception $error) {
                    $this->logger->debug(
                        sprintf('%s: failed to remove user %s from following user %s', self::class, $id, $followerUser)
                    );
                    $this->logger->trace(
                        sprintf('%s: Remove following error: %s', self::class, $error->getMessage())
                    );
                }
            }
            $delConfig->save();
        } catch (Exception $error) {
            // Do nothing right now.
            $this->logger->trace(
                sprintf(
                    '%s: We ran into a problem fetching follower and following records for user %s, error: %s',
                    self::class,
                    $id,
                    $error->getMessage()
                )
            );
        }
    }

    /**
     * Generate activity logs related to entity cleanup.
     *
     * @param Event       $event      Event
     * @param string      $entity     Entity - workflow, testdefinition, group, project etc.
     * @param string      $entityName Name of the entity
     * @param string|null $role       Role of the user - owner, member, follower etc.
     * @param array       $streams    Stream names (e.g. group-qa, project-swarm)
     * @return void
     * @throws ContainerExceptionInterface
     * @throws NotFoundExceptionInterface
     */
    private function createCleanupActivity(
        Event $event,
        string $entity,
        string $entityName,
        ?string $role,
        array $streams = []
    ): void {
        $userId   = $event->getParam('id');
        $activity = new Activity();
        $activity->set(
            [
                'type'        => $entity,
                'user'        => $this->services->get(ConnectionFactory::P4_ADMIN)->getUser(),
                'target'      => "$entity ($entityName)",
                'action'      => 'updated',
                'description' => "removed user $userId from $entity ($entityName)'s $role",
                'streams'     => $streams,
            ]
        );
        $event->setParam('activity', $activity);
        $this->events->trigger(ListenerFactory::TASK_CLEANUP_DELETED_USER_ACTIVITY, null, $event->getParams());
    }
}
