<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.4/2843222
 */
namespace Projects\Validator;

use Application\Validator\AbstractValidator;
use Application\Validator\ConnectedAbstractValidator;
use Groups\Model\Group;
use Groups\Validator\Groups as GroupsValidator;
use Projects\Model\IProject;
use Users\Validator\Users as UsersValidator;

/**
 * Validates defaults on a project or branch
 */
class Defaults extends AbstractValidator
{
    private $connection;

    /**
     * Construct the validator. Supports the 'connection' option which must be specified
     * @param mixed $options options
     */
    public function __construct($options = null)
    {
        $this->connection = $options[ConnectedAbstractValidator::CONNECTION];
        parent::__construct($options);
    }

    /**
     * Validate the reviewers value of defaults. The reviewers field is expected to be an array with values of user ids
     * and/or group ids. Group ids are expected to be prefixed with swarm-group-.
     * If invalid a message will be built in this form (presuming there are invalid users and groups in the list)
     *
     * 'defaults' => [
     *     'reviewers' => [
     *         'unknownUserIds' => 'Unknown user id(s): arthur, prefect',
     *         'unknownGroupIds' => 'Unknown user id(s): group2'
     *     ]
     * ]
     * @param array $value
     * @return bool
     */
    private function validateReviewers(array $value): bool
    {
        if ($value && isset($value[IProject::REVIEWERS])) {
            $errors = [];
            $users  = [];
            $groups = [];
            foreach ((array)$value[IProject::REVIEWERS] as $id => $defaultReviewer) {
                if (Group::isGroupName($id)) {
                    $groups[] = Group::getGroupName($id);
                } else {
                    $users[] = $id;
                }
            }
            if ($groups) {
                $groupsValidator = new GroupsValidator(
                    [
                        ConnectedAbstractValidator::CONNECTION => $this->connection,
                    ]
                );
                if (!$groupsValidator->isValid($groups)) {
                    $errors[GroupsValidator::UNKNOWN_IDS]
                        = $groupsValidator->getMessages()[GroupsValidator::UNKNOWN_IDS];
                }
            }
            if ($users) {
                $usersValidator =
                    new UsersValidator(
                        [
                            ConnectedAbstractValidator::CONNECTION => $this->connection
                        ]
                    );
                if (!$usersValidator->isValid($users)) {
                    $errors[UsersValidator::UNKNOWN_IDS] = $usersValidator->getMessages()[UsersValidator::UNKNOWN_IDS];
                }
            }

            if ($errors) {
                foreach ($errors as $key => $message) {
                    $this->abstractOptions['messages'][IProject::REVIEWERS][$key] = $message;
                }
                return false;
            }
        }
        return true;
    }

    /**
     * Validates the defaults field. Currently we only support 'reviewers' in defaults
     * @param mixed $value the defaults array
     * @return bool
     */
    public function isValid($value): bool
    {
        return $this->validateReviewers($value);
    }
}
