<?php
/**
 * Perforce Swarm
 *
 * @copyright 2013-2025 Perforce Software. All rights reserved.
 * @license   Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version   2025.3/2828589
 */
namespace Projects\Validator;

use Application\Config\Services;
use Application\Validator\ConnectedAbstractValidator;
use Application\Validator\GreaterThanInt;
use Application\Validator\IsArray;
use Application\Validator\IsBool;
use Exception;
use Groups\Validator\Groups as GroupsValidator;
use Projects\Checker\IProject as IProjectChecker;
use Projects\Model\IProject;
use Users\Validator\Users as UsersValidator;

/**
 * Validate the branches of a project
 */
class Branches extends ConnectedAbstractValidator
{
    const WORKFLOW_VALIDATOR  = 'workflowValidator';
    const NOT_ARRAY           = 'notArray';
    const NO_ID               = 'noId';
    const NO_NAME             = 'noName';
    const DUPLICATE_ID        = 'duplicateId';
    const NOT_ALLOWED         = 'notAllowed';
    const CHECK_PERMISSION    = 'checkPermission';
    const EXISTING_BRANCH_IDS = 'existingBranchIds';

    protected $checkPermission  = false;
    protected $messageTemplates = [];
    protected $services         = null;
    protected $branchIds        = [];
    private $connection;
    private $workflowValidator;

    /**
     * Construct the validator. Supports 'connection' and 'workflowValidator' options
     *
     * @param mixed $options options
     */
    public function __construct($options = null)
    {
        $this->workflowValidator = $options[self::WORKFLOW_VALIDATOR] ?? null;
        $this->connection        = $options[ConnectedAbstractValidator::CONNECTION];
        $this->messageTemplates  = [
            self::NOT_ARRAY    => $this->tGen("Branches must be an array with each branch in array form."),
            // Given our normalization, we assume an empty id results from a bad name
            self::NO_ID        => $this->tGen("Branch name must contain at least one letter or number."),
            self::NO_NAME      => $this->tGen("All branches require a name."),
            self::DUPLICATE_ID => $this->tGen("Branches cannot have the same id, '%value%' is already in use."),
            self::NOT_ALLOWED  => $this->tGen('Only administrators can change the branches within a project.')
        ];
        $this->services          = $options[self::SERVICES];
        $this->checkPermission   = $options[self::CHECK_PERMISSION];
        $this->branchIds         = $options[self::EXISTING_BRANCH_IDS]??[];
        parent::__construct($options);
    }

    /**
     * Validate that name is set and that it is not empty
     *
     * @param  array $branch the branch
     * @return bool
     */
    protected function validateName(array $branch): bool
    {
        if (!isset($branch[IProject::NAME]) || !strlen($branch[IProject::NAME])) {
            $this->error(self::NO_NAME);
            return false;
        }
        return true;
    }

    /**
     * Validate that the branch is an array
     *
     * @param  mixed $branch the branch
     * @return bool
     */
    protected function validateArray($branch): bool
    {
        if (!is_array($branch)) {
            $this->error(self::NOT_ARRAY);
            return false;
        }
        return true;
    }

    /**
     * Validate that id is set and that it has a value
     *
     * @param  array $branch the branch
     * @return bool
     */
    protected function validateId(array $branch): bool
    {
        if (!isset($branch[IProject::ID]) || !strlen($branch[IProject::ID])) {
            $this->error(self::NO_ID);
            return false;
        }
        return true;
    }

    /**
     * Validate that the id for the branch has not already been seen by checking an array of ids
     *
     * @param  array $branch the branch
     * @param  array $ids    cumulative branch ids
     * @return bool
     */
    protected function validateDuplicateId(array $branch, array $ids): bool
    {
        if (in_array($branch[IProject::ID], $ids)) {
            $this->error(self::DUPLICATE_ID, $branch[IProject::ID]);
            return false;
        }
        return true;
    }

    /**
     * Validate that if retain default reviewers is set, it should be a boolean value
     *
     * @param  array $branch the branch
     * @return bool
     */
    protected function validateRetainDefault(array $branch): bool
    {
        if (isset($branch[IProject::RETAIN_DEFAULT_REVIEWERS])) {
            $isBool = new IsBool();
            if (!$isBool->isValid($branch[IProject::RETAIN_DEFAULT_REVIEWERS])) {
                $this->abstractOptions['messages'][IProject::RETAIN_DEFAULT_REVIEWERS] = $isBool->getMessages();
                return false;
            }
        }
        return true;
    }

    /**
     * Validate that paths are valid
     *
     * @param  array $branch the branch data
     * @return bool
     * @see    BranchPath
     */
    protected function validatePaths(array $branch): bool
    {
        if (isset($branch[IProject::BRANCH_PATHS])) {
            $pathValidator = new BranchPath(
                [
                    ConnectedAbstractValidator::CONNECTION => $this->connection,
                    ConnectedAbstractValidator::SERVICES => $this->services
                ]
            );
            if (!$pathValidator->isValid($branch[IProject::BRANCH_PATHS])) {
                $this->abstractOptions['messages'][IProject::BRANCH_PATHS] = $pathValidator->getMessages();
                return false;
            }
        }
        return true;
    }

    /**
     * Validate that if moderators are set the values are existing user ids
     *
     * @param  array $branch the branch
     * @return bool
     * @see    UsersValidator
     */
    protected function validateModerators(array $branch): bool
    {
        if (isset($branch[IProject::MODERATORS])) {
            $usersValidator =
                new UsersValidator(
                    [
                        ConnectedAbstractValidator::CONNECTION => $this->connection
                    ]
                );
            if (!$usersValidator->isValid($branch[IProject::MODERATORS])) {
                $this->abstractOptions['messages'][IProject::MODERATORS] = $usersValidator->getMessages();
                return false;
            }
        }
        return true;
    }

    /**
     * Validate that if moderator groups are set that they are valid group ids
     *
     * @param  array $branch the branch
     * @return bool
     * @see    GroupsValidator
     */
    protected function validateModeratorGroups(array $branch): bool
    {
        if (isset($branch[IProject::MODERATORS_GROUPS])) {
            $isArray = new IsArray();
            if (!$isArray->isValid($branch[IProject::MODERATORS_GROUPS])) {
                $this->abstractOptions['messages'][IProject::MODERATORS_GROUPS] = $isArray->getMessages();
                return false;
            }
            $groupsValidator = new GroupsValidator(
                [
                    ConnectedAbstractValidator::CONNECTION => $this->connection,
                    'allowProject' => false
                ]
            );
            if (!$groupsValidator->isValid($branch[IProject::MODERATORS_GROUPS])) {
                $this->abstractOptions['messages'][IProject::MODERATORS_GROUPS] = $groupsValidator->getMessages();
                return false;
            }
        }
        return true;
    }

    /**
     * Validate that the minimum up votes value is an integer greater than 0.
     *
     * @param  array $branch the branch
     * @return bool
     */
    protected function validateMinimumUpVotes(array $branch): bool
    {
        if (isset($branch[IProject::MINIMUM_UP_VOTES])) {
            $validator = new GreaterThanInt(['min' => -1, 'nullable' => true]);
            if (!$validator->isValid($branch[IProject::MINIMUM_UP_VOTES])) {
                $this->abstractOptions['messages'][IProject::MINIMUM_UP_VOTES] = $validator->getMessages();
                return false;
            }
        }
        return true;
    }

    /**
     * Validate the workflow, handing responsibility to the workflow validator
     *
     * @param  array $branch the branch
     * @return bool
     */
    protected function validateWorkflow(array $branch) : bool
    {
        if ($this->workflowValidator && isset($branch[IProject::WORKFLOW])) {
            if (!$this->workflowValidator->isValid($branch[IProject::WORKFLOW])) {
                $this->abstractOptions['messages'][IProject::WORKFLOW] = $this->workflowValidator->getMessages();
                return false;
            }
        }
        return true;
    }

    /**
     * Validate defaults
     *
     * @param  array $branch the branch
     * @return bool
     */
    protected function validateDefaults(array $branch) : bool
    {
        if (isset($branch[IProject::DEFAULTS])) {
            $validator = new Defaults([ConnectedAbstractValidator::CONNECTION=>$this->connection]);
            if (!$validator->isValid($branch[IProject::DEFAULTS])) {
                $this->abstractOptions['messages'][IProject::DEFAULTS] = $validator->getMessages();
                return false;
            }
        }
        return true;
    }

    /**
     * Validate 'branches' overall.
     * - id must be specified
     * - name must be specified
     * - no duplicate ids
     * - retain default reviewers must be a boolean
     * - moderators must be valid users
     * - moderator groups must be valid groups
     * - paths must be valid according to the branch paths validation
     * - minimum votes must be valid if set
     * - workflow must be valid according to the supplied workflow validator
     * - defaults must be valid
     *
     * @param  mixed $value value of the branches field
     * @return bool
     */
    public function isValid($value): bool
    {
        if ($this->checkPermission) {
            try {
                $this->services->get(Services::CONFIG_CHECK)->check(IProjectChecker::BRANCHES_ADMIN_ONLY_CHECKER);
            } catch (Exception $exception) {
                $this->error(self::NOT_ALLOWED);
                return false;
            }
        }

        $ids = $this->branchIds;
        foreach ((array)$value as $branch) {
            $valid = $this->validateArray($branch)
                && $this->validateName($branch)
                && $this->validateId($branch)
                && $this->validateDuplicateId($branch, $ids)
                && $this->validateRetainDefault($branch)
                && $this->validatePaths($branch)
                && $this->validateModerators($branch)
                && $this->validateModeratorGroups($branch)
                && $this->validateMinimumUpVotes($branch)
                && $this->validateWorkflow($branch)
                && $this->validateDefaults($branch);

            if (!$valid) {
                return false;
            }
            $ids[] = $branch[IProject::ID];
        }
        return true;
    }
}
