<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.3/2828589
 */
namespace Projects\Filter;

use Application\Connection\ConnectionFactory;
use Application\InputFilter\DirectInput;
use Application\InputFilter\InputFilter;
use Application\Validator\AbstractValidator;
use Application\Validator\ConnectedAbstractValidator;
use Application\Validator\IsArray;
use Interop\Container\ContainerInterface;
use Projects\Model\IProject;
use Projects\Validator\Branches as BranchesValidator;
use Projects\Validator\Name;
use Projects\Validator\Workflow as WorkflowValidator;

/**
 * Abstract input filter to provide common functionality for projects/branches
 */
abstract class ProjectFields extends InputFilter
{
    protected $services;
    protected $workflowValidator;
    protected $p4;

    /**
     * Generic set up
     * @param ContainerInterface $services
     * @param array|null $options
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        // General set up only, sub-classes should add input fields as needed
        $this->services          = $services;
        $this->p4                = $options[ConnectionFactory::P4] ?? $services->get(ConnectionFactory::P4_ADMIN);
        $this->workflowValidator = new WorkflowValidator(
            [
                AbstractValidator::SERVICES => $this->services,
                "validateWorkflowOwner" => $options["validateWorkflowOwner"] ?? true,
            ] +
            (array)$options
        );
        $this->setMode($options[self::MODE]??self::MODE_VIEW);
    }

    /**
     * Handle the branches field. Use a filter to process any values and then use the branch validator to ensure values
     * are correct.
     * @return void
     * @see Branches
     * @see BranchesValidator
     */
    protected function addBranches(array $existingBranchIds = [])
    {
        $input = new DirectInput(IProject::BRANCHES);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new Branches());
        $input->getValidatorChain()
            ->attach(new IsArray())
            ->attach(
                new BranchesValidator(
                    [
                        AbstractValidator::SERVICES => $this->services,
                        ConnectedAbstractValidator::CONNECTION => $this->p4,
                        BranchesValidator::WORKFLOW_VALIDATOR => $this->workflowValidator,
                        Name::CHECK_PERMISSION => $this->isEdit(),
                        BranchesValidator::EXISTING_BRANCH_IDS => $existingBranchIds
                    ]
                )
            );
        $this->add($input);
    }
}
