<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.3/2828589
 */

namespace Projects\Controller;

use Api\Controller\AbstractRestfulController;
use Api\IRequest;
use Application\Checker;
use Application\Config\ConfigManager;
use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\I18n\TranslatorFactory;
use Application\InputFilter\InputFilter;
use Application\Log\SwarmLogger;
use Application\Model\IModelDAO;
use Application\Permissions\Exception\ForbiddenException;
use Application\Permissions\Exception\UnauthorizedException;
use Application\Permissions\IPermissions;
use Application\Permissions\Permissions;
use Application\Validator\AbstractValidator;
use Application\Validator\ConnectedAbstractValidator;
use Events\Listener\ListenerFactory;
use Exception;
use InvalidArgumentException;
use Jobs\Controller\JobTrait;
use Jobs\Filter\IGetJobs;
use Laminas\Http\Request;
use P4\Log\Logger;
use P4\Model\Fielded\Iterator;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use Projects\Checker\IProject as ProjectChecks;
use Projects\Model\IProject;
use Projects\Model\Project;
use Projects\Validator\BranchPath as BranchPathValidator;
use Projects\Validator\Workflow;
use Queue\Manager;
use Record\Exception\NotFoundException as RecordNotFoundException;

/**
 * Class ProjectApi
 * @package Projects\Controller
 */
class ProjectApi extends AbstractRestfulController implements IProjectApi
{
    use JobTrait;
    const DATA_FOLLOWERS = 'followers';

    /**
     * Gets a project
     * Example success response
     * {
     *  "error": null,
     *  "messages": [],
     *  "data": {
     *        "projects": [
     *          {
     *              "name": Jam,
     *              "defaults": {
     *                 reviewers: [],
     *              },
     *              "description: "This is the Jam project.",
     *              "members": [
     *                "bruno",
     *                "rupert",
     *              ],
     *              "subgroups": [],
     *              "owners": [],
     *              "branches": [
     *                 {
     *                    id: "main",
     *                    name: "Main",
     *                    workflow: null,
     *                    paths: [],
     *                    ...
     *                 },
     *              ],
     *              "jobView": "",
     *              "emailFlags": {
     *                  change_email_project_users: 1,
     *                  review_email_project_members: 1,
     *               },
     *              "tests": {
     *                  enabled: false,
     *                  url: "",
     *                  postBody: "",
     *                  postFormat: "",
     *               },
     *              "deploy": {
     *                  enabled: false,
     *                  url: "",
     *               },
     *              "delete": false,
     *              "private": false,
     *              "workflow": null,
     *              "retainDefaultReviewers": false,
     *              "minimumUpVotes": null,
     *              "id": "Jam",
     *          }
     *         ]
     *    }
     * }
     *
     * Query parameters supported:
     *  fields - filter by fields
     *  metadata - include a metadata field for each project containing extra information for ease of use
     *      "metadata": {
     *          "userRoles": [ each role user has ],
     *      }
     *
     *
     * Example error response
     *
     * Unauthorized response 401, if require_login is true
     * {
     *   "error": "Unauthorized"
     * }
     *
     * 500 error response
     * {
     *   "error": 500,
     *   "messages": [
     *       {
     *           "code": 500,
     *           "text": "Something really bad happened"
     *       }
     *   ],
     *   "data": null
     * }
     * @param mixed $id The Project ID
     * @return mixed|JsonModel
     */
    public function get($id)
    {
        $p4Admin     = $this->services->get(ConnectionFactory::P4_ADMIN);
        $dao         = $this->services->get(IModelDAO::PROJECT_DAO);
        $errors      = null;
        $projectData = null;
        try {
            $project     = $dao->fetchById($id, $p4Admin, [IModelDAO::FILTER_PRIVATES => true]);
            $fields      = $this->getRequest()->getQuery(IRequest::FIELDS);
            $projectData = $this->modelsToArray([$project], [IRequest::METADATA =>true]);
            $projectData = $this->limitFieldsForAll($projectData, $fields);
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = $this->buildMessage(Response::STATUS_CODE_404, $e->getMessage());
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = $this->buildMessage(Response::STATUS_CODE_500, $e->getMessage());
        }
        if ($errors) {
            $json = $this->error([$errors], $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_PROJECTS => $projectData]);
        }
        return $json;
    }

    /**
     * Gets all projects data according to logged in user and return all project that the
     * user has permission for based on  project visibility
     * Example success response
     * {
     *  "error": null,
     *  "messages": [],
     *  "data": {
     *        "projects": [
     *          {
     *              "name": Jam,
     *              "defaults": {
     *                 reviewers: [],
     *              },
     *              "description: "This is the Jam project.",
     *              "members": [
     *                "bruno",
     *                "rupert",
     *              ],
     *              "subgroups": [],
     *              "owners": [],
     *              "branches": [
     *                 {
     *                    id: "main",
     *                    name: "Main",
     *                    workflow: null,
     *                    paths: [],
     *                    ...
     *                 },
     *              ],
     *              "jobView": "",
     *              "emailFlags": {
     *                  change_email_project_users: 1,
     *                  review_email_project_members: 1,
     *               },
     *              "tests": {
     *                  enabled: false,
     *                  url: "",
     *                  postBody: "",
     *                  postFormat: "",
     *               },
     *              "deploy": {
     *                  enabled: false,
     *                  url: "",
     *               },
     *              "delete": false,
     *              "private": false,
     *              "workflow": null,
     *              "retainDefaultReviewers": false,
     *              "minimumUpVotes": null,
     *              "id": "Jam",
     *          },
     *          ...
     *          ...
     *         ]
     *    }
     * }
     *
     * Query parameters supported:
     *  fields - filter by fields
     *
     *
     * Example error response
     *
     * Unauthorized response 401, if require_login is true
     * {
     *   "error": "Unauthorized"
     * }
     *
     * 500 error response
     * {
     *   "error": 500,
     *   "messages": [
     *       {
     *           "code": 500,
     *           "text": "Something really bad happened"
     *       }
     *   ],
     *   "data": null
     * }
     * @return mixed|JsonModel
     */
    public function getList()
    {
        $p4Admin       = $this->services->get(ConnectionFactory::P4_ADMIN);
        $errors        = null;
        $projectsArray = [];
        $projects      = null;
        $request       = $this->getRequest();
        $query         = $request->getQuery();
        try {
            $filter  = $this->services->get(Services::GET_PROJECTS_FILTER);
            $options = $query->toArray();
            $fields  = $query->get(IRequest::FIELDS);

            $filter->setData($options);
            if ($filter->isValid()) {
                $options = $filter->getValues();
                $this->editOptions($options);
                $dao           = $this->services->get(IModelDAO::PROJECT_DAO);
                $projects      = $dao->fetchAll($options, $p4Admin);
                $projectsArray = $this->limitFieldsForAll($this->modelsToArray($projects, $options), $fields);
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $errors = $filter->getMessages();
            }
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success(
                [
                    self::DATA_PROJECTS => $projectsArray,
                ]
            );
        }
        return $json;
    }

    /**
     * Create a new project
     * Example: POST http://<host>>/api/<version>/projects
     * 200:
     * {
     *   "error": null,
     *   "messages": [],
     *   "data": {
     *     "projects": [
     *     {
     *       "id": "v9-api-new-name",
     *       "name": "v9 Api New name",
     *       "defaults": {
     *         "reviewers": []
     *       },
     *       "description": "much better description",
     *       "members": [
     *         "mei"
     *       ],
     *       "subgroups": [],
     *       "owners": [],
     *       "branches": [],
     *       "jobview": null,
     *       "emailFlags": {
     *       "change_email_project_users": true,
     *       "review_email_project_members": false
     *       },
     *       "tests": {
     *         "enabled": false,
     *         "url": null
     *       },
     *       "deploy": {
     *         "enabled": false,
     *         "url": null
     *       },
     *       "deleted": false,
     *       "private": false,
     *       "workflow": null,
     *       "retainDefaultReviewers": false,
     *       "minimumUpVotes": null
     *     }]
     *   }
     * }
     *  Where a user cannot create a project
     *
     * 501:
     * {
     *       "error": 501,
     *       "messages": [
     *          {
     *              "code": 501
     *              "text": "Your account does not have admin privileges."
     *          }
     *       ],
     *       "data": null
     * }
     *
     * For errors in the provided data (for example)
     * {
     *   "error": 400,
     *   "messages" : {
     *     "id": {
     *       "isEmpty": "Name must contain at least one letter or number."
     *     },
     *     "members": {
     *       "members": "Project must have at least one member or subgroup."
     *     }
     *   }
     * }
     * Unauthorized
     * {
     *       "error": 401,
     *       "messages": [
     *           {
     *               "code": 401,
     *               "text": "Unauthorized"
     *           }
     *       ],
     *       "data": null
     *  }
     * @param mixed         $data       data for create
     * @return JsonModel|mixed
     */
    public function create($data)
    {
        $errors   = null;
        $services = $this->services;
        $config   = $services->get(ConfigManager::CONFIG);
        $project  = null;
        try {
            $checker = $this->services->get(Services::CONFIG_CHECK);
            $checker->checkAll([IPermissions::AUTHENTICATED_CHECKER, ProjectChecks::PROJECT_ADD_CHECKER]);
            $filter = $services->build(Services::PROJECTS_FILTER, [InputFilter::MODE => InputFilter::MODE_ADD])
                ->setData(
                    [ IProject::ID => $data[IProject::NAME] ] +
                    (array)$data +
                    [
                        IProject::PRIVATE =>
                            ConfigManager::getValue($config, IConfigDefinition::PROJECTS_PRIVATE_BY_DEFAULT, false)
                    ]
                );
            if ($filter->isValid()) {
                $projectDao = $services->get(IModelDAO::PROJECT_DAO);
                $project    = (new Project($services->get(ConnectionFactory::P4_ADMIN)))->set($filter->getValues());
                $project    = $projectDao->save($project);
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $errors = $filter->getMessages();
            }
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_501);
            $errors = [$this->buildMessage(Response::STATUS_CODE_501, $e->getMessage())];
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, 'Unauthorized')];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }

        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_PROJECTS => [$project->toArray()]]);
            $this->addTask($project, ListenerFactory::PROJECT_CREATED);
        }
        return $json;
    }

    // Responses are documented in Postman
    /**
     * Update part of project
     * @param mixed $id project id to update
     * @param mixed $data partial project data to update project
     * @return JsonModel
     */
    public function patch($id, $data): JsonModel
    {
        $services               = $this->services;
        $errors                 = null;
        $project                = null;
        $isPathPermission       = true;
        $branchValidationErrors = [];
        try {
            $p4Admin  = $services->get(ConnectionFactory::P4_ADMIN);
            $dao      = $services->get(IDao::PROJECT_DAO);
            $project  = $dao->fetchById($id, $p4Admin, [IModelDAO::FILTER_PRIVATES => true]);
            $checkers = [
                IPermissions::AUTHENTICATED_CHECKER,
                ProjectChecks::PROJECT_EDIT_CHECKER + [Checker::OPTIONS => [Checker::VALUE => $project]]
            ];
            if (array_key_exists('name', $data)) {
                $checkers[] = ProjectChecks::PROJECT_NAME_ADMIN_ONLY_CHECKER;
            }
            $services->get(Services::CONFIG_CHECK)->checkAll($checkers);
            $this->populateDependantFields($project, $data);
            $filter = $services->build(
                Services::PROJECTS_FILTER,
                [
                    InputFilter::MODE => InputFilter::MODE_EDIT,
                    Workflow::CHECK_WORKFLOW_PERMISSION => $this->needsWorkflowPermissionCheck($project, $data)
                ]
            )->setNotAllowed(IProject::ID)
                ->setData($data)
                ->setValidationGroupSafe(array_keys($data));

            // check if user having correct permission for all branch paths when the workflow of project level is edited
            if (array_key_exists(IProject::WORKFLOW, $data)) {
                $config          = $this->services->get(ConfigManager::CONFIG);
                $checkPermission = isset($config[ConfigManager::PROJECTS][ConfigManager::PERMISSION_CHECK]) &&
                    (bool)$config[ConfigManager::PROJECTS][ConfigManager::PERMISSION_CHECK];
                if ($checkPermission) {
                    $arrMixBranches = $project->getBranches();
                    if (!empty($arrMixBranches)) {
                        $pathValidator = new BranchPathValidator(
                            [
                                ConnectedAbstractValidator::CONNECTION => $p4Admin,
                                ConnectedAbstractValidator::SERVICES => $services
                            ]
                        );
                        foreach ($arrMixBranches as $branch) {
                            if (!$pathValidator->isValid($branch[IProject::BRANCH_PATHS])) {
                                $isPathPermission = false;
                                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                                $branchValidationErrors[IProject::WORKFLOW] = $pathValidator->getMessages();
                                break;
                            }
                        }
                    }
                }
            }
            if ($filter->isValid() && $isPathPermission) {
                foreach ($filter->getValues() as $field => $value) {
                    if ($project->hasField($field)) {
                        $project->set($field, $value);
                    } else {
                        throw new \InvalidArgumentException("Unknown field project->".$data['field']);
                    }
                }
                $project = $dao->save($project);
            } else {
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
                $filterErrors = [];
                if (!$filter->isValid()) {
                    $filterErrors = $filter->getMessages();
                }
                $errors = array_merge($branchValidationErrors, $filterErrors);
            }
        } catch (\InvalidArgumentException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = [$this->buildMessage(Response::STATUS_CODE_400, $e->getMessage())];
        } catch (UnauthorizedException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, 'Unauthorized')];
        } catch (ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_403);
            $errors = [$this->buildMessage(Response::STATUS_CODE_403, $e->getMessage())];
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success(
                [self::DATA_PROJECTS => $this->modelsToArray([$project], [IRequest::METADATA=>true])]
            );
            $this->addTask($project, ListenerFactory::PROJECT_UPDATED);
        }
        return $json;
    }

    /**
     * Determine whether to ignore workflow permissions. This will enable the permission checking unless
     * the branches field is in the data and the workflow values in the branches match those of the project
     * @param Project $project
     * @param array $data
     * @return array
     */
    protected function needsWorkflowPermissionCheck(Project $project, array $data) :array
    {
        $needsValidation = [];
        if (in_array(IProject::BRANCHES, array_keys($data))) {
            $dataBranches       = $data[IProject::BRANCHES];
            $dataBranchCount    = count($dataBranches);
            $projectBranches    = $project->getBranches();
            $projectBranchCount = count($projectBranches);
            // If there is a new branch, so validation is required
            if ($dataBranchCount > $projectBranchCount) {
                // There is a new branch, check the workflow for that one
                $projectBranchIds = array_map(
                    function ($branch) {
                        return $branch[IProject::ID];
                    },
                    $projectBranches
                );
                $needsValidation  = array_map(
                    function ($newBranch) {
                        return $newBranch[IProject::WORKFLOW] ?? null;
                    },
                    array_values(
                        array_filter(
                            $dataBranches,
                            function ($branch) use ($projectBranchIds) {
                                return !in_array($branch[IProject::ID] ?? null, $projectBranchIds);
                            }
                        )
                    )
                );
            }
            // If a branch is removed, adjust the project branch array
            if (!$needsValidation && $dataBranchCount < $projectBranchCount) {
                $dataBranchIds   = array_map(
                    function ($branch) {
                        return $branch[IProject::ID];
                    },
                    $dataBranches
                );
                $projectBranches = array_values(
                    array_filter(
                        $projectBranches,
                        function ($branch) use ($dataBranchIds) {
                            return in_array($branch[IProject::ID], $dataBranchIds);
                        }
                    )
                );
            }
            if (!$needsValidation) {
                foreach ($dataBranches as $index => $branch) {
                    $branchWorkflow = $branch[IProject::WORKFLOW];
                    if ($branchWorkflow !== $projectBranches[$index][IProject::WORKFLOW]) {
                        $needsValidation[] = $branchWorkflow;
                    }
                }
            }
        }
        return $needsValidation;
    }

    /**
     * Populate data from the existing project data to the data provided to the API if it was not provided.
     * @param Project $project existing project data
     * @param array $data data provided to the API
     * @return void
     */
    protected function populateDependantFields(Project $project, array &$data)
    {
        /*
         * One of members or subgroups must be set so if only one is provided in data we want to set the other from
         * existing project data. In this way we ensure that the context provided to the Members validator has current
         * data, it can then reliably check if these fields have values when only one is provided as a patch to the API
         *
         * Email flags are also populated as dependent fields if one of them is absent, especially during the manual API
         * hit in order to maintain the absent ones present state instead of resetting it to the default value.
         */
        if (isset($data[IProject::MEMBERS]) && !isset($data[IProject::SUBGROUPS])) {
            $data[IProject::SUBGROUPS] = $project->getRawValue(IProject::SUBGROUPS);
        }
        if (isset($data[IProject::SUBGROUPS]) && !isset($data[IProject::MEMBERS])) {
            $data[IProject::MEMBERS] = $project->getRawValue(IProject::MEMBERS);
        }
        if (isset($data[IProject::EMAIL_FLAGS][IProject::EMAIL_ON_SUBMIT]) &&
            !isset($data[IProject::EMAIL_FLAGS][IProject::EMAIL_ON_REVIEW_UPDATE])) {
            $data[IProject::EMAIL_FLAGS][IProject::EMAIL_ON_REVIEW_UPDATE] =
                $project->getRawValue(IProject::EMAIL_FLAGS)[IProject::EMAIL_ON_REVIEW_UPDATE];
        }
        if (isset($data[IProject::EMAIL_FLAGS][IProject::EMAIL_ON_REVIEW_UPDATE]) &&
            !isset($data[IProject::EMAIL_FLAGS][IProject::EMAIL_ON_SUBMIT])) {
            $data[IProject::EMAIL_FLAGS][IProject::EMAIL_ON_SUBMIT] =
                $project->getRawValue(IProject::EMAIL_FLAGS)[IProject::EMAIL_ON_SUBMIT];
        }
    }

    /**
     * Helper to set the filter private and include deleted projects to true
     * Add any other options here later as required.
     * @param array $options Update the options with new values.
     */
    private function editOptions(&$options)
    {
        $options[IModelDAO::FILTER_PRIVATES]     = true;
        $options[Project::FETCH_INCLUDE_DELETED] = true;
    }

    /**
     * Convert an iterator of Projects to an array representation merging in any required metadata
     * @param Iterator|array $projects iterator of projects
     * @param array $options options for merging arrays. Supports IRequest::METADATA to merge in metadata
     * @param array $metadataOptions options for metadata. Supports:
     *      IProjectApi::USER_ROLES    summary of user roles for project
     * @return array
     */
    public function modelsToArray($projects, $options, $metadataOptions = [])
    {
        $projectsArray = [];
        if (isset($options) && isset($options[IRequest::METADATA]) && $options[IRequest::METADATA] === true) {
            $metadataOptions += [
                self::USER_ROLES => true,
            ];
            $dao              = $this->services->get(IModelDAO::PROJECT_DAO);
            $metadata         = $dao->fetchAllMetadata($projects, $metadataOptions);
            if ($metadata) {
                $count = 0;
                foreach ($projects as $project) {
                    $projectData = $project->toArray();
                    // ensure only admin/super or project members/owners can see tests/deploy
                    $checks = $project->hasOwners()
                        ? ['admin', 'owner' => $project]
                        : ['admin', 'member' => $project];

                    $this->unsetPrivilegedFields($projectData, $checks);
                    $projectsArray[] = array_merge($projectData, $metadata[$count++]);
                }
            }
        } else {
            foreach ($projects as $project) {
                $projectData = $project->toArray();
                // ensure only admin/super or project members/owners can see tests/deploy
                $checks = $project->hasOwners()
                    ? ['admin', 'owner' => $project]
                    : ['admin', 'member' => $project];

                $this->unsetPrivilegedFields($projectData, $checks);
                $projectsArray[] = $projectData;
            }
        }
        return array_values($projectsArray);
    }
    /**
     * Remove fields that should be hidden but have been unhidden to enable us to cache
     * the full model. So instead we now put fields into this function that should be
     * hidden to be removed from the return output.
     *
     * @param array $data   This is the array format of the full project.
     * @param array $checks The checks we should validate before un setting data.
     */
    private function unsetPrivilegedFields(&$data, $checks = [])
    {
        // check if we are one of the required permissions to see the data.
        if ($this->services->get(Permissions::PERMISSIONS)->isOne($checks)) {
            return;
        }
        unset($data['tests']);
        unset($data['deploy']);
    }

    /**
     * Fetches the jobs associated with the project. Applies 'filter' and 'max' from the query parameters and
     * additionally adds the 'job view' field as a filter if it is set on the project.
     * @return JsonModel
     * @see JobTrait::getJobsByFilter()
     */
    public function projectJobsAction() : JsonModel
    {
        $id      = $this->getEvent()->getRouteMatch()->getParam(Project::FIELD_ID);
        $p4Admin = $this->services->get(ConnectionFactory::P4_ADMIN);
        $dao     = $this->services->get(IDao::PROJECT_DAO);
        try {
            $project = $dao->fetchById($id, $p4Admin, [IModelDAO::FILTER_PRIVATES => true]);
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
            return $this->error($errors, $this->getResponse()->getStatusCode());
        }
        $query   = $this->getRequest()->getQuery();
        $filter  = $query->toArray()[IGetJobs::FILTER_PARAMETER] ?? null;
        $jobView = $project ? trim($project->get('jobview')) : null;
        if ($jobView) {
            $filter  = $filter  ? "($filter) "  : "";
            $filter .= "($jobView)";
        }
        return $this->getJobsByFilter($filter);
    }

    /**
     * Manages followers of a given project. The request method determines the action performed
     *
     * GET    - returns the current list of followers
     * POST   - includes the current user in the list of followers
     * DELETE - removes the current user from the list of followers
     *
     * Example success response
     * {
     *   "error": null,
     *   "messages": [],
     *   "data": {
     *     "followers": [
     *      "bruno",
     *      "rupert",
     *     ],
     *   }
     * }
     *
     * Example error response
     *
     * Unauthorized response 404, if project does not exist
     * {
     *   "error":404,
     *   "messages": [
     *       {
     *           "code":404,
     *           "text":"Cannot fetch entry. Id does not exist."
     *       }
     *   ],
     *   "data":null
     * }
     *
     * 500 error response
     * {
     *   "error": 500,
     *   "messages": [
     *       {
     *           "code": 500,
     *           "text": "Something really bad happened"
     *       }
     *   ],
     *   "data": null
     * }
     * @return JsonModel
     */
    public function projectFollowersAction() : JsonModel
    {
        $services  = $this->services;
        $id        = $this->getEvent()->getRouteMatch()->getParam(Project::FIELD_ID);
        $p4Admin   = $services->get(ConnectionFactory::P4_ADMIN);
        $dao       = $services->get(IDao::PROJECT_DAO);
        $errors    = null;
        $followers = [];
        try {
            $checks = [IPermissions::AUTHENTICATED_CHECKER];
            switch ($this->getRequest()->getMethod()) {
                case Request::METHOD_DELETE:
                    $this->services->get(Services::CONFIG_CHECK)->checkAll($checks);
                    $dao->removeFollower($id);
                    break;
                case Request::METHOD_GET:
                    break;
                case Request::METHOD_POST:
                    $this->services->get(Services::CONFIG_CHECK)->checkAll($checks);
                    $dao->addFollower($id);
                    break;
                default:
                    $this->getResponse()->setStatusCode(Response::STATUS_CODE_405);
                    $errors[] = $this->buildMessage(
                        Response::STATUS_CODE_405,
                        $services->get(TranslatorFactory::SERVICE)->t('Method Not Allowed')
                    );
                    break;
            }
            $followers = $dao->fetchById($id, $p4Admin, [IModelDAO::FILTER_PRIVATES => true])->getFollowers(true);
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (UnauthorizedException $e) {
            // Not logged in
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, $e->getMessage())];
        } catch (InvalidArgumentException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = [$this->buildMessage(Response::STATUS_CODE_400, $e->getMessage())];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }

        if ($errors) {
            return $this->error($errors, $this->getResponse()->getStatusCode());
        }
        sort($followers);
        return $this->success([self::DATA_FOLLOWERS => $followers]);
    }

    /**
     * This is the combine method for both delete and undelete of the project
     * sets the deleted value as true/false depends upon the call
     * @param $id
     * @param $delete
     * @return JsonModel
     */
    private function handleDelete($id, $delete) : JsonModel
    {
        $errors   = [];
        $project  = null;
        $messages = [];
        try {
            $p4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
            $projectDAO = $this->services->get(IDao::PROJECT_DAO);
            $options    = [
                Project::FETCH_INCLUDE_DELETED => !$delete,
                IModelDAO::FILTER_PRIVATES => true
            ];
            $project    = $projectDAO->fetchById($id, $p4Admin, $options);
            $checker    = $this->services->get(Services::CONFIG_CHECK);
            // Authentication is required, delete is permitted if
            // - user is an admin
            // - user is an owner if the project has owners
            // - user is a member if there are no owners
            $checker->checkAll(
                [
                    IPermissions::AUTHENTICATED_CHECKER,
                    ProjectChecks::PROJECT_EDIT_CHECKER + [Checker::OPTIONS => [Checker::VALUE => $project]]
                ]
            );
            $project->setDeleted($delete);
            if ($delete === false) {
                $currentBranches   = $project->getBranches();
                $validBranches     = [];
                $branchPath        = new BranchPathValidator(
                    [
                        ConnectedAbstractValidator::CONNECTION => $p4Admin,
                        ConnectedAbstractValidator::SERVICES => $this->services
                    ]
                );
                $workflowValidator = new Workflow(
                    [
                        AbstractValidator::SERVICES => $this->services,
                        'connection' => $p4Admin,
                        "validateWorkflowOwner" => true,
                    ]
                );
                foreach ($currentBranches as $branch) {
                    $addToValidBranches = true;
                    // validate branch paths
                    if (!$branchPath->isValid($branch['paths'])) {
                        $messages[] =  ["Reason"=> "Invalid path for branch " . $branch['name'] .
                            " so skipped it. Reason: " . implode(' ', $branchPath->getMessages()),
                            'Skipped_Branch_Data' => json_encode($branch, true)];
                        Logger::log(
                            Logger::INFO,
                            "Invalid path for branch " . $branch['name'] .
                            " so skipped it. Reason: " . implode(' ', $branchPath->getMessages()) .
                            " Branch Details: " . json_encode($branch, true)
                        );
                        $addToValidBranches = false;
                    }
                    // validate Branch workflows
                    if ($branch['workflow'] && !$workflowValidator->isValid($branch['workflow'])) {
                        $messages[] =  ["Reason"=> "Invalid workflow id: " . $branch['workflow'] .
                            " for branch " . $branch['name'] . " so unsetting it.",
                            'Skipped_Branch_Data' => json_encode($branch, true)];
                        Logger::log(
                            Logger::INFO,
                            "Invalid workflow id: " . $branch['workflow'] . " for branch " . $branch['name'] .
                            " so unsetting it.",
                            " Branch Details: " . json_encode($branch, true)
                        );
                        $branch['workflow'] = null;
                    }
                    // add only valid branches to be set on project
                    if ($addToValidBranches) {
                        $validBranches[] = $branch;
                    }
                }
                $project->setBranches($validBranches);
                // validate project workflows
                if ($project->getWorkflow() && !$workflowValidator->isValid($project->getWorkflow())) {
                    $messages[] =  ["Reason"=> "Invalid workflow id: " . $project->getWorkflow() .
                        " for project " . $project->getId() .
                        " so skipped it & set the project workflow to null"];
                    Logger::log(
                        Logger::INFO,
                        "Invalid workflow id: " . $project->getWorkflow() . " for project " . $project->getId() .
                        " so skipped it & set the project workflow to null"
                    );
                    $project->setWorkflow(null);
                }
            }
            $projectDAO->save($project);
        } catch (UnauthorizedException|ForbiddenException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_401);
            $errors = [$this->buildMessage(Response::STATUS_CODE_401, $e->getMessage())];
        } catch (RecordNotFoundException $e) {
            // Project id is good but no record found or private project filtered from dao
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $e->getMessage())];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_PROJECTS => $this->modelsToArray([$project], [])], $messages);
        }
        return $json;
    }

    /**
     * Soft delete a project. Requires authentication and that the user must be an owner if the project has owners or
     * a member if there are no owners.
     * Example: DELETE http://<host>>/api/<version>/projects/<<project-id>>, responds with the project
     * 200:
     * {
     *   "error": null,
     *   "messages": [],
     *   "data": {
     *     "projects": [
     *     {
     *       "id": "v9-api-new-name",
     *       "name": "v9 Api New name",
     *       "defaults": {
     *         "reviewers": []
     *       },
     *       "description": "much better description",
     *       "members": [
     *         "mei"
     *       ],
     *       "subgroups": [],
     *       "owners": [],
     *       "branches": [],
     *       "jobview": null,
     *       "emailFlags": {
     *       "change_email_project_users": true,
     *       "review_email_project_members": false
     *       },
     *       "tests": {
     *         "enabled": false,
     *         "url": null
     *       },
     *       "deploy": {
     *         "enabled": false,
     *         "url": null
     *       },
     *       "deleted": true,
     *       "private": false,
     *       "workflow": null,
     *       "retainDefaultReviewers": false,
     *       "minimumUpVotes": null
     *     }]
     *   }
     * }
     *
     * Where a project does not exist or access is not permitted
     *
     * 404:
     * {
     *       "error": 404,
     *       "messages": [
     *          {
     *              "code": 404
     *              "text": "Cannot fetch entry. Id does not exist."
     *          }
     *       ],
     *       "data": null
     * }
     *
     * When a user does not have the correct privilege
     *
     * 401:
     * {
     *       "error": 401,
     *       "messages": [
     *          {
     *              "code": 401
     *              "text": "This operation is limited to project or group owners."
     *          }
     *       ],
     *       "data": null
     * }
     *
     * Unauthorized
     * {
     *   "error": "Unauthorized",
     * }
     * @param mixed $id id of the project to delete
     * @return JsonModel the project that has been soft deleted
     */
    public function delete($id): JsonModel
    {
        return $this->handleDelete($id, true);
    }

    /**
     * undelete the previously Soft deleted project. Requires authentication and that the user must be an owner
     * if the project has owners or a member if there are no owners.
     * Example: POST http://<host>>/api/<version>/projects/<<project-id>>/undelete, responds with the project
     * 200:
     * {
     *   "error": null,
     *   "messages": [],
     *   "data": {
     *     "projects": [
     *     {
     *       "id": "v9-api-new-name",
     *       "name": "v9 Api New name",
     *       "defaults": {
     *         "reviewers": []
     *       },
     *       "description": "much better description",
     *       "members": [
     *         "mei"
     *       ],
     *       "subgroups": [],
     *       "owners": [],
     *       "branches": [],
     *       "jobview": null,
     *       "emailFlags": {
     *       "change_email_project_users": true,
     *       "review_email_project_members": false
     *       },
     *       "tests": {
     *         "enabled": false,
     *         "url": null
     *       },
     *       "deploy": {
     *         "enabled": false,
     *         "url": null
     *       },
     *       "deleted": false,
     *       "private": false,
     *       "workflow": null,
     *       "retainDefaultReviewers": false,
     *       "minimumUpVotes": null
     *     }]
     *   }
     * }
     *
     * Where a project does not exist or access is not permitted
     *
     * 404:
     * {
     *       "error": 404,
     *       "messages": [
     *          {
     *              "code": 404
     *              "text": "Cannot fetch entry. Id does not exist."
     *          }
     *       ],
     *       "data": null
     * }
     *
     * When a user does not have the correct privilege
     *
     * 401:
     * {
     *       "error": 401,
     *       "messages": [
     *          {
     *              "code": 401
     *              "text": "This operation is limited to project or group owners."
     *          }
     *       ],
     *       "data": null
     * }
     *
     * Unauthorized
     * {
     *   "error": "Unauthorized",
     * }
     * @return JsonModel the project that has undeleted
     */
    public function unDeleteAction() : JsonModel
    {
        return $this->handleDelete($this->getEvent()->getRouteMatch()->getParam(Project::FIELD_ID), false);
    }

    /**
     * // TODO comment when completing implementation
     * @param mixed $id id of project to update
     * @param mixed $data project data
     * @return JsonModel
     */
    public function update($id, $data) : JsonModel
    {
        $this->getResponse()->setStatusCode(Response::STATUS_CODE_405);
        return $this->error(["Not supported yet"], Response::STATUS_CODE_405);
    }

    /**
     * Add a task to the queue for a project api action.
     * @param mixed     $project   the project affected
     * @param string $task      the name of the task
     */
    protected function addTask($project, string $task)
    {
        try {
            $queue = $this->services->get(Manager::SERVICE);
            $queue->addTask(
                $task,
                $project->getId(),
                [
                    IProject::NAME => $project->getName(),
                    ListenerFactory::USER => $this->services->get(ConnectionFactory::P4_USER)->getUser()
                ]
            );
        } catch (Exception $e) {
            // Catch and log any exception, allows a request to progress regardless
            $this->services->get(SwarmLogger::SERVICE)
                ->err(sprintf("Error queuing a project activity task [%s]", $e->getMessage()));
        }
    }
}
