<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.3/2828589
 */
namespace Files\Filter\Diff;

use Application\I18n\TranslatorFactory;
use Application\InputFilter\DirectInput;
use Application\InputFilter\InputFilter;
use Application\Validator\ArrayValuesValidator;
use Application\Validator\BetweenInt;
use Application\Validator\GreaterThanInt;
use Files\Filter\RevisionValidator;
use Interop\Container\ContainerInterface;
use Laminas\Filter\StringTrim;
use Laminas\InputFilter\Input;
use Laminas\Validator\StringLength;

/**
 * Class TestRun to filter and validate TestRun
 * @package TestIntegration\Filter
 */
class Diff extends InputFilter implements IDiff
{
    private $services;

    /**
     * @inheritDoc
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services = $services;
        $this->addFromFilter();
        $this->addToFilter();
        $this->addContextLinesFilter();
        $this->addIgnoreWSFilter();
        $this->addMaxSizeFilter();
        $this->addMaxDiffsFilter();
        $this->addOffsetFilter();
        $this->addTypeFilter();
        $this->addFromFileFilter();
        $this->addContentChunkStartFilter();
        $this->addContentChunkEndFilter();
        $this->addContentChunkStartLineFilter();
        $this->addContentChunkEndLineFilter();
    }

    /**
     * Add filters/validation for 'from' revision
     */
    private function addFromFilter()
    {
        $input = new DirectInput(self::FROM);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new RevisionValidator($this->services->get(TranslatorFactory::SERVICE)));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'to' version
     */
    private function addToFilter()
    {
        $input = new Input(self::TO);
        $input->getValidatorChain()->attach(new RevisionValidator($this->services->get(TranslatorFactory::SERVICE)));
        $this->add($input);
    }

    /**
     * Add filters/validation for lines
     */
    private function addContextLinesFilter()
    {
        $input = new DirectInput(self::LINES);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -1], $this->services));
        $this->add($input);
    }

    /**
     * Add filters/validation for ignore_ws
     */
    private function addIgnoreWSFilter()
    {
        $input = new DirectInput(self::IGNORE_WS);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new BetweenInt(['min' => 0, 'max' => 3], $this->services));
        $this->add($input);
    }

    /**
     * Add filters/validation for max
     */
    private function addMaxSizeFilter()
    {
        $input = new DirectInput(self::MAX_SIZE);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2], $this->services));
        $this->add($input);
    }

    /**
     * Add filters/validation for max_diffs
     */
    private function addMaxDiffsFilter()
    {
        $input = new DirectInput(self::MAX_DIFFS);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2], $this->services));
        $this->add($input);
    }

    /**
     * Add filters/validation for offset
     */
    private function addOffsetFilter()
    {
        $input = new DirectInput(self::OFFSET);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -1], $this->services));
        $this->add($input);
    }

    /**
     * Add filters/validation for type
     * Only allow 'file' and 'stream' as valid types
     */
    private function addTypeFilter()
    {
        $valid = ['file', 'stream'];
        $input = new DirectInput(self::TYPE);
        $input->setRequired(false);
        $translator = $this->services->get(TranslatorFactory::SERVICE);
        $validator  = new ArrayValuesValidator($translator, $valid, self::TYPE_ERROR_KEY, self::TYPE);
        $input->getValidatorChain()->attach($validator);
        $this->add($input);
    }

    /**
     * Add filters/validation for fromFile
     */
    private function addFromFileFilter()
    {
        $input = new DirectInput(self::FROM_FILE);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()->attach(new StringLength(['min' => 1]));
        $this->add($input);
    }
    /**
     * Add filters/validation for contentChunkStart
     */
    private function addContentChunkStartFilter()
    {
        $input = new DirectInput(self::CONTENT_CHUNK_START);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2], $this->services));
        $this->add($input);
    }
    /**
     * Add filters/validation for contentChunkEnd
     */
    private function addContentChunkEndFilter()
    {
        $input = new DirectInput(self::CONTENT_CHUNK_END);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2], $this->services));
        $this->add($input);
    }
    /**
     * Add filters/validation for contentChunkStartLine
     */
    private function addContentChunkStartLineFilter()
    {
        $input = new DirectInput(self::CONTENT_CHUNK_START_LINE);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2], $this->services));
        $this->add($input);
    }
    /**
     * Add filters/validation for contentChunkEndLine
     */
    private function addContentChunkEndLineFilter()
    {
        $input = new DirectInput(self::CONTENT_CHUNK_END_LINE);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2], $this->services));
        $this->add($input);
    }
}
