<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.3/2828589
 */

namespace Authentication\Service;

use Application\Cache\AbstractCacheService;
use Application\Config\ConfigException;
use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\I18n\TranslatorFactory;
use Application\Log\SwarmLogger;
use Application\Permissions\Exception\BasicAuthFailedException;
use Application\Permissions\Exception\UnauthorizedException;
use Authentication\Adapter;
use Authentication\Helper\ILoginHelper;
use Authentication\Helper\LoginHelper;
use Laminas\Authentication\Result;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use P4\Connection\Exception\ServiceNotFoundException;
use Exception;
use Users\Model\IUser;

class SsoLoginAdapter extends AbstractLoginAdapter implements LoginAdapterInterface
{
    const HAS_EXTENSION = 'Auth::loginhook';
    const LOG_PREFIX    = SsoLoginAdapter::class;
    /* Redis key where the url for the user to be navigated for sso login is saved.*/
    const SSO_NAV_URL_REDIS_KEY = 'navigationUrl';
    const TTL                   = 60;

    /**
     * @param $data
     * @return JsonModel
     */
    public function login($data): JsonModel
    {
        $services   = $this->services;
        $translator = $this->services->get(TranslatorFactory::SERVICE);
        /** @var LoginHelper $helper */
        $helper  = $services->get(ILoginHelper::HELPER_NAME);
        $session = $services->get(ILoginHelper::SESSION);
        try {
            /* Set the clientProgName to something else by default it goes P4PHP then HAS ext goes for 1-step mode
            authentication, so we are setting a string "SwarmSSOWithHAS" */
            $clientProgName = Adapter::CLIENT_PROG_NAME;
            $p4Admin        = $services->get(ConnectionFactory::P4_ADMIN);
            
            if (!$this->isSsoEnabled()) {
                $msg[] = self::buildMessage(
                    ILoginHelper::CODE_SSO_NOT_CONFIGURED,
                    $translator->t(ILoginHelper::TEXT_SSO_NOT_CONFIGURED)
                );
                return self::error($msg, Response::STATUS_CODE_500);
            }

            $session->start();
            $session->getStorage()->clear();
            $result                       = $this->handleSsoLogin($data['username'], $clientProgName);
            $data[ILoginHelper::PASSWORD] = $result[ILoginHelper::PASSWORD];
            $authUser                     = $result[ILoginHelper::AUTH_USER];
            if ($result[ILoginHelper::IS_VALID]) {
                $helper->setCookieInformation($data);
                $helper->invalidateCache($authUser, $p4Admin);
            } else {
                try {
                    /* verify that the admin connection is still valid if so then the user credentials
                    provided are incorrect. */
                    $userDao = $services->get(IDao::USER_DAO);
                    $userDao->fetchById($p4Admin->getUser(), $p4Admin);
                    throw new BasicAuthFailedException(ILoginHelper::TEXT_LOGIN_INCORRECT, Response::STATUS_CODE_401);
                } catch (Exception $exception) {
                    if ($exception instanceof BasicAuthFailedException) {
                        throw $exception;
                    }
                    throw new ServiceNotFoundException(
                        ILoginHelper::TEXT_SWARM_CONNECTION,
                        Response::STATUS_CODE_500
                    );
                }
            }
            $session->writeClose();
            $msg[] = self::buildMessage(
                ILoginHelper::CODE_LOGIN_SUCCESSFUL,
                $translator->t(ILoginHelper::TEXT_LOGIN_SUCCESSFUL)
            );
            $user  = $this->currentUser();
            return self::success([IUser::USER => [$user]], $msg);
        } catch (UnauthorizedException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_NOT_LOGGED_IN, $translator->t($error->getMessage()));
        } catch (ServiceNotFoundException $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_SWARM_CONNECTION, $translator->t($error->getMessage()));
        } catch (\P4\Exception $error) {
            $msg[] = self::buildMessage(ILoginHelper::CODE_LOGIN_MISSING, $translator->t($error->getMessage()));
        } catch (Exception $error) {
            $msg[] = self::buildMessage(ILoginHelper::USER_ERROR, $translator->t($error->getMessage()));
        }
        $session->destroy(['send_expire_cookie' => true, 'clear_storage' => true]);
        $session->writeClose();
        $this->getResponse()->setStatusCode($error->getCode());
        return self::error($msg, $this->getResponse()->getStatusCode());
    }

    /**
     * @param $userName
     * @param $clientProgName
     * @param $password
     * @return array
     * @throws ConfigException
     */
    public function handleSsoLogin($userName, $clientProgName, $password = ''): array
    {
        $services     = $this->services;
        $config       = $services->get(IConfigDefinition::CONFIG);
        $p4Admin      = $services->get(ConnectionFactory::P4_ADMIN);
        $cacheService = $services->get(AbstractCacheService::CACHE_SERVICE);
        $logger       = $services->get(SwarmLogger::SERVICE);
        /** @var Auth $auth */
        $auth = $services->get(Services::AUTH);
        /** @var LoginHelper $helper */
        $helper     = $services->get(ILoginHelper::HELPER_NAME);
        $error      = null;
        $authResult = null;
        $authUser   = null;
        // loop through all login candidates, stop on first success
        foreach ($helper->getLoginCandidates($userName) as $user) {
            if (strpos($userName, '@')) {
                $cacheService = $this->services->get(AbstractCacheService::CACHE_SERVICE);
                $cacheKey     = strstr($userName, '@', true);
                $cacheService->delete($cacheKey);
                $cacheService->set($cacheKey, $user, self::TTL);
            }
            $logger->info(sprintf("[%s]: Trying to log in as [%s]", self::LOG_PREFIX, $user));
            $adapter = new Adapter(
                $user,
                $password,
                $p4Admin,
                $config,
                false,
                $clientProgName,
                $cacheService,
                $logger
            );
            try {
                // break if we hit a working candidate
                $authResult = $auth->authenticate($adapter);
                if ($authResult->getCode() === Result::SUCCESS) {
                    $authUser = $user;
                    $logger->info(sprintf("[%s]: Authenticated user [%s]", self::LOG_PREFIX, $user));
                    break;
                } else {
                    $errorMessage = property_exists($authResult, 'messages') &&
                    $authResult->getMessages() ? current($authResult->getMessages()) : 'failed to generate ticket';
                    $logger->info(
                        sprintf(
                            "[%s]: Attempt to log in as [%s] failed with [%s]",
                            self::LOG_PREFIX,
                            $user,
                            $errorMessage
                        )
                    );
                    $error = $errorMessage;
                }
            } catch (Exception $e) {
                $logger->info(
                    sprintf(
                        "[%s]: Attempt to log in as [%s] failed with [%s]",
                        self::LOG_PREFIX,
                        $user,
                        $e->getMessage()
                    )
                );
                $error = $e->getMessage();
            }
        }

        $logger->info(sprintf("[%s]: Returning Authenticated user [%s]", self::LOG_PREFIX, $authUser));
        $ticket = $authResult ? $authResult->getIdentity()['ticket'] : '';

        return [
            ILoginHelper::AUTH_USER => $authUser,
            ILoginHelper::USERNAME  => $userName,
            ILoginHelper::PASSWORD  => $ticket,
            ILoginHelper::IS_VALID  => $authUser && $ticket,
            ILoginHelper::ERROR     => $error
        ];
    }

    /**
     * @return bool
     */
    private function isSsoEnabled(): bool
    {
        $helper        = $this->services->get(ILoginHelper::HELPER_NAME);
        $ssoConfig     = $helper->getSSO(P4_SERVER_ID);
        $isSamlEnabled = $this->services
            ->get(IConfigDefinition::CONFIG)['saml']['sp']['assertionConsumerService']['url'];
        if (($ssoConfig === IConfigDefinition::ENABLED || $ssoConfig === IConfigDefinition::OPTIONAL)
            && !$isSamlEnabled) {
            return true;
        }

        return false;
    }

    /**
     * @param string $username Username for which sso authentication url have to be retrieved from redis cache.
     * @return JsonModel
     */
    public function getSsoUrl(string $username): JsonModel
    {
        $cacheService = $this->services->get(AbstractCacheService::CACHE_SERVICE);
        if (strpos($username, '@')) {
            $cacheKey = strstr($username, '@', true);
            $username = $cacheService->get($cacheKey);
            $cacheService->delete($cacheKey);
        }
        $logger   = $this->services->get(SwarmLogger::SERVICE);
        $username = $this->services->get(ConnectionFactory::P4)->isCaseSensitive() ? $username : strtolower($username);
        // Retrieve the navigation url from Redis Cache
        $navigationUrl = $cacheService->get($username . self::SSO_NAV_URL_REDIS_KEY);
        $logger->info(
            sprintf(
                'Getting navigation url for the redis key: [%s] value: [%s]',
                $username . self::SSO_NAV_URL_REDIS_KEY,
                $navigationUrl
            )
        );
        $cacheService->delete($username . self::SSO_NAV_URL_REDIS_KEY);
        $jsonModel = filter_var($navigationUrl, FILTER_VALIDATE_URL) ?
            new JsonModel(
                [
                    'url' => $navigationUrl,
                    'isValid' => true,
                    'errors' => null,
                ]
            ) :
            new JsonModel(
                [
                    'url' => null,
                    'isValid' => false,
                    'errors' => ["Navigation url not received from redis cache"]
                ]
            );
        $data      = $jsonModel->getVariables();
        $jsonModel->clearVariables();
        $jsonModel->setVariables(
            [
                'data' => $data,
                'messages' => !empty($data) && array_key_exists('errors', $data) && !empty($data['errors']) ?
                    $data['errors'] : [],
                'error' => !empty($data) && array_key_exists('errors', $data) && !empty($data['errors']) ?
                    Response::STATUS_CODE_400 : null
            ]
        );
        return $jsonModel;
    }
}
