<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.3/2828589
 */
namespace Attachments\Model;

use Application\Config\Services;
use Application\Connection\ConnectionFactory;
use Application\Helper\StringHelper;
use Application\Helper\SwarmBlurHash;
use Application\Model\AbstractDAO;
use Application\Option;
use Events\Listener\ListenerFactory;
use P4\Connection\ConnectionInterface;
use P4\File\File;
use P4\Model\Fielded\Iterator;
use Queue\Manager;
use Record\Exception\NotFoundException;
use Record\Key\AbstractKey;
use Attachments\Model\Attachment as Model;
use P4\Connection\Exception\CommandException;
use P4\File\Exception\Exception as P4FileException;

/**
 * DAO class to handle CRUD operations for attachments
 */
class AttachmentDAO extends AbstractDAO implements IAttachmentDAO
{
    const LOG_PREFIX = AttachmentDAO::class;
    const MODEL      = Attachment::class;

    /**
     * Fetch attachments
     * @param array                     $options        Options for the fetch. Supported options:
     *                                                  FETCH_BY_IDS - single string or integer value or an array of
     *                                                  integer/string values to retrieve attachments with the chosen
     *                                                  id values. Unknown values are ignored
     * @param ConnectionInterface|null  $connection     Connection to use
     * @return Iterator iterator over the models fetched
     */
    public function fetchAll(array $options = [], ConnectionInterface $connection = null) : Iterator
    {
        // Validate that the given options are supported
        Option::validate($options, [AbstractKey::FETCH_BY_IDS => null]);

        if (isset($options[AbstractKey::FETCH_BY_IDS])) {
            // Cast the ids to an array
            $ids    = (array)$options[AbstractKey::FETCH_BY_IDS];
            $numIds = count($ids);
            if ($numIds === 0) {
                $models = new Iterator();
            } elseif ($numIds === 1) {
                // Do a single fetch by the id if there is one id as this should be more efficient
                try {
                    $attachment = Model::fetch(current($ids), $connection);
                    $models     = new Iterator([$attachment->getId() => $attachment]);
                } catch (NotFoundException $nfe) {
                    $models = new Iterator();
                }
            } else {
                $models = Model::fetchAll($options, $connection);
            }
        } else {
            $models = Model::fetchAll($options, $connection);
        }
        return $models;
    }

    /**
     * Fetches an attachment based on the id and the file. The file value is expected to be an encoded value that when
     * decoded should match the 'name' value on the attachment. In order for a NotFoundException to not be thrown the
     * following criteria must be met:
     *      - id must match an attachment in key data
     *      - decoded file must match the 'name' value from the attachment
     *      - the depot file from the attachment must exist
     * @param mixed                     $id         the attachment id
     * @param string                    $file       the encoded file name (compared to 'name' from the attachment)
     * @param ConnectionInterface|null  $connection connection to use
     * @return AbstractKey|callable the attachment
     * @throws CommandException
     * @throws NotFoundException
     * @throws P4FileException
     */
    public function fetchAttachment($id, string $file, ConnectionInterface $connection)
    {
        $exception  = null;
        $attachment = null;
        try {
            $attachment  = Attachment::fetch($id, $connection);
            $depotFile   = $attachment->get(Attachment::FIELD_DEPOT_FILE);
            $name        = $attachment->get(Attachment::FIELD_NAME);
            $decodedFile = StringHelper::base64DecodeUrl($file);
            $fileMatch   = $name === $decodedFile;
            if (!$fileMatch || !File::exists($depotFile, $connection, true)) {
                if ($fileMatch) {
                    $this->logger->debug(
                        sprintf(
                            "[%s]: File [%s] does not exist",
                            self::LOG_PREFIX,
                            $depotFile
                        )
                    );
                } else {
                    $this->logger->debug(
                        sprintf(
                            "[%s]: File name [%s] does not match decoded value [%s]",
                            self::LOG_PREFIX,
                            $name,
                            $decodedFile
                        )
                    );
                }
                $exception = true;
            }
        } catch (NotFoundException $nfe) {
            $exception = true;
            $this->logger->debug(sprintf("[%s]: %s", self::LOG_PREFIX, $nfe->getMessage()));
        }
        if ($exception) {
            // We do not want to give too much information away here. Attachment::fetch failure will result in the
            // message 'Cannot fetch entry. Id does not exist.', we'll just catch a NotFoundException and provide
            // a generic message that could be due to the id, a bad file name encode, or the file not existing in
            // the depot
            throw new NotFoundException("Attachment not found");
        }
        return $attachment;
    }

    /**
     * Create attachment key data and save the file to the depot. On completion the temporary file
     * referenced in the data 'tmp_name' will be deleted.
     * A clean-up operation is placed in the task queue for processing 24 hours later so that if the comment
     * associated with an attachment is not posted the attachment will be cleaned up
     * @param array $data   data containing file information
     */
    public function createAttachment(array $data) : Attachment
    {
        $p4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
        $queue      = $this->services->get(Manager::SERVICE);
        $blurString = SwarmBlurHash::createBlur($data[Attachment::FIELD_TYPE], $data[self::TMP_NAME]);

        $attachment = new Attachment($p4Admin);
        $attachment->set(
            [
                Attachment::FIELD_NAME => urldecode($data[Attachment::FIELD_NAME]),
                Attachment::FIELD_SIZE => $data[Attachment::FIELD_SIZE],
                Attachment::FIELD_TYPE => $data[Attachment::FIELD_TYPE],
                Attachment::FIELD_BLUR => $blurString,
            ]
        )->save($data[self::TMP_NAME]);
        $id = $attachment->getId();
        // Add a task to generate a thumbnail
        $queue->addTask(ListenerFactory::ATTACHMENT_THUMBNAIL, $id);
        // Cleanup is required in case the user never posts the comment that the attachment is intended for.
        // Cleanup only performs delete, not obliterate, so the data is still recoverable
        $queue->addTask(ListenerFactory::CLEANUP_ATTACHMENT, $id, null, strtotime('+24 hours'));
        return $attachment;
    }

    /**
     * Gets the thumbnail from the attachment if it exists.
     * The file value is expected to be an encoded value that when decoded should match the 'name' value on the
     * attachment. In order for a NotFoundException to not be thrown the following criteria must be met:
     *      - id must match an attachment in key data
     *      - decoded file must match the 'name' value from the attachment
     *      - the depot file from the attachment must exist
     * @param mixed                     $id         the attachment id
     * @param string                    $file       the encoded file name (compared to 'name' from the attachment)
     * @param ConnectionInterface|null  $connection connection to use
     * @return AbstractKey|callable the attachment thumbnail in base64 encoded format, or null if there is no thumbnail
     * @throws CommandException
     * @throws NotFoundException
     * @throws P4FileException
     */
    public function getThumbnail($id, string $file, ConnectionInterface $connection)
    {
        $p4Admin     = $this->services->get(ConnectionFactory::P4_ADMIN);
        $attachment  = $this->fetchAttachment($id, $file, $connection);
        $fileService = $this->services->get(Services::FILE_SERVICE);
        $depotFile   = $attachment->get(Attachment::FIELD_DEPOT_FILE);
        $fstat       = $fileService->fstat($p4Admin, ['flags' => ['-Oa']], $depotFile)->getData()[0];
        $this->logger->debug(
            sprintf(
                "[%s]: attr-%s for %s %s",
                self::LOG_PREFIX,
                IAttachmentDAO::THUMB_ATTRIBUTE,
                $depotFile,
                isset($fstat['attr-' . IAttachmentDAO::THUMB_ATTRIBUTE]) ? "found" : "not found"
            )
        );
        return $fstat['attr-' . IAttachmentDAO::THUMB_ATTRIBUTE] ?? null;
    }
}
