<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.3/2828589
 */
namespace Attachments\Controller;

use Api\Controller\AbstractRestfulController;
use Api\IRequest;
use Application\Config\IConfigDefinition;
use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Filter\ShorthandBytes;
use Application\I18n\TranslatorFactory;
use Application\Response\CallbackResponse;
use Attachments\Filter\IFile;
use Attachments\Filter\IGetList;
use Attachments\Model\Attachment;
use Attachments\Model\IAttachmentDAO;
use Laminas\Http\Response;
use Laminas\View\Model\JsonModel;
use Exception;
use InvalidArgumentException;
use Record\Exception\NotFoundException;

/**
 * Api controller to manage attachments
 */
class AttachmentApi extends AbstractRestfulController implements IAttachmentApi
{
    /**
     * Get a list of attachments
     * A query parameters of 'ids' must be specified to retrieve attachments matching the ids, as either:
     * - A single string value
     * - A single integer value
     * - An array of values that can contain integers or strings
     *
     * Example 200 response for comment based attachments
     *  {
     *      "error": null,
     *      "messages": [],
     *      "data": {
     *          "attachments": [
     *              {
     *                  "id": 1,
     *                  "name": "phptest.png",
     *                  "type": "image/png",
     *                  "size": 19844,
     *                  "depotFile": "//.swarm/attachments/0000000001-phptest.png",
     *                  "references": {
     *                  "comment": [
     *                      502
     *                  ]
     *               }
     *              },
     *              {
     *                  "id": 2,
     *                  "name": "newcommentlink.png",
     *                  "type": "image/png",
     *                  "size": 16109,
     *                  "depotFile": "//.swarm/attachments/0000000002-newcommentlink.png",
     *                  "references": {
     *                      "comment": [
     *                          503
     *                      ]
     *                  }
     *              }
     *          ]
     *      }
     *  }
     *
     * Unauthorized response 401, when require_login is true and no credentials are provided
     * {
     *   "error": "Unauthorized"
     * }
     *
     * @return JsonModel
     */
    public function getList() : JsonModel
    {
        $errors = null;
        $data   = null;
        try {
            $dao    = $this->services->get(IDao::ATTACHMENT_DAO);
            $filter = $this->services->get(IGetList::FILTER);
            $params = $this->getRequest()->getQuery()->toArray();
            $filter->setData($params);
            if ($filter->isValid()) {
                $fields = $this->getRequest()->getQuery(IRequest::FIELDS);
                $data   = $this->limitFieldsForAll(
                    array_values(
                        $dao->fetchAll(
                            $filter->getValues(),
                            $this->services->get(ConnectionFactory::P4_ADMIN)
                        )->toArray()
                    ),
                    $fields
                );
            } else {
                $errors = $filter->getMessages();
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            }
        } catch (InvalidArgumentException $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            $errors = [$this->buildMessage(Response::STATUS_CODE_400, $e->getMessage())];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_ATTACHMENTS => $data]);
        }
        return $json;
    }

    /**
     * Build headers on the response for the given content type and size
     * @param Response      $response       response to set headers on
     * @param string        $contentType    the content type
     * @param mixed         $contentLength  the content length
     * @return void
     */
    private function buildResponseHeaders(Response $response, string $contentType, $contentLength)
    {
        $cacheOffset = 12 * 60 * 60; // 12 hours
        $response->getHeaders()
            ->addHeaderLine('Content-Type', $contentType)
            ->addHeaderLine('Content-Transfer-Encoding', 'binary')
            ->addHeaderLine('Expires', strftime('%a, %d %b %Y %H:%M:%S %Z', time() + $cacheOffset))
            ->addHeaderLine('Cache-Control', 'max-age=' . $cacheOffset)
            ->addHeaderLine('Content-Length', $contentLength);
    }

    /**
     * Build a CallbackResponse that can be used to stream from a depot location.
     * @param bool $download    whether the download header be added. If true it will be added, if false it will not be
     *                          added unless the attachment content is not a web safe image. For example a request of
     *                          false for a text file will still result in it being downloaded
     * @return CallbackResponse
     */
    private function getContentResponse(bool $download) : CallbackResponse
    {
        $p4Admin    = $this->services->get(ConnectionFactory::P4_ADMIN);
        $dao        = $this->services->get(IDao::ATTACHMENT_DAO);
        $id         = $this->getEvent()->getRouteMatch()->getParam(Attachment::FIELD_ID);
        $file       = $this->getEvent()->getRouteMatch()->getParam(IAttachmentApi::PARAM_FILE);
        $attachment = $dao->fetchAttachment($id, $file, $p4Admin);
        $depot      = $this->services->get(IConfigDefinition::DEPOT_STORAGE);
        $response   = new CallbackResponse();
        $this->buildResponseHeaders(
            $response,
            $attachment->get(Attachment::FIELD_TYPE),
            $attachment->get(Attachment::FIELD_SIZE)
        );
        if (!$download) {
            $download = !$attachment->isWebSafeImage();
        }
        if ($download) {
            $disposition = sprintf(
                "attachment; filename=\"%s\"",
                strtr($attachment->get(Attachment::FIELD_NAME), "\",\r\n", '-')
            );
            $response->getHeaders()->addHeaderLine('Content-Disposition', $disposition);
        }
        $response->setCallback(
            function () use ($depot, $attachment) {
                return $depot->stream($attachment->get(Attachment::FIELD_DEPOT_FILE));
            }
        );
        return $response;
    }

    /**
     * Handle a call to fetch content. Will return a CallbackResponse to stream from a depot on success, or a standard
     * error response on failure
     * @param bool $download    whether the download header be added. If true it will be added, if false it will not be
     *                          added unless the attachment content is not a web safe image. For example a request of
     *                          false for a text file will still result in it being downloaded
     */
    private function handleContentAction(bool $download)
    {
        $errors   = null;
        $response = null;
        try {
            $response = $this->getContentResponse($download);
        } catch (NotFoundException $nfe) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $nfe->getMessage())];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            return $this->error($errors, $this->getResponse()->getStatusCode());
        }
        return $response;
    }

    /**
     * Download the attachment. Will return a CallbackResponse to stream from a depot on success, or a standard
     * error response on failure. On success the response will have the correct content disposition header to enable
     * browsers to save to file.
     *
     * attachments/<id>/content/<encoded-file-name>/download
     * For success:
     *      - id must match an attachment in key data
     *      - encoded-file-name must match the 'name' value from the attachment once decoded
     *      - the depot file from the attachment must exist
     *
     * Example error for an attachment that cannot be loaded
     * {
     *   "error": 404,
     *   "data": null,
     *   "messages": [
     *       "code": 404
     *       "text": "Attachment not found"
     *   ]
     * }
     *
     * Unauthorized response 401, when require_login is true and no credentials are provided
     * {
     *   "error": "Unauthorized"
     * }
     */
    public function downloadContentAction()
    {
        return $this->handleContentAction(true);
    }

    /**
     * Open the attachment. Will return a CallbackResponse to stream from a depot on success, or a standard
     * error response on failure. The response will indicate that it can be opened if the content is a web safe image,
     * otherwise the open call will have the headers set to download the content
     *
     * attachments/<id>/content/<encoded-file-name>
     * For success:
     *      - id must match an attachment in key data
     *      - encoded-file-name must match the 'name' value from the attachment once decoded
     *      - the depot file from the attachment must exist
     *
     * Example error for an attachment that cannot be loaded
     * {
     *   "error": 404,
     *   "data": null,
     *   "messages": [
     *       "code": 404
     *       "text": "Attachment not found"
     *   ]
     * }
     *
     * Unauthorized response 401, when require_login is true and no credentials are provided
     * {
     *   "error": "Unauthorized"
     * }
     */
    public function openContentAction()
    {
        return $this->handleContentAction(false);
    }

    /**
     * Create an attachment. It is expected that the information for the uploaded file will be present in $_FILES.
     * This will happen for example if the 'Content-Type' header is not set and the file is uploaded using FormData.
     * @param mixed     $data   not used by this function. The data for the file upload is sourced from $_FILES.
     *
     * Example 200 response
     *  {
     *      "error": null,
     *      "messages": [],
     *      "data": {
     *          "attachments": [
     *              {
     *                  "id": 1,
     *                  "name": "phptest.png",
     *                  "type": "image/png",
     *                  "size": 19844,
     *                  "depotFile": "//.swarm/attachments/0000000001-phptest.png"
     *              }
     *          ]
     *      }
     *  }
     *
     * Unauthorized response 401, when require_login is true and no credentials are provided
     * {
     *   "error": "Unauthorized"
     * }
     * @return JsonModel
     */
    public function create($data) : JsonModel
    {
        $errors = null;
        try {
            $filter   = $this->services->get(IFile::FILTER);
            $fileData = $_FILES[IAttachmentDAO::FILE] ?? [];
            if (empty($fileData) || (isset($fileData['error']) && $fileData['error'] === 1)) {
                // There is no attachment file, check for limits
                $postLimit   = ShorthandBytes::toBytes(ini_get('post_max_size'));
                $uploadLimit = ShorthandBytes::toBytes(ini_get('upload_max_filesize'));
                $contentSize = $_SERVER['CONTENT_LENGTH'];
                if ($contentSize > $postLimit || $contentSize > $uploadLimit) {
                    // The attachment exceeded one of the php limits
                    $response = $this->getResponse();
                    $response->setStatusCode(Response::STATUS_CODE_413);
                    return $this->error(
                        [
                            $this->buildMessage(
                                Response::STATUS_CODE_413,
                                $this->services->get(TranslatorFactory::SERVICE)->t(
                                    'Exceeded %s',
                                    [
                                        $uploadLimit < $postLimit
                                            ? ini_get('upload_max_filesize')
                                            : ini_get('post_max_size')]
                                )
                            )
                        ],
                        $response->getStatusCode()
                    );
                }
            }

            $filter->setData($fileData);
            if ($filter->isValid()) {
                $dao    = $this->services->get(IDao::ATTACHMENT_DAO);
                $fields = $this->getRequest()->getQuery(IRequest::FIELDS);
                $data[] = $this->limitFields($dao->createAttachment($filter->getValues())->toArray(), $fields);
            } else {
                $errors = $filter->getMessages();
                $this->getResponse()->setStatusCode(Response::STATUS_CODE_400);
            }
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            $json = $this->error($errors, $this->getResponse()->getStatusCode());
        } else {
            $json = $this->success([self::DATA_ATTACHMENTS => $data]);
        }
        return $json;
    }

    /**
     * Get an attachment thumbnail. Will return a response with the thumbnail image, or a standard
     * error response on failure. If no thumbnail is present the response will be empty
     *
     * attachments/<id>/content/<encoded-file-name>/thumb
     * For success:
     *      - id must match an attachment in key data
     *      - encoded-file-name must match the 'name' value from the attachment once decoded
     *      - the depot file from the attachment must exist
     *
     * Example error for an attachment that cannot be loaded
     * {
     *   "error": 404,
     *   "data": null,
     *   "messages": [
     *       "code": 404
     *       "text": "Attachment not found"
     *   ]
     * }
     *
     * Unauthorized response 401, when require_login is true and no credentials are provided
     * {
     *   "error": "Unauthorized"
     * }
     */
    public function thumbAction()
    {
        $errors   = null;
        $response = null;
        try {
            $response = new Response();
            $p4Admin  = $this->services->get(ConnectionFactory::P4_ADMIN);
            $dao      = $this->services->get(IDao::ATTACHMENT_DAO);
            $id       = $this->getEvent()->getRouteMatch()->getParam(Attachment::FIELD_ID);
            $file     = $this->getEvent()->getRouteMatch()->getParam(IAttachmentApi::PARAM_FILE);
            $thumb    = $dao->getThumbnail($id, $file, $p4Admin);
            $length   = 0;
            if ($thumb) {
                $length = strlen($thumb);
            }
            $this->buildResponseHeaders($response, 'image/png', $length);
            $response->setContent($thumb);
        } catch (NotFoundException $nfe) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_404);
            $errors = [$this->buildMessage(Response::STATUS_CODE_404, $nfe->getMessage())];
        } catch (Exception $e) {
            $this->getResponse()->setStatusCode(Response::STATUS_CODE_500);
            $errors = [$this->buildMessage(Response::STATUS_CODE_500, $e->getMessage())];
        }
        if ($errors) {
            return $this->error($errors, $this->getResponse()->getStatusCode());
        }
        return $response;
    }
}
