<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.3/2828589
 */
namespace AiAnalysis\Filter;

use Application\InputFilter\DirectInput;
use Application\InputFilter\InputFilter;
use Application\Validator\GreaterThanInt;
use Interop\Container\ContainerInterface;
use Laminas\Filter\StringTrim;
use Laminas\Validator\StringLength;

/**
 * Class TestRun to filter and validate TestRun
 * @package TestIntegration\Filter
 */
class AiAnalysis extends InputFilter implements IAiAnalysis
{
    private $services;

    /**
     * @inheritDoc
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        $this->services = $services;
        $this->addFileIdFilter();
        $this->addReviewIdFilter();
        $this->addFileNameFilter();
        $this->addDiffContentFilter();
        $this->addToVersionFilter();
        $this->addFromVersionFilter();
        $this->addDiffStartFilter();
        $this->addDiffEndFilter();
        $this->addDigestFilter();
        $this->addIdFilter();
    }

    /**
     * Add filters/validation for 'file id'
     * @return void
     */
    private function addFileIdFilter(): void
    {
        $input = new DirectInput(self::FILE_ID);
        $input->setRequired(true);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()->attach(new StringLength(['min'=>1]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'review id'
     * @return void
     */
    protected function addReviewIdFilter(): void
    {
        $input = new DirectInput(self::REVIEW_ID);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => 0]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'file name'
     * @return void
     */
    private function addFileNameFilter(): void
    {
        $input = new DirectInput(self::FILE_NAME);
        $input->setRequired(true);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()->attach(new StringLength(['min'=>1]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'to version number'
     * @return void
     */
    private function addToVersionFilter(): void
    {
        $input = new DirectInput(self::TO_VERSION);
        $input->setRequired(true);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'from version number'
     * @return void
     */
    private function addFromVersionFilter(): void
    {
        $input = new DirectInput(self::FROM_VERSION);
        $input->setRequired(true);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -2]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'diff content'
     * @return void
     */
    private function addDiffContentFilter(): void
    {
        $input = new DirectInput(self::DIFF_CONTENT);
        $input->setRequired(true);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()->attach(new StringLength(['min'=>1]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'diff start'
     * @return void
     */
    private function addDiffStartFilter(): void
    {
        $input = new DirectInput(self::DIFF_START);
        $input->setRequired(true);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => 0]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'diff end'
     * @return void
     */
    private function addDiffEndFilter(): void
    {
        $input = new DirectInput(self::DIFF_END);
        $input->setRequired(true);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => 0]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'file id'
     * @return void
     */
    private function addDigestFilter(): void
    {
        $input = new DirectInput(self::DIGEST);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()->attach(new StringLength(['min'=>1]));
        $this->add($input);
    }

    /**
     * Add filters/validation for 'id'
     * @return void
     */
    protected function addIdFilter(): void
    {
        $input = new DirectInput(self::ID);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => 0]));
        $this->add($input);
    }
}
