/*
 * Copyright 2011 Perforce Software Inc., All Rights Reserved.
 */
package com.perforce.p4java.impl.generic.core;

import com.perforce.p4java.Log;
import com.perforce.p4java.client.IClientViewMapping;
import com.perforce.p4java.core.IStream;
import com.perforce.p4java.core.IStreamComponentMapping;
import com.perforce.p4java.core.IStreamIgnoredMapping;
import com.perforce.p4java.core.IStreamRemappedMapping;
import com.perforce.p4java.core.IStreamSummary;
import com.perforce.p4java.core.IStreamViewMapping;
import com.perforce.p4java.core.IStreamViewMapping.PathType;
import com.perforce.p4java.core.ViewMap;
import com.perforce.p4java.exception.AccessException;
import com.perforce.p4java.exception.ConnectionException;
import com.perforce.p4java.exception.NullPointerError;
import com.perforce.p4java.exception.P4JavaException;
import com.perforce.p4java.exception.RequestException;
import com.perforce.p4java.impl.generic.client.ClientView;
import com.perforce.p4java.impl.mapbased.MapKeys;
import com.perforce.p4java.option.server.StreamOptions;
import com.perforce.p4java.server.IOptionsServer;
import com.perforce.p4java.server.IServer;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Map;

import static com.perforce.p4java.impl.mapbased.MapKeys.SPACE;

/**
 * Simple default implementation class for the IStream interface.
 */

public class Stream extends StreamSummary implements IStream {

	protected ViewMap<IStreamViewMapping> streamView = null;
	protected ViewMap<IStreamRemappedMapping> remappedView = null;
	protected ViewMap<IStreamIgnoredMapping> ignoredView = null;
	protected ViewMap<IClientViewMapping> clientView = null;

	protected ViewMap<IStreamComponentMapping> components = null;
	protected List<IExtraTag> extraTags = null;

	/**
	 * Default description for use in newStream method when no explicit
	 * description is given.
	 */
	public static final String DEFAULT_DESCRIPTION = "New stream created by P4Java";

	/**
	 * Simple default generic IExtraTag implementation class.
	 */
	public static class ExtraTag implements IExtraTag {

		private String name = null;
		private String type = null;
		private String value = null;

		/**
		 * Default constructor; sets all fields to false.
		 */
		public ExtraTag() {
		}

		/**
		 * Explicit-value constructor.
		 *
		 * @param name  name
		 * @param type  type
		 * @param value value
		 */
		public ExtraTag(String name, String type, String value) {
			if (name == null) {
				throw new NullPointerError("null name in Stream.ExtraTag constructor.");
			}
			if (type == null) {
				throw new NullPointerError("null type in Stream.ExtraTag constructor.");
			}
			if (value == null) {
				throw new NullPointerError("null value in Stream.ExtraTag constructor.");
			}

			this.name = name;
			this.type = type;
			this.value = value;
		}

		/**
		 * @return name
		 */
		public String getName() {
			return this.name;
		}

		/**
		 * @param name name
		 */
		public void setName(String name) {
			this.name = name;
		}

		/**
		 * @return type
		 */
		public String getType() {
			return this.type;
		}

		/**
		 * @param type type
		 */
		public void setType(String type) {
			this.type = type;
		}

		/**
		 * @return value
		 */
		public String getValue() {
			return this.value;
		}

		/**
		 * @param value value
		 */
		public void setValue(String value) {
			this.value = value;
		}
	}

	/**
	 * Simple factory / convenience method for creating a new local Stream
	 * object with defult values.
	 *
	 * @param server           server
	 * @param streamPath       stream path
	 * @param type             type
	 * @param parentStreamPath parent stream
	 * @param name             name
	 * @param description      description
	 * @param options          options
	 * @param viewPaths        view paths
	 * @param remappedPaths    remapped paths
	 * @param ignoredPaths     ignored paths
	 * @return stream object
	 */
	public static Stream newStream(IOptionsServer server, String streamPath,
								   String type, String parentStreamPath, String name,
								   String description, String options, String[] viewPaths,
								   String[] remappedPaths, String[] ignoredPaths) {

		return getStream(server, streamPath, type, parentStreamPath, name,
				description, options, viewPaths, remappedPaths, ignoredPaths,
				null, null);
	}

	/**
	 * Simple factory / convenience method for creating a new local Stream
	 * object with defult values.
	 *
	 * @param server           non-null server to be associated with the new stream spec.
	 * @param streamPath       non-null stream's path in a stream depot, of the form
	 *                         //depotname/streamname.
	 * @param type             non-null stream type of 'mainline', 'development', or
	 *                         'release'.
	 * @param parentStreamPath parent of this stream. Can be null if the stream type is
	 *                         'mainline', otherwise must be set to an existing stream.
	 * @param name             an alternate name of the stream, for use in display outputs.
	 *                         Defaults to the 'streamname' portion of the stream path. Can
	 *                         be changed.
	 * @param description      if not null, used as the new stream spec's description field;
	 *                         if null, uses the Stream.DEFAULT_DESCRIPTION field.
	 * @param options          flags to configure stream behavior: allsubmit/ownersubmit
	 *                         [un]locked [no]toparent [no]fromparent mergedown/mergeany.
	 * @param viewPaths        one or more lines that define file paths in the stream view.
	 *                         Each line is of the form: path_type view_path depot_path
	 * @param remappedPaths    optional; one or more lines that define how stream view paths
	 *                         are to be remapped in client views. Each line is of the form:
	 *                         view_path_1 view_path_2
	 * @param ignoredPaths     optional; a list of file or directory names to be ignored in
	 *                         client views. For example:
	 *                         /tmp      # ignores files named 'tmp'
	 *                         /tmp/...  # ignores dirs named 'tmp'
	 *                         .tmp      # ignores file names ending in '.tmp'
	 * @param clientViewPaths  automatically generated; maps files in the depot to files in
	 *                         your client workspace. For example:
	 *                         //p4java_stream/dev/... ...
	 *                         //p4java_stream/dev/readonly/sync/p4cmd/%%1 readonly/sync/p4cmd/%%1
	 *                         -//p4java_stream/.../temp/... .../temp/...
	 *                         -//p4java_stream/....class ....class
	 * @return new local Stream object.
	 */
	public static Stream newStream(IOptionsServer server, String streamPath,
								   String type, String parentStreamPath, String name,
								   String description, String options, String[] viewPaths,
								   String[] remappedPaths, String[] ignoredPaths,
								   String[] clientViewPaths) {
		return getStream(server, streamPath, type, parentStreamPath, name, description, options, viewPaths, remappedPaths,
				ignoredPaths, clientViewPaths, null);
	}

	/**
	 * Simple factory / convenience method for creating a new local Stream
	 * object with defult values.
	 *
	 * @param server           non-null server to be associated with the new stream spec.
	 * @param streamPath       non-null stream's path in a stream depot, of the form
	 *                         //depotname/streamname.
	 * @param type             non-null stream type of 'mainline', 'development', or
	 *                         'release'.
	 * @param parentStreamPath parent of this stream. Can be null if the stream type is
	 *                         'mainline', otherwise must be set to an existing stream.
	 * @param name             an alternate name of the stream, for use in display outputs.
	 *                         Defaults to the 'streamname' portion of the stream path. Can
	 *                         be changed.
	 * @param description      if not null, used as the new stream spec's description field;
	 *                         if null, uses the Stream.DEFAULT_DESCRIPTION field.
	 * @param options          flags to configure stream behavior: allsubmit/ownersubmit
	 *                         [un]locked [no]toparent [no]fromparent mergedown/mergeany.
	 * @param viewPaths        one or more lines that define file paths in the stream view.
	 *                         Each line is of the form: path_type view_path depot_path
	 * @param remappedPaths    optional; one or more lines that define how stream view paths
	 *                         are to be remapped in client views. Each line is of the form:
	 *                         view_path_1 view_path_2
	 * @param ignoredPaths     optional; a list of file or directory names to be ignored in
	 *                         client views. For example:
	 *                         /tmp      # ignores files named 'tmp'
	 *                         /tmp/...  # ignores dirs named 'tmp'
	 *                         .tmp      # ignores file names ending in '.tmp'
	 * @param clientViewPaths  automatically generated; maps files in the depot to files in
	 *                         your client workspace. For example:
	 *                         //p4java_stream/dev/... ...
	 *                         //p4java_stream/dev/readonly/sync/p4cmd/%%1 readonly/sync/p4cmd/%%1
	 *                         -//p4java_stream/.../temp/... .../temp/...
	 *                         -//p4java_stream/....class ....class
	 * @param components       Add stream components.
	 *                         Syntax: component_type component_folder component_stream[@[change | automatic_label]]
	 *                         eg: readonly dirB //streams/B
	 * @return new local Stream object.
	 */
	public static Stream newStream(IOptionsServer server, String streamPath,
								   String type, String parentStreamPath, String name,
								   String description, String options, String[] viewPaths,
								   String[] remappedPaths, String[] ignoredPaths,
								   String[] clientViewPaths, String[] components) {
		return getStream(server, streamPath, type, parentStreamPath, name, description, options, viewPaths, remappedPaths,
				ignoredPaths, clientViewPaths, components);
	}

	private static Stream getStream(IOptionsServer server, String streamPath, String type, String parentStreamPath,
									String name, String description, String options, String[] viewPaths, String[] remappedPaths,
									String[] ignoredPaths, String[] clientViewPaths, String[] components) {
		if (server == null) {
			throw new NullPointerError("null server in Stream.newStream()");
		}
		if (type == null) {
			throw new NullPointerError("null stream type in Stream.newStream()");
		}
		if (streamPath == null) {
			throw new NullPointerError("null stream in Stream.newStream()");
		}

		ViewMap<IStreamViewMapping> streamView = new ViewMap<>();
		if (viewPaths != null) {
			int i = 0;
			for (String mapping : viewPaths) {
				if (mapping == null) {
					throw new NullPointerError("null view mapping string passed to Stream.newStream");
				}
				streamView.addEntry(new StreamViewMapping(i, mapping));
				i++;
			}
		} else {
			streamView.addEntry(new StreamViewMapping(0, PathType.SHARE, "...", null));
		}

		ViewMap<IStreamRemappedMapping> remappedView = null;
		if (remappedPaths != null) {
			remappedView = new ViewMap<>();
			int i = 0;
			for (String mapping : remappedPaths) {
				if (mapping == null) {
					throw new NullPointerError("null remapped mapping string passed to Stream.newStream");
				}
				remappedView.addEntry(new StreamRemappedMapping(i, mapping));
				i++;
			}
		}

		ViewMap<IStreamIgnoredMapping> ignoredView = null;
		if (ignoredPaths != null) {
			ignoredView = new ViewMap<>();
			int i = 0;
			for (String mapping : ignoredPaths) {
				if (mapping == null) {
					throw new NullPointerError("null ignored path string passed to Stream.newStream");
				}
				ignoredView.addEntry(new StreamIgnoredMapping(i, mapping));
				i++;
			}
		}

		List<IClientViewMapping> clientView = null;
		if (clientViewPaths != null) {
			clientView = new ArrayList<>();
			int i = 0;
			for (String mapping : clientViewPaths) {
				if (mapping == null) {
					throw new NullPointerError("null client view mapping string passed to Stream.newStream");
				}
				clientView.add(new ClientView.ClientViewMapping(i, mapping));
				i++;
			}
		}

		ViewMap<IStreamComponentMapping> streamComponents = null;
		if (components != null) {
			streamComponents = new ViewMap<>();
			int i = 0;
			for (String mapping : components) {
				if (mapping == null) {
					throw new NullPointerError("null client view mapping string passed to Stream.newStream");
				}
				streamComponents.addEntry(new StreamComponentMapping(i, mapping));
				i++;
			}
		}

		IOptions streamOptions = new Options();
		if (options != null) {
			streamOptions = new Options(options);
		}

		if (parentStreamPath == null) {
			parentStreamPath = "none";
		}

		if (name == null) {
			int idx = streamPath.lastIndexOf("/");
			if (idx != -1 && idx < (streamPath.length() - 1)) {
				name = streamPath.substring(idx + 1);
			}
		}

		return new Stream(
				streamPath,
				Type.fromString(type.toUpperCase(Locale.ENGLISH)),
				parentStreamPath,
				null,
				null,
				name,
				description == null ? Stream.DEFAULT_DESCRIPTION : description,
				server.getUserName(),
				streamOptions,
				streamView,
				remappedView,
				ignoredView,
				null,
				streamComponents);
	}

	/**
	 * Simple default implementation of the IStreamViewMapping interface.
	 */
	public static class StreamViewMapping extends MapEntry implements
			IStreamViewMapping {

		protected PathType pathType = null;

		/**
		 * Default constructor -- calls super() only.
		 */
		public StreamViewMapping() {
			super();
		}

		/**
		 * Explicit value constructor -- calls super(order, target, targetSpec).
		 *
		 * @param order     order
		 * @param pathType  path type
		 * @param viewPath  view path
		 * @param depotPath depot path
		 */
		public StreamViewMapping(int order, PathType pathType, String viewPath,
								 String depotPath) {
			super(order, viewPath, depotPath);
			if (pathType == null) {
				throw new NullPointerError("null stream view path type passed to Stream.StreamViewMapping constructor.");
			}
			this.pathType = pathType;
		}

		/**
		 * Construct a mapping from the passed-in string, which is assumed to be
		 * in the format.
		 *
		 * @param order         order
		 * @param rawViewString raw view string
		 */
		public StreamViewMapping(int order, String rawViewString) {
			this.order = order;
			if (rawViewString != null) {
				String viewString = stripComments(rawViewString);
				// The first part of the stream path should be the type
				int idx = viewString.indexOf(" ");
				if (idx != -1) {
					this.pathType = PathType.fromString(viewString.substring(0, idx));
					// Remove the type part from the original path
					viewString = viewString.substring(idx + 1);
				}
				String[] entries = parseViewMappingString(viewString);
				this.type = EntryType.fromString(entries[0]);
				this.left = stripTypePrefix(entries[0]);
				this.right = entries[1];
				this.comment = parseComments(rawViewString);
			}
		}

		/**
		 * @see com.perforce.p4java.core.IStreamViewMapping#getPathType()
		 */
		public PathType getPathType() {
			return this.pathType;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamViewMapping#setPathType(com.perforce.p4java.core.IStreamViewMapping.PathType)
		 */
		public void setPathType(PathType pathType) {
			this.pathType = pathType;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamViewMapping#getViewPath()
		 */
		public String getViewPath() {
			return this.left;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamViewMapping#setViewPath(java.lang.String)
		 */
		public void setViewPath(String viewPath) {
			this.left = viewPath;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamViewMapping#getDepotPath()
		 */
		public String getDepotPath() {
			return this.right;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamViewMapping#setDepotPath(java.lang.String)
		 */
		public void setDepotPath(String depotPath) {
			this.right = depotPath;
		}

		@Override
		public String toString(String sepString, boolean quoteBlanks) {
			String returnedString = super.toString(sepString, quoteBlanks);
			if (this.getPathType() != null) {
				return this.getPathType().getValue() + " " + returnedString;
			}
			return returnedString;
		}

	}

	/**
	 * Simple default implementation of the IStreamRemappedMapping interface.
	 */
	public static class StreamRemappedMapping extends MapEntry implements
			IStreamRemappedMapping {

		/**
		 * Default constructor -- calls super() only.
		 */
		public StreamRemappedMapping() {
			super();
		}

		/**
		 * Explicit value constructor -- calls super(order, target, targetSpec).
		 *
		 * @param order          order
		 * @param leftRemapPath  left remapped path
		 * @param rightRemapPath right remapped path
		 */
		public StreamRemappedMapping(int order, String leftRemapPath,
									 String rightRemapPath) {
			super(order, leftRemapPath, rightRemapPath);
		}

		/**
		 * Construct a mapping from the passed-in string, which is assumed to be
		 * in the format described in MapEntry.parseViewString(String).
		 *
		 * @param order      order
		 * @param viewString view string
		 */
		public StreamRemappedMapping(int order, String viewString) {
			super(order, viewString);
		}

		/**
		 * @see com.perforce.p4java.core.IStreamRemappedMapping#getLeftRemapPath()
		 */
		public String getLeftRemapPath() {
			return this.left;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamRemappedMapping#setLeftRemapPath(java.lang.String)
		 */
		public void setLeftRemapPath(String leftRemapPath) {
			this.left = leftRemapPath;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamRemappedMapping#getRightRemapPath()
		 */
		public String getRightRemapPath() {
			return this.right;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamRemappedMapping#setRightRemapPath(java.lang.String)
		 */
		public void setRightRemapPath(String rightRemapPath) {
			this.right = rightRemapPath;
		}
	}

	/**
	 * Simple default implementation of the IStreamIgnoredMapping interface.
	 */
	public static class StreamIgnoredMapping extends MapEntry implements
			IStreamIgnoredMapping {

		protected PathType pathType = PathType.SHARE;

		/**
		 * Default constructor -- calls super() only.
		 */
		public StreamIgnoredMapping() {
			super();
		}

		/**
		 * Explicit value constructor -- calls super(order, target, targetSpec).
		 *
		 * @param order      order
		 * @param ignorePath ignore path
		 */
		public StreamIgnoredMapping(int order, String ignorePath) {
			super(order, ignorePath, null);
		}

		/**
		 * @see com.perforce.p4java.core.IStreamIgnoredMapping#getIgnorePath()
		 */
		public String getIgnorePath() {
			return this.left;
		}

		/**
		 * @see com.perforce.p4java.core.IStreamIgnoredMapping#setIgnorePath(java.lang.String)
		 */
		public void setIgnorePath(String ignorePath) {
			this.left = ignorePath;
		}
	}

	public static class StreamComponentMapping extends MapEntry implements IStreamComponentMapping {
		Type type = null;

		public StreamComponentMapping(int order, Type type, String folder, String stream) {
			super(order, folder, stream);
			this.type = type;
		}

		public StreamComponentMapping(int order, String rawComponentString) {
			String[] s = rawComponentString.split(SPACE);
			if (s.length == 3) {
				this.order = order;
				this.type = Type.fromString(s[0]);
				this.left = s[1];
				this.right = s[2];
			} else {
				Log.error("Bad stream components mapping. " + rawComponentString);
			}
		}

		@Override
		public Type getComponentType() {
			return type;
		}

		@Override
		public void setComponentType(Type type) {
			this.type = type;
		}

		@Override
		public String getDirectory() {
			return left;
		}

		@Override
		public void setDirectory(String directory) {
			this.left = directory;
		}

		@Override
		public String getStream() {
			return right;
		}

		@Override
		public void setStream(String stream) {
			this.right = stream;
		}

		@Override
		public String toString() {
			return type.getText() + SPACE + left + SPACE + right;
		}
	}

	/**
	 * Default constructor. All fields set to null or false.
	 */
	public Stream() {
		super();
	}

	/**
	 * Construct a new Stream from explicit field values.
	 *
	 * @param stream       stream
	 * @param type         type
	 * @param parent       parent
	 * @param accessed     accessed
	 * @param updated      updated
	 * @param name         name
	 * @param description  description
	 * @param ownerName    ownerName
	 * @param options      options
	 * @param streamView   streamView
	 * @param remappedView remappedView
	 * @param ignoredView  ignoredView
	 */
	public Stream(String stream, Type type, String parent, Date accessed,
				  Date updated, String name, String description, String ownerName,
				  IOptions options, ViewMap<IStreamViewMapping> streamView,
				  ViewMap<IStreamRemappedMapping> remappedView,
				  ViewMap<IStreamIgnoredMapping> ignoredView) {

		this(stream, type, parent, accessed, updated, name, description,
				ownerName, options, streamView, remappedView, ignoredView, null, null);
	}

	/**
	 * Construct a new Stream from explicit field values.
	 *
	 * @param stream       stream
	 * @param type         type
	 * @param parent       parent
	 * @param accessed     accessed
	 * @param updated      updated
	 * @param name         name
	 * @param description  description
	 * @param ownerName    ownerName
	 * @param options      options
	 * @param streamView   streamView
	 * @param remappedView remappedView
	 * @param ignoredView  ignoredView
	 * @param clientView   clientView
	 */
	public Stream(String stream, Type type, String parent, Date accessed,
				  Date updated, String name, String description, String ownerName,
				  IOptions options, ViewMap<IStreamViewMapping> streamView,
				  ViewMap<IStreamRemappedMapping> remappedView,
				  ViewMap<IStreamIgnoredMapping> ignoredView, ViewMap<IClientViewMapping> clientView) {
		this(stream, type, parent, accessed, updated, name, description,ownerName, options, streamView, remappedView,
				ignoredView, clientView, null);
	}

	/**
	 * Construct a new Stream from explicit field values.
	 *
	 * @param stream       stream
	 * @param type         type
	 * @param parent       parent
	 * @param accessed     accessed
	 * @param updated      updated
	 * @param name         name
	 * @param description  description
	 * @param ownerName    ownerName
	 * @param options      options
	 * @param streamView   streamView
	 * @param remappedView remappedView
	 * @param ignoredView  ignoredView
	 * @param clientView   clientView
	 * @param components   components
	 */
	public Stream(String stream, Type type, String parent, Date accessed,
				  Date updated, String name, String description, String ownerName,
				  IOptions options, ViewMap<IStreamViewMapping> streamView,
				  ViewMap<IStreamRemappedMapping> remappedView,
				  ViewMap<IStreamIgnoredMapping> ignoredView,
				  ViewMap<IClientViewMapping> clientView, ViewMap<IStreamComponentMapping> components) {
		setStream(stream);
		setType(type);
		setParent(parent);
		setAccessed(accessed);
		setUpdated(updated);
		setName(name);
		setOwnerName(ownerName);
		setDescription(description);
		setOptions(options);

		this.streamView = streamView;
		this.remappedView = remappedView;
		this.ignoredView = ignoredView;
		this.clientView = clientView;
		this.components = components;
	}

	/**
	 * Construct a Stream from a map passed back from the Perforce server in
	 * response to a getStream command.
	 *
	 * @param map    spec map
	 * @param server server
	 */
	public Stream(Map<String, Object> map, IServer server) {
		super(map, false);

		this.server = server;
		this.streamView = new ViewMap<>();
		this.remappedView = new ViewMap<>();
		this.ignoredView = new ViewMap<>();
		this.clientView = new ViewMap<>();
		this.extraTags = new ArrayList<>();
		this.components = new ViewMap<>();

		if (map != null) {
			String key = MapKeys.PATHS_KEY;
			String commentKey = MapKeys.PATHS_KEY + "Comment";
			for (int i = 0; ; i++) {
				if (!map.containsKey(key + i) && !map.containsKey(commentKey + i)) {
					break;
				}
				StreamViewMapping viewMap = new StreamViewMapping();
				try {
					if (map.get(commentKey + i) != null) {
						if (map.get(commentKey + i) != null) {
							String comment = (String) map.get(commentKey + i);
							comment = comment.substring(comment.indexOf("##") + 2);
							comment = comment.trim();
							viewMap.setComment(comment);
						}
					}

					if (map.get(key + i) != null) {
						PathType type = null;
						String path = (String) map.get(key + i);
						// The first part of the stream path should be the type
						int idx = path.indexOf(" ");
						if (idx != -1) {
							type = PathType.fromString(path.substring(0, idx));
							// Remove the type part from the original path
							path = path.substring(idx + 1);
						}
						String[] matchStrs = MapEntry
								.parseViewMappingString(path);
						viewMap.setOrder(i);
						viewMap.setPathType(type);
						viewMap.setLeft(matchStrs[0]);
						viewMap.setRight(matchStrs[1]);
					}
				} catch (Throwable thr) {
					Log.error("Unexpected exception in Stream map-based constructor: "
							+ thr.getLocalizedMessage());
					Log.exception(thr);
				}
				this.streamView.getEntryList().add(viewMap);
			}

			key = MapKeys.REMAPPED_KEY;
			commentKey = MapKeys.REMAPPED_KEY + "Comment";
			for (int i = 0; ; i++) {
				if (!map.containsKey(key + i) && !map.containsKey(commentKey + i)) {
					break;
				}
				StreamRemappedMapping remappedMap = new StreamRemappedMapping();
				try {
					if (map.get(commentKey + i) != null) {
						String comment = (String) map.get(commentKey + i);
						comment = comment.substring(comment.indexOf("##") + 2);
						comment = comment.trim();
						remappedMap.setComment(comment);
					}

					if (map.get(key + i) != null) {
						String path = (String) map.get(key + i);
						String[] matchStrs = MapEntry
								.parseViewMappingString(path);
						remappedMap.setOrder(i);
						remappedMap.setLeft(matchStrs[0]);
						remappedMap.setRight(matchStrs[1]);
					}
				} catch (Throwable thr) {
					Log.error("Unexpected exception in Stream map-based constructor: "
							+ thr.getLocalizedMessage());
					Log.exception(thr);
				}
				this.remappedView.getEntryList().add(remappedMap);
			}


			key = MapKeys.IGNORED_KEY;
			commentKey = MapKeys.IGNORED_KEY + "Comment";
			for (int i = 0; ; i++) {
				if (!map.containsKey(key + i) && !map.containsKey(commentKey + i)) {
					break;
				}
				StreamIgnoredMapping ignoreMap = new StreamIgnoredMapping();
				try {
					if (map.get(commentKey + i) != null) {

						String comment = (String) map.get(commentKey + i);
						comment = comment.substring(comment.indexOf("##") + 2);
						comment = comment.trim();
						ignoreMap.setComment(comment);
					}

					if (map.get(key + i) != null) {
						String path = (String) map.get(key + i);
						String[] matchStrs = MapEntry
								.parseViewMappingString(path);
						ignoreMap.setOrder(i);
						ignoreMap.setLeft(matchStrs[0]);
					}
				} catch (Throwable thr) {
					Log.error("Unexpected exception in Stream map-based constructor: "
							+ thr.getLocalizedMessage());
					Log.exception(thr);
				}
				this.ignoredView.getEntryList().add(ignoreMap);
			}

			key = MapKeys.VIEW_KEY;
			for (int i = 0; ; i++) {
				if (!map.containsKey(key + i)) {
					break;
				} else if (map.get(key + i) != null) {
					try {
						String path = (String) map.get(key + i);
						this.clientView.getEntryList().add(
								new ClientView.ClientViewMapping(i, path));

					} catch (Throwable thr) {
						Log.error("Unexpected exception in Stream map-based constructor: "
								+ thr.getLocalizedMessage());
						Log.exception(thr);
					}
				}
			}
			key = MapKeys.EXTRATAG_KEY;
			for (int i = 0; ; i++) {
				if (!map.containsKey(key + i)) {
					break;
				} else if (map.get(key + i) != null) {
					try {
						String tagName = (String) map.get(key + i);
						String tagType = (String) map.get(MapKeys.EXTRATAGTYPE_KEY + i);
						String tagValue = (String) map.get(tagName);
						this.extraTags.add(new ExtraTag(tagName, tagType, tagValue));

					} catch (Throwable thr) {
						Log.error("Unexpected exception in Stream map-based constructor: "
								+ thr.getLocalizedMessage());
						Log.exception(thr);
					}
				}
			}

			key = MapKeys.COMPONENTS_KEY;
			for (int i = 0; ; i++) {
				if (!map.containsKey(key + i)) {
					break;
				} else if (map.get(key + i) != null) {
					try {
						String components = (String) map.get(key + i);
						this.components.getEntryList().add(new StreamComponentMapping(i, components));

					} catch (Throwable thr) {
						Log.error("Unexpected exception in Stream map-based constructor: "
								+ thr.getLocalizedMessage());
						Log.exception(thr);
					}
				}
			}

		}
	}

	/**
	 * Construct a new Stream from the passed-in summary stream spec. If the
	 * summary is null, this is equivalent to calling the default Stream
	 * constructor; otherwise after name initialization a refresh() is done on
	 * the new (empty) Stream.
	 *
	 * @param summary summary class
	 * @throws ConnectionException if the Perforce server is unreachable or is not connected.
	 * @throws RequestException    if the Perforce server encounters an error during its
	 *                             processing of the request
	 * @throws AccessException     if the Perforce server denies access to the caller
	 */

	public Stream(IStreamSummary summary) throws ConnectionException,
			RequestException, AccessException {
		super(false);
		this.streamView = new ViewMap<>();
		if (summary != null) {
			this.setName(summary.getName());

			if (this.getName() != null) {
				this.refresh();
			}
		}
	}

	private void updateFlags() {
	}

	/**
	 * This method will refresh by getting the complete stream model. If this
	 * refresh is successful then this stream will be marked as complete.
	 *
	 * @see com.perforce.p4java.impl.generic.core.ServerResource#refresh()
	 */
	public void refresh() throws ConnectionException, RequestException, AccessException {
		IServer refreshServer = server;
		String refreshStreamPath = getStream();
		if (refreshServer != null && refreshStreamPath != null) {
			try {
				IStream refreshedStream = refreshServer.getStream(refreshStreamPath);
				if (refreshedStream != null) {
					if (refreshedStream.getRawFields() != null) {
						setRawFields(refreshedStream.getRawFields());
					} else {
						clearRawFields();
					}
					// TODO migrate views to rawFields map in ServerResource
					setStreamView(refreshedStream.getStreamView());
					setRemappedView(refreshedStream.getRemappedView());
					setIgnoredView(refreshedStream.getIgnoredView());
					setExtraTags(refreshedStream.getExtraTags());
				}
			} catch (P4JavaException exc) {
				throw new RequestException(exc.getMessage(), exc);
			}
		}
		updateFlags();
	}

	/**
	 * @see com.perforce.p4java.impl.generic.core.ServerResource#update()
	 */
	public void update() throws ConnectionException, RequestException,
			AccessException {
		try {
			this.server.updateStream(this, null);
		} catch (P4JavaException exc) {
			throw new RequestException(exc.getMessage(), exc);
		}
	}

	/**
	 * @see com.perforce.p4java.impl.generic.core.ServerResource#update(boolean)
	 */
	public void update(boolean force) throws ConnectionException, RequestException, AccessException {
		try {
			this.server.updateStream(this, new StreamOptions().setForceUpdate(force));
		} catch (P4JavaException exc) {
			throw new RequestException(exc.getMessage(), exc);
		}
	}

	/**
	 * @see com.perforce.p4java.core.IStream#getStreamView()
	 */
	public ViewMap<IStreamViewMapping> getStreamView() {
		return this.streamView;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#setStreamView(com.perforce.p4java.core.ViewMap)
	 */
	public void setStreamView(ViewMap<IStreamViewMapping> streamView) {
		this.streamView = streamView;
	}

	/**
	 * @see com.perforce.p4java.core.IServerResource#setServer(com.perforce.p4java.server.IServer)
	 */
	public void setServer(IOptionsServer server) {
		this.server = server;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#getRemappedView()
	 */
	public ViewMap<IStreamRemappedMapping> getRemappedView() {
		return this.remappedView;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#setStreamView(com.perforce.p4java.core.ViewMap)
	 */
	public void setRemappedView(ViewMap<IStreamRemappedMapping> remappedView) {
		this.remappedView = remappedView;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#getIgnoredView()
	 */
	public ViewMap<IStreamIgnoredMapping> getIgnoredView() {
		return this.ignoredView;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#setIgnoredView(com.perforce.p4java.core.ViewMap)
	 */
	public void setIgnoredView(ViewMap<IStreamIgnoredMapping> ignoredView) {
		this.ignoredView = ignoredView;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#getClientView()
	 */
	public ViewMap<IClientViewMapping> getClientView() {
		return this.clientView;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#setClientView(com.perforce.p4java.core.ViewMap)
	 */
	public void setClientView(ViewMap<IClientViewMapping> clientView) {
		this.clientView = clientView;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#getExtraTags()
	 */
	public List<IExtraTag> getExtraTags() {
		return this.extraTags;
	}

	/**
	 * @see com.perforce.p4java.core.IStream#setExtraTags(java.util.List)
	 */
	public void setExtraTags(List<IExtraTag> extraTags) {
		this.extraTags = extraTags;
	}

	@Override
	public ViewMap<IStreamComponentMapping> getComponents() {
		return components;
	}

	@Override
	public void setComponents(ViewMap<IStreamComponentMapping> components) {
		this.components = components;
	}
}
