package com.perforce.p4java;

import com.perforce.p4java.server.callback.ILogCallback;
import com.perforce.p4java.server.callback.ILogCallback.LogTraceLevel;

import static com.perforce.p4java.common.base.StringHelper.format;
import static java.util.Objects.nonNull;

/**
 * Simple P4Java-wide logger class based on the ILogCallback callback
 * interface. Useful for letting P4Java consumers report errors,
 * warnings, etc., generated within P4Java into their own logs.
 * <p>
 * Note that absolutely no guarantees or specifications
 * are made about the format or content of strings that are passed through
 * the logging mechanism, but in general all such strings are useful for
 * Perforce support staff, and many info and stats strings passed to
 * the callback may be generally useful for API consumers.
 * <p>
 * The Log class is itself used from within P4Java to report log
 * messages; the intention here is to allow consumers to call the
 * setLogCallback static method with a suitable log listener that
 * the P4Java API will log to internally. Most of the methods below
 * besides the setLogCallback method are mainly intended for API-internal
 * use, but participating apps may find the other methods useful for
 * interpolating marker text or other messages to the API's log.
 */
public class Log {

	private static ILogCallback logCallback = null;

	/**
	 * @return the current log callback, if any. May return null.
	 */
	public static ILogCallback getLogCallback() {
		return Log.logCallback;
	}

	/**
	 * Set the P4Java API's internal logger to log to the passed-in
	 * ILogCallback log callback. If the passed-in parameter is null,
	 * no logging will be performed. The caller is responsible for ensuring
	 * that there are not thread issues with the passed-in callback, and
	 * that callbacks to the callback object will not block or deadlock.
	 *
	 * @param logCallback callback to be used by P4Java to report log messages to; if null, stop
	 *                    logging.
	 * @return the previous callback registered, or null if no such callback existed.
	 */
	public static ILogCallback setLogCallback(ILogCallback logCallback) {
		ILogCallback oldCallback = Log.logCallback;
		Log.logCallback = logCallback;
		return oldCallback;
	}

	/**
	 * Report a P4Java-internal error to the log callback (if it exists).
	 *
	 * @param errorString non-null error string.
	 * @param args        args
	 */
	public static void error(String errorString, Object... args) {
		if (nonNull(logCallback)) {
			String errorMessage = errorString;
			if (nonNull(args)) {
				errorMessage = format(errorString, args);
			}
			logCallback.internalError(errorMessage);
		}
	}

	/**
	 * Report a P4Java-internal warning to the log callback (if it exists).
	 *
	 * @param warnString non-null warning message.
	 * @param args       args
	 */
	public static void warn(String warnString, Object... args) {
		if (nonNull(logCallback)) {
			String warningMessage = warnString;
			if (nonNull(args)) {
				warningMessage = format(warnString, args);
			}
			logCallback.internalWarn(warningMessage);
		}
	}

	/**
	 * Report a P4Java-internal informational event to the log callback (if it exists).
	 *
	 * @param infoString non-null info message.
	 * @param args       args
	 */
	public static void info(String infoString, Object... args) {
		if (nonNull(logCallback)) {
			String infoMessage = infoString;
			if (nonNull(args)) {
				infoMessage = format(infoString, args);
			}
			logCallback.internalInfo(infoMessage);
		}
	}

	/**
	 * Report a P4Java-internal statistics message to the log callback (if it exists).
	 *
	 * @param statsString non-null stats message.
	 * @param args        args
	 */
	public static void stats(String statsString, Object... args) {
		if (nonNull(logCallback)) {
			String statsMessage = statsString;
			if (nonNull(args)) {
				statsMessage = format(statsString, args);
			}
			logCallback.internalStats(statsMessage);
		}
	}

	/**
	 * Report a P4Java-internal unexpected exception to the log callback
	 * (if it exists).
	 *
	 * @param thr non-null Throwable
	 */
	public static void exception(Throwable thr) {
		if (nonNull(logCallback) && nonNull(thr)) {
			logCallback.internalException(thr);
		}
	}

	/**
	 * Report a P4Java-internal trace message to the log callback
	 * (if it exists).
	 *
	 * @param traceLevel   traceLevel
	 * @param traceMessage traceMessage
	 */
	public static void trace(LogTraceLevel traceLevel, String traceMessage) {
		if (nonNull(logCallback) && nonNull(traceLevel) && nonNull(traceMessage) && isTracingAtLevel(traceLevel)) {
			logCallback.internalTrace(traceLevel, traceMessage);
		}
	}

	/**
	 * @param traceLevel traceLevel
	 * @return true if the
	 */
	public static boolean isTracingAtLevel(LogTraceLevel traceLevel) {
		return nonNull(logCallback) && nonNull(logCallback.getTraceLevel()) && (traceLevel.compareTo(logCallback.getTraceLevel()) <= 0);
	}
}
