<?php

namespace Projects\Validator;

use Application\Config\IDao;
use Application\Connection\ConnectionFactory;
use Application\Validator\AbstractValidator;
use Application\Validator\ConnectedAbstractValidator;
use Laminas\Validator\NotEmpty;
use P4\Validate\GroupName;
use Projects\Model\Project;
use Projects\Model\Project as ProjectModel;

class ID extends AbstractValidator
{
    const NOT_EMPTY  = 'isEmpty';
    const NOT_UNIQUE = 'notUnique';

    public function __construct($options = null)
    {
        $this->services         = $options[self::SERVICES];
        $this->p4               = $options[ConnectedAbstractValidator::CONNECTION
            ] ?? $this->services->get($this->services->get(ConnectionFactory::P4_ADMIN));
        $this->messageTemplates = [
            self::NOT_EMPTY => $this->tGen('Name must contain at least one letter or number.'),
            self::NOT_UNIQUE => $this->tGen('This name is taken. Please pick a different name.'),
        ];
        parent::__construct($options);
    }

    /**
     * A project id valus is considered valid if is
     * - not empty after being filtered
     * - a valid identifier for a group
     * - unique
     * @param $value
     * @return bool
     * @throws \Exception
     */
    public function isValid($value)
    {
        // Non valid characters will have been filtered out
        if (!(new NotEmpty())->isValid($value)) {
            $this->error(self::NOT_EMPTY);
            return false;
        }
        // A project id must be also be a valid group name
        $groupNameValidator = new GroupName();
        $groupNameValidator->setAllowMaxValue(Project::MAX_VALUE);
        if (!$groupNameValidator->isValid($value)) {
            $this->abstractOptions['messages'] = $groupNameValidator->getMessages();
            return false;
        }
        // There must not already be a project with this id, including an old deleted one
        $matchingProjects = $this->services->get(IDao::PROJECT_DAO)->fetchAll(
            [
                ProjectModel::FETCH_INCLUDE_DELETED => true,
                ProjectModel::FETCH_BY_IDS          => [$value]
            ],
            $this->p4
        );
        if ($matchingProjects->count()) {
            $this->error(self::NOT_UNIQUE);
            return false;
        }

        return true;
    }
}
