<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.2/2785633
 */
namespace Notifications\Model;

use InvalidArgumentException;
use P4\Connection\ConnectionInterface as Connection;
use Record\Exception\NotFoundException;
use Record\Key\AbstractKey;

/**
 * Main class to be extended for other notification modules like Slack.
 */
abstract class Notification extends AbstractKey implements INotification
{
    protected $fields  = [
        INotification::FIELD_THREAD_IDS    => [
            'accessor'  => 'getThreadIds',
            'mutator'   => 'setThreadIds'
        ],
    ];
    protected $topicId = null;
    const KEY_PREFIX   = 'notification-';

    /**
     * Constructor that puts id into topicId.
     *
     * @param Connection|null $connection The p4 connection needed
     * @param int|string|null $topicId    The topic id to be used (Review id or Change id)
     */
    public function __construct(Connection $connection = null, $topicId = null)
    {
        parent::__construct($connection);
        if ($topicId && (is_int($topicId) || is_string($topicId))) {
            $this->topicId = $topicId;
        }
    }

    /**
     * Overriding the save method to ensure we have an exception if no topicId is set.
     *
     * @return Notification
     * @throws InvalidArgumentException
     */
    public function save(): Notification
    {
        if (is_null($this->topicId)) {
            throw new InvalidArgumentException("Topic id must be set");
        }
        return parent::save();
    }

    /**
     * @inheritDoc
     */
    public function getThreadIds(): ?array
    {
        return $this->getRawValue(INotification::FIELD_THREAD_IDS);
    }

    /**
     * @inheritDoc
     */
    public function setThreadIds($threadIds)
    {
        $this->setRawValue(INotification::FIELD_THREAD_IDS, (array) $threadIds);
    }

    /**
     * Override the default makeId as we don't want to make increment a count here. We just want to use the
     * ID provide to create this model.
     * @return int|string
     */
    protected function makeId()
    {
        return static::encodeId($this->topicId);
    }

    /**
     * Fetch a record by its id
     * @param mixed $id id to fetch
     * @param Connection|null $p4 connection to user
     * @return AbstractKey
     * @throws NotFoundException
     */
    public static function fetchById($id, Connection $p4 = null): AbstractKey
    {
        $p4 = $p4 ?: parent::getDefaultConnection();
        return parent::fetch($id, $p4);
    }
}
