<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.2/2785633
 */

namespace Application\Validator;

use Application\I18n\TranslatorFactory;
use Interop\Container\ContainerInterface;
use Laminas\Validator\AbstractValidator;

/**
 * Validate that a value is a string or array of strings, with the option to also accept an integer or array of integers
 */
class IdsValidator extends AbstractValidator
{
    private $field;
    private $allowInt;
    private $services;

    const ALLOW_INT = "allowInt";
    const INVALID   = "arrayOrValueInvalid";

    /**
     * IdsValidator constructor.
     * @param ContainerInterface $services  application services
     * @param string             $field     The model or field this ids validation is for.
     * @param array              $options   Supports ALLOW_INT => true to accept int values as well as string values, by
     *                                      default this is false
     */
    public function __construct(ContainerInterface $services, string $field, array $options = [])
    {
        parent::__construct();
        $this->field    = $field;
        $this->allowInt = $options[self::ALLOW_INT] ?? false;
        $this->services = $services;
    }
    /**
     * Valid that if is array of strings or a single string.
     * @param mixed     $value      The array or string.
     * @return bool true if valid
     */
    public function isValid($value): bool
    {
        $valid = false;

        if (is_string($value) || ($this->allowInt && is_int($value))) {
            $valid = true;
        } elseif (is_array(($value))) {
            $valid = array_sum(
                array_map(
                    function ($arrayValue) {
                        return $this->allowInt ? is_int($arrayValue) || is_string($arrayValue) : is_string($arrayValue);
                    },
                    $value
                )
            ) == count($value);
        }
        if (!$valid) {
            $translator    = $this->services->get(TranslatorFactory::SERVICE);
            $intMessage    = $translator->t(
                "%s ids must be valid string, int, array of strings, or array of int",
                [$this->field]
            );
            $stringMessage = $translator->t(
                "%s ids must be valid string or array of strings",
                [$this->field]
            );

            $this->abstractOptions['messages'][self::INVALID] = $this->allowInt ? $intMessage : $stringMessage;
        }
        return $valid;
    }
}
