<?php
/**
 * Perforce Swarm
 *
 * @copyright   2013-2025 Perforce Software. All rights reserved.
 * @license     Please see LICENSE.txt in top-level readme folder of this distribution.
 * @version     2025.1/2745343
 */
namespace Projects\Filter;

use Application\Connection\ConnectionFactory;
use Application\Factory\InvokableService;
use Application\Filter\ArrayValues;
use Application\Filter\FormBoolean;
use Application\Filter\StringToId;
use Application\InputFilter\DirectInput;
use Application\InputFilter\InputFilter;
use Application\Validator\AbstractValidator;
use Application\Validator\ConnectedAbstractValidator;
use Application\Validator\FlatArray;
use Application\Validator\GreaterThanInt;
use Application\Validator\IsArray;
use Application\Validator\IsBool;
use Application\Validator\IsString;
use Groups\Validator\Groups as GroupsValidator;
use Interop\Container\ContainerInterface;
use Laminas\Filter\StringTrim;
use Laminas\InputFilter\Input;
use Laminas\Validator\StringLength;
use Projects\Filter\Defaults as ProjectDefaultsFilter;
use Projects\Filter\Deploy as DeployFilter;
use Projects\Filter\TestAutomation as TestAutomationFilter;
use Projects\Filter\Workflow as WorkflowFilter;
use Projects\Model\IProject;
use Projects\Validator\Deploy as DeployValidator;
use Projects\Validator\EmailFlags as EmailFlagsValidator;
use Projects\Validator\ID;
use Projects\Validator\JobView;
use Projects\Validator\Members;
use Projects\Validator\Defaults as ProjectDefaultsValidator;
use Projects\Validator\Name;
use Projects\Validator\TestAutomation as TestAutomationValidator;
use Users\Validator\Users as UsersValidator;

class Project extends ProjectFields implements InvokableService
{
    /**
     * @inheritDoc
     */
    public function __construct(ContainerInterface $services, array $options = null)
    {
        parent::__construct($services, $options);
        $this->addIdFilter();
        $this->addName();
        $this->addDescriptionFilter();
        $this->addBranches();
        $this->addMinimumUpVotes();
        $this->addEmailFlagsFilter();
        $this->addJobViewFilter();
        $this->addPrivateFilter();
        $this->addDeployFilter();
        $this->addProjectDefaults();
        $this->addRetainDefaultReviewersFilter();
        $this->addWorkflow();
        $this->addOwners();
        $this->addMembers();
        $this->addSubGroups();
        $this->addTestAutomation();
    }

    /**
     * Id is a required field for a new project, which should be normalised to a valid p4d id value
     * @return void
     */
    protected function addIdFilter()
    {
        $input = new DirectInput(IProject::ID);
        $input->getFilterChain()->attach(new StringTrim())->attach(new StringToId());
        $input->getValidatorChain()->attach(
            new ID(
                [   AbstractValidator::SERVICES => $this->services,
                    ConnectedAbstractValidator::CONNECTION => $this->p4]
            )
        );
        $this->add($input);
    }

    /**
     * Trim the name, make it optional
     * @return void
     */
    protected function addName()
    {
        $input = new DirectInput(IProject::NAME);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()
            ->attach(new StringLength(['min'=>1]))
            ->attach(
                new Name(
                    [
                        AbstractValidator::SERVICES => $this->services,
                        Name::CHECK_PERMISSION => $this->isEdit()
                    ]
                )
            );
        $this->add($input);
    }

    /**
     * Trim the description and make it optional
     * @return void
     */
    protected function addDescriptionFilter()
    {
        $input = new DirectInput(IProject::DESCRIPTION);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new StringTrim());
        $this->add($input);
    }

    /**
     * Add a filter and validator for an array of boolean values
     * @return void
     */
    protected function addEmailFlagsFilter()
    {
        $input = new Input(IProject::EMAIL_FLAGS);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new EmailFlags());
        $input->getValidatorChain()
            ->attach(new FlatArray())
            ->attach(new EmailFlagsValidator());
        $this->add($input);
    }

    /**
     * Trim the job view and validate the value
     * @return void
     */
    protected function addJobViewFilter()
    {
        $input = new Input(IProject::JOB_VIEW);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new StringTrim());
        $input->getValidatorChain()
            ->attach(new IsString())
            ->attach(new JobView());
        $this->add($input);
    }

    /**
     * The private project indicator is a simple boolean flag
     * @return void
     */
    protected function addPrivateFilter()
    {
        $input = new DirectInput(IProject::PRIVATE);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new FormBoolean());
        $input->getValidatorChain()->attach(new IsBool());
        $this->add($input);
    }

    /**
     * The deployment details has a custom filter to normalise the value and a custom filter
     * to verify its contents
     * @return void
     */
    protected function addDeployFilter()
    {
        $input = new Input(IProject::DEPLOY);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new DeployFilter());
        $input->getValidatorChain()->attach(new FlatArray())->attach(new DeployValidator());
        $this->add($input);
    }

    /**
     * Add validation for the 'default->reviewers' field. Filtered to an array that must contain
     * valid users if populated
     * @return void
     */
    protected function addProjectDefaults()
    {
        $input = new DirectInput(IProject::DEFAULTS);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new ProjectDefaultsFilter());
        $input->getValidatorChain()->attach(new IsArray())->attach(
            new ProjectDefaultsValidator(
                [
                    ConnectedAbstractValidator::CONNECTION => $this->services->get(ConnectionFactory::P4)
                ]
            )
        );
        $this->add($input);
    }

    /**
     * The default reviewers retention should be a boolean flag
     * @return void
     */
    protected function addRetainDefaultReviewersFilter()
    {
        $input = new DirectInput(IProject::RETAIN_DEFAULT_REVIEWERS);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new FormBoolean());
        $input->getValidatorChain()->attach(new IsBool());
        $this->add($input);
    }

    /**
     * Minimum up votes be an integer (allowing conversion) that is greater than 0
     * @return void
     */
    protected function addMinimumUpVotes()
    {
        $input = new DirectInput(IProject::MINIMUM_UP_VOTES);
        $input->setRequired(false);
        $input->getValidatorChain()->attach(new GreaterThanInt(['min' => -1, 'nullable' => true]));
        $this->add($input);
    }

    /**
     * A Workflow has a bespoke validator that can optionally verify permissions
     * @return void
     */
    protected function addWorkflow()
    {
        $input = new DirectInput(IProject::WORKFLOW);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new StringTrim())->attach(new WorkflowFilter());
        $input->getValidatorChain()->attach($this->workflowValidator);
        $this->add($input);
    }

    /**
     * Add validation for the 'owners' field. Filtered to an array that must contain valid users if populated
     * @return void
     */
    protected function addOwners()
    {
        $input = new DirectInput(IProject::OWNERS);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new ArrayValues());
        $input->getValidatorChain()->attach(
            new UsersValidator(
                [
                    ConnectedAbstractValidator::CONNECTION => $this->services->get(ConnectionFactory::P4)
                ]
            )
        );
        $this->add($input);
    }

    /**
     * Add validation for the subgroups field, filtering to an array and validating that every item in the array is
     * a valid group id.
     * @return void
     */
    protected function addSubGroups()
    {
        $input = new DirectInput(IProject::SUBGROUPS);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new ArrayValues());
        $input->getValidatorChain()->attach(
            new GroupsValidator(
                [
                    ConnectedAbstractValidator::CONNECTION => $this->services->get(ConnectionFactory::P4)
                ]
            )
        );
        $this->add($input);
    }

    /**
     * Add validation for the members field, filtering to an array and validating that every item in the array is
     * a valid user. Also check that either members or subgroups are provided
     * @return void
     */
    protected function addMembers()
    {
        $input = new DirectInput(IProject::MEMBERS);
        $input->getFilterChain()->attach(new ArrayValues());
        $input->getValidatorChain()
            ->attach(new FlatArray(), true)
            ->attach(
                new UsersValidator(
                    [
                        ConnectedAbstractValidator::CONNECTION => $this->services->get(ConnectionFactory::P4)
                    ]
                )
            )
            ->attach(new Members());
        $this->add($input);
    }

    /**
     * The test automation details has a custom filter to normalise the value and a custom validator
     * to verify its contents
     * @return void
     */
    protected function addTestAutomation()
    {
        $input = new DirectInput(IProject::TESTS);
        $input->setRequired(false);
        $input->getFilterChain()->attach(new TestAutomationFilter());
        $input->getValidatorChain()->attach(new FlatArray())->attach(new TestAutomationValidator());
        $this->add($input);
    }
}
